package org.lsst.ccs.gconsole.base;

import java.awt.Window;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.file.Path;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.TimeoutException;
import javax.swing.Action;
import org.freehep.util.FreeHEPLookup;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.gconsole.agent.AgentStatusAggregator;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.jas3.Jas3Console;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.services.optpage.LsstOptionalPageService;
import org.lsst.ccs.messaging.CommandRejectedException;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Graphical Console subsystem.
 * Provides all functionality of a CCS subsystem. Specifies additional services that
 * should be provided by specific console implementations for use by plugins and tools.
 * 
 * @author onoprien
 */
abstract public class Console extends Subsystem {
    
// -- Fields : -----------------------------------------------------------------
    
    static private final Jas3Console agent = new Jas3Console();
    
    private final AgentStatusAggregator aggregator = new AgentStatusAggregator();
    
    // Messags in the application status bar:
    
    private int statusMessageID = 0;
    private final TreeMap<Integer,String> statusMessageHistory = new TreeMap<>();
    private final int STATUS_HISTORY_SIZE = 30;
    
// -- Life cycle : -------------------------------------------------------------
    
    protected Console() {
        super("CCS-Graphical-Console", AgentInfo.AgentType.CONSOLE);
    }

    @Override
    public void preStart() {
        
        // Initialize status aggergator
        
        aggregator.initialize(this);
        getConsoleLookup().add(aggregator);
        
        // Add items from ConsoleLookup.RESOURCE to lookup

        try {
            Enumeration<URL> e = getClass().getClassLoader().getResources(ConsoleLookup.RESOURCE);
            while (e.hasMoreElements()) {
                URL url = e.nextElement();
                try (
                        InputStream ins = url.openStream();
                        BufferedReader in = new BufferedReader(new InputStreamReader(ins));
                    ) 
                {
                    String line;
                    while ((line = in.readLine()) != null) {
                        String[] ss = line.trim().split("=");
                        if (ss.length != 2) break;
                        if (ss[0].isEmpty()) {
                            getConsoleLookup().add(ss[1]);
                        } else {
                            getConsoleLookup().add(ss[1], ss[0]);
                        }
                    }
                } catch (IOException x) {
                }
            }
        } catch (IOException x) {
        }
    }

    @Override
    public void preShutdown() {
        aggregator.shutdown();
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns the singleton instance of {@code Console}.
     * <p>
     * <i>Implementation note: At the moment, this method returns an instance of
     * {@link Jas3Console} created when this class is loaded, breaking abstraction 
     * from the {@code Console} implementation. This should be fixed once the {@code freehep-application}
     * framework initialization have been reworked to make them thread-safe and more flexible,
     * so that {@link Jas3Console} can be loaded as a service.</i>
     * 
     * @return Singleton instance of {@code Console}.
     */
    static public Console getConsole() {
        return agent;
    }
    
    /**
     * Returns a lookup instance that can be used objects registered by various 
     * components of the graphical console.
     * @return Lookup instance.
     */
    abstract public FreeHEPLookup getConsoleLookup();

    /**
     * Returns the root component for the main Graphical Console window.
     * @return root component for the main Graphical Console window, or {@code null}
     *         if the root component is not a {@link Window}.
     */
    abstract public Window getWindow();
    
    
    /**
     * Returns the default logger associated with the graphical console.
     * @return The logger associated with the graphical console.
     */
//    @Override  // REVIEW
    public Logger getLoggerUI() {
        return Logger.getLogger("org.lsst.ccs.ui");
    }
    
    /**
     * Returns the path to the graphical console home directory.
     * User preferences, console extensions, and miscellaneous auxiliary files saved
     * by the graphical console are kept in this directory and its subdirectories.
     * @return The path to the graphical console home directory.
     */
    abstract public Path getHomeDirectory();
    
    /**
     * Returns the instance of {@code PanelManager} that can be used to create and manage GUI panels.
     * @return Panel manager used by this graphical console.
     */
    abstract public PanelManager getPanelManager();
    
    /**
     * Returns the status aggregator maintained by this console.
     * @return Status aggregator.
     */
    public AgentStatusAggregator getStatusAggregator() {
        return aggregator;
    }
    
// -- Miscellaneous services : -------------------------------------------------
    
    /**
     * Notifies the user of an application error.
     * @param message Message to be displayed.
     */
    abstract public void error(String message);
    
    /**
     * Notifies the user of an application error.
     * @param message Message to be displayed.
     * @param x Exception to be reported.
     */
    abstract public void error(String message, Exception x);
    
    /**
     * Display the specified web site in a browser.
     * @param url URL of the site to be displayed.
     */
    abstract public void showInBrowser(String url);
    
    /**
     * Sets the message in the graphical console status bar.
     * Using this method modifies the text in the status bar without affecting the additional
     * bookkeeping provided by the {@link #setStatusMessage(String, int)} method.
     * @param message The message to display; if {@code null}, the current message is cleared.
     */
    abstract public void setStatusMessage(String message);
    
    /**
     * Sets the message in the graphical console status bar.
     * This method provides simple bookkeeping that reduces interference between
     * messages posted by various graphical console components.
     * <p>
     * If a message is submitted with zero ID, it is displayed immediately, and this
     * method returns its newly assigned ID. If a message is submitted with non-zero
     * ID, it replaces any previous messages with the same ID, but it is only displayed
     * if or when the message it replaces would be displayed.
     * The message keeps the ID of the message it replaces.
     * <p>
     * Calling this method with {@code message == null} clears the set of messages 
     * submitted with the same ID. If there is another set with a different ID that
     * has not been cleared, the latest message of that set will be displayed. If
     * this method is called with {@code message == null} and zero ID, all message sets are cleared.
     * <p>
     * Typically, this machinery is used when some graphical console component needs
     * to display a status message temporarily. It first calls this method with zero
     * ID to display the message, and then calls it with the ID returned by the first
     * call and {@code null} message to clear the status bar.
     * 
     * @param message The message to display, or {@code null} to clear the status bar.
     * @param id The ID of the previously posted message this message should replace,
     *           or zero if this message is not associated with any of the previous messages.
     * @return The ID assigned to the posted message.
     */
    public int setStatusMessage(String message, int id) {
        synchronized (statusMessageHistory) {
            if (id == 0) {
                if (message == null) {
                    statusMessageHistory.clear();
                    statusMessageID = 0;
                    setStatusMessage(null);
                } else {
                    statusMessageID = statusMessageHistory.isEmpty() ? 1 : statusMessageHistory.lastKey() + 1;
                    if (statusMessageID == Integer.MAX_VALUE) {
                        statusMessageHistory.clear();
                        statusMessageID = 1;
                    } 
                    while (statusMessageHistory.size() > STATUS_HISTORY_SIZE) {
                        statusMessageHistory.pollFirstEntry();
                    }
                    statusMessageHistory.put(statusMessageID, message);
                    setStatusMessage(message);
                    return statusMessageID;
                }
            } else if (id == statusMessageID) {
                if (message == null) {
                    statusMessageHistory.remove(id);
                    Map.Entry<Integer,String> e = statusMessageHistory.lastEntry();
                    if (e == null) {
                        statusMessageID = 0;
                        setStatusMessage(null);
                    } else {
                        statusMessageID = e.getKey();
                        setStatusMessage(e.getValue());
                    }
                } else {
                    statusMessageHistory.replace(id, message);
                    setStatusMessage(message);
                }
            } else {
                if (message == null) {
                    statusMessageHistory.remove(id);
                } else {
                    statusMessageHistory.replace(id, message);
                }
            }
            return id;
        }
    }
    
    /**
     * Registers an optional page descriptor with the console.
     * Once registered, the descriptor will trigger creation of console pages whenever
     * compatible subsystems are discovered on the buses. 
     * See {@link OptionalPage} documentation for details.
     * The demo subsystem GUI provides an example of use: {@code org.lsst.ccs.subsystem.demo.gui.plugins.OptionalPagePlugin}.
     * 
     * @param descriptor Page descriptor.
     * @deprecated Use {@code getOptionalPageService().add(OptionalPage descriptor)} instead.
     */
    @Deprecated
    public void addOptionalPage(OptionalPage descriptor) {
        LsstOptionalPageService service = (LsstOptionalPageService) getConsoleLookup().lookup(LsstOptionalPageService.class);
        if (service != null) {
            service.add(descriptor);
        }
    }
    
    /**
     * Removes a previously registered optional page with the specified path.
     * See {@link #addOptionalPage(OptionalPage)}.
     * 
     * @deprecated Use {@code getOptionalPageService().remove(OptionalPage descriptor)} instead.
     * @param path Page path.
     */
    @Deprecated
    public void removeOptionalPage(String path) {
        LsstOptionalPageService service = (LsstOptionalPageService) getConsoleLookup().lookup(LsstOptionalPageService.class);
        if (service != null) {
            service.remove(path);
        }
    }
    
    /**
     * Removes a previously registered optional page.
     * See {@link #addOptionalPage(OptionalPage)}.
     * 
     * @param descriptor Descriptor of the page to be removed
     * @deprecated Use {@code getOptionalPageService().remove(OptionalPage descriptor)} instead.
     */
    @Deprecated
    public void removeOptionalPage(OptionalPage descriptor) {
        LsstOptionalPageService service = getOptionalPageService();
        if (service != null) {
            service.remove(descriptor);
        }
    }
    
    public LsstOptionalPageService getOptionalPageService() {
        return (LsstOptionalPageService) getConsoleLookup().lookup(LsstOptionalPageService.class);
    }


// -- Handling properties and preferences : ------------------------------------
    
    /**
     * Defines a console property with the specified default value.
     * Console properties can be later modified by calls to the {@link #setProperty(String, Object)} method.
     * Properties that have been modified from their default values are saved between the graphical console sessions.
     * <p>
     * If a property with the specified key already exists:<ul>
     * <li>if the specified default value is of the same type as the existing one, the call has no effect;
     * <li>if the existing default value is of type {@code String} but can be converted to the type of the newly specified
     *     default value, the existing default value is retained but the type of object returned by subsequent calls to
     *     {@link #getProperty(String)} method is modified;
     * <li>if the specified default value is incompatible with the existing one, {@code IllegalArgumentException} is thrown;
     * <li>if the existing property does not have a default value, its current value is reset to the provided default.
     * </ul>
     * When a property is retrieved with a call to {@link #getProperty(String)} method, the type of the
     * returned object is determined by the type of the default value supplied when the property was defined.
     * 
     * @param key Property key. Can be used to retrieve the property value.
     * @param defaultValue Default value for the property. 
     *        Must be of type {@code Boolean}, {@code Enum}, {@code Integer}, {@code Long}, {@code Double}, {@code String},
     *        or an array containing objects of these types.
     * @return Current value of the property.
     * @throws IllegalArgumentException If the specified key already exists, and its default value is incompatible with {@code defaultValue};
     *                                  if either {@code key} or {@code defaultValue} is {@code null};
     *                                  if the type of {@code defaultValue} is not supported.
     */
    abstract public Object addProperty(String key, Object defaultValue);
    
    /**
     * Removes a console property.
     * 
     * @param key Property key.
     * @return Current value of the property, or {@code null} if there was no such property.
     */
    abstract public Object removeProperty(String key);
    
    /**
     * Returns the value of the property with the specified key.
     * The type of the returned object is determined by the type of the default value supplied when the property was defined.
     * If the property with the specified key has never been defined through a call to {@link #addProperty(String, Object)}
     * but exists in the console storage, it will be returned as a {@code String}.
     * 
     * @param key Property key.
     * @return Value of the specified property, or {@code null} if the property does not exist.
     */
    abstract public Object getProperty(String key);
   
    /**
     * Sets the value of the specified property.
     * If the property with the specified key has never been defined through a call 
     * to {@link #addProperty(String, Object)} method, the value will be converted 
     * to a {@code String} and stored. Such keys cannot be used in setter format 
     * strings passed to {@link #addPreference(String[], String, String)} method.
     * <p>
     * If the property value is set to {@code null}, it is removed from storage, so
     * subsequent calls to {@link #getProperty(String)} will return the default
     * value, if it has been set.
     * 
     * @param key Property key.
     * @param value Property value, or {@code null} if the property is being reset to its default value.
     * @return The previous value of the specified property, or {@code null} if the property did not exist.
     * @throws IllegalArgumentException If the specified value is not type-compatible with the default value for the specified key.
     */
    abstract public Object setProperty(String key, Object value);
   
    /**
     * Sets values of a set of properties.
     * See {@link #setProperty(String, Object)} for details on setting properties.
     * 
     * @param properties
     */
    public void setProperties(Map<String,Object> properties) {
        properties.forEach((key,value) -> setProperty(key,value));
    }
    
    /**
     * Registers a listener that will be notified of property changes.
     * 
     * @param listener The listener to be notified.
     * @param filter Regular expression the property key has to match to trigger notification.
     */
    abstract public void addPropertyListener(PropertyListener listener, String filter);
    
    /**
     * Removes the specified property listener.
     * 
     * @param listener Listener to be removed.
     * @return {@code True} if the specified listener was registered with this console.
     */
    abstract public boolean removePropertyListener(PropertyListener listener);
    
    public interface PropertyListener {
        
        /**
         * Called by the framework when plugin properties are modified.
         *
         * @param source Source of notification.
         * @param changes For the changed properties, map of keys to their new values.
         */
        void propertiesChanged(Object source, Map<String,Object> changes);
        
    }

     
    /**
     * Adds a setter for one or more previously defined properties to the preferences menu.
     * The {@code format} string contains one or more property references in the ${key#qualifiers} form.
     * Qualifiers are separated by "#". At the moment, the only supported qualifier is "history=N".
     * 
     * @param path Path in the preferences tree to the page where the property setter should be displayed.
     * @param group Group inside the preferences page. If {@code null}, the property setter is displayed in the default group.
     * @param format Specifies the format of the property setter.
     * @throws IllegalArgumentException If a property with the specified key already exists, and its value is incompatible with the provided default.
     */
    abstract public void addPreference(String[] path, String group, String format);
   
    
// -- Adding menus, etc. : -----------------------------------------------------
    
    /**
     * Adds an item to the graphical console menu bar.
     * <p>
     * The location strings specify a hierarchy of menus above the added item. Each
     * string should be in the {@code name[[:group]:position]} format, where
     * {@code name} is the text label of the menu the item should be added to,
     * {@code group} is the ordinal number of the item group inside the menu, and 
     * {@code position} is the position inside the group.
     * The first location string can also be prefixed with the desired location 
     * on the menu bar: {@code [barPosition:]name[[:group]:position]}.
     * 
     * The menu is divided into groups by separators.
     * If the group number is negative, items added with the same group number but 
     * different owners are put into the same group. If the group is not specified
     * or is equal to 0, the item is added to the default group. The default group
     * is shared by all owners.
     * 
     * @param action Action for the menu item to be added.
     * @param owner Owner of the added menu item.
     * @param locations Strings that specifies locations of the added item at all levels in the menu hierarchy.
     */
    abstract public void addMenu(Action action, String owner, String... locations);
    
    /**
     * Removes the specified menu item and its descendents.
     * 
     * @param locations Hierarchy of menu item names, starting from the top level menu.
     * @return {@code true} is some items have been removed.
     */
    abstract public boolean removeMenu(String... locations);
    
    
// -- Sending commands : -------------------------------------------------------
    
    /**
     * Sends a command over the buses and returns the result once it is received.
     * 
     * @param timeout Timeout.
     * @param command Command in {@code agent[/component]/method} format.
     * @param args Command arguments.
     * @return The result of the command. If the result is an exception, it is thrown rather than returned.
     * @throws TimeoutException if the command times out.
     * @throws CommandRejectedException if the target subsystem rejects the command by sending NACK.
     * @throws Exception if the command execution results in an exception.
     */
    public Object sendCommand(Duration timeout, String command, Object... args) throws Exception {
        ConcurrentMessagingUtils cmu = new ConcurrentMessagingUtils(getMessagingAccess());
        int i = command.lastIndexOf("/");
        if (i == -1) {
            throw new IllegalArgumentException("Illegal command: "+ command +". Use \"agent[/component]/method\" format.");
        }
        String dst = command.substring(0, i);
        String cmnd = command.substring(i+1);
        CommandRequest cmd = new CommandRequest(dst, cmnd, args);
        return cmu.sendSynchronousCommand(cmd, timeout);
    }
    
    /**
     * Sends a command over the buses and returns immediately, without waiting for response.
     * 
     * @param command Command in {@code agent[/component]/method} format.
     * @param args Command arguments.
     */
    public void sendCommand(String command, Object... args) {
        Thread t = new Thread(() -> {
            try {
                sendCommand(Duration.ofSeconds(1L), command, args);
            } catch (Exception x) {}
        }, "Temporary thread sending command");
        t.start();
    }
    
    /**
     * Access to command service.
     * The service provides functionality related to sending commands to remote subsystems and processing responses.
     * @return Singleton instance of {@code CommandService}.
     */
    public CommandService getCommandService() {
        return (CommandService) getConsoleLookup().lookup(CommandService.class);
    }
    
}
