package org.lsst.ccs.gconsole.agent.filter;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;

/**
 * Filter to be associated with a view that handles a single page of a multi-page parent view.
 *
 * @author onoprien
 */
public class PageFilter implements AgentChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final AgentChannelsFilter filter;
    private final String prefix;

// -- Life cycle : -------------------------------------------------------------
    
    public PageFilter(AgentChannelsFilter filter, String page) {
        this.filter = filter;
        prefix = page == null || page.isEmpty() ? "" : page +"//";
    }

    
// -- Forward methods : --------------------------------------------------------

    @Override
    public String getName() {
        String name = filter.getName();
        if (name != null) {
            name += ":";
            if (!prefix.isEmpty()) {
                name += prefix.substring(0, prefix.length()-2);
            }
        }
        return name;
    }

    @Override
    public List<String> getAgents() {
        return filter.getAgents();
    }

    @Override
    public List<String> getOriginChannels() {
        return filter.getOriginChannels();
    }

    @Override
    public List<String> getDisplayChannels() {
        List<String> cc = filter.getDisplayChannels();
        if (cc == null) return null;
        return filterPage(cc);
        
    }

    @Override
    public String getOriginPath(String displayPath) {
        return filter.getOriginPath(prefix + displayPath);
    }

    @Override
    public List<String> getDisplayPath(AgentChannel channel) {
        return filterPage(filter.getDisplayPath(channel));
    }

    @Override
    public List<String> getDisplayPath(String originPath) {
        return filterPage(filter.getDisplayPath(originPath));
    }

    @Override
    public Map<String, Object> getAttributes(String path) {
        return filter.getAttributes(prefix + path);
    }

    @Override
    public List<String> getFields(boolean compact) {
        return filter.getFields(compact);
    }

    @Override
    public void addListener(Listener listener) {
        filter.addListener(listener);
    }

    @Override
    public void removeListener(Listener listener) {
        filter.removeListener(listener);
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    List<String> filterPage(List<String> in) {
        if (prefix.isEmpty()) {
            return in.stream().filter(dp -> !dp.contains("//")).collect(Collectors.toList());
        } else {
            int i = prefix.length();
            return in.stream().filter(dp -> dp.startsWith(prefix)).map(dp -> dp.substring(i)).collect(Collectors.toList());
        }        
    }
    
    
    
    
}
