package org.lsst.ccs.gconsole.plugins.trending;

import com.sun.jersey.api.client.WebResource;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JToolBar;
import javax.swing.SwingUtilities;
import org.freehep.application.Application;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageManager;
import org.freehep.application.studio.Studio;
import org.freehep.jas.plugin.tree.FTree;
import org.freehep.jas.plugin.tree.FTreeNode;
import org.freehep.jas.plugin.tree.FTreePath;
import org.freehep.jas.plugin.tree.FTreeProvider;
import org.freehep.jas.services.PlotFactory;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.freehep.jas.services.PlotterProvider;
import org.freehep.util.FreeHEPLookup;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.jas3.Jas3Console;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindowSelector;
import org.openide.util.Lookup;
import org.openide.util.LookupListener;

/**
 * Graphical console plugin that plots historical trending data.
 *
 * @author onoprien
 */
@Plugin(name="LSST Trending Plugin",
        id="trending",
        description="Allows plotting time histories for channels from the trending database and other sources.")
public class LsstTrendingPlugin extends ConsolePlugin implements PlotterProvider {

// -- Fields : -----------------------------------------------------------------
    
//<editor-fold defaultstate="collapsed">
    final String PANEL_GROUP = "TrendingPlots";
    
    private TrendingPreferences prefs;
    private volatile TrendingSourcesManager treeMan;
    private volatile AutoRefreshManager refreshMan;
    private volatile PlotFactory plotFactory;
    
    private RestServerSource restSource;
    private final Object restSourceLock = new Object();
    
    private volatile TimeWindowSelector timeWindowBox;
    private JToolBar toolbar;
    private JCheckBoxMenuItem autoRefreshMenu;
    
    private final Action trendingAction;
    
    private final ArrayList<TrendingPanel> pages = new ArrayList<>(0);
    
    private ThreadPoolExecutor dataFetcher;
    private final int DATA_FETCHER_MAX_THREADS = 3;
    private final int DATA_FETCHER_KEEP_ALIVE = 120;
//</editor-fold>

// -- Life cycle : -------------------------------------------------------------

// <editor-fold defaultstate="collapsed">
    public LsstTrendingPlugin() {
        trendingAction = new AbstractAction("Trending") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean) getValue(Action.SELECTED_KEY)) {
                    startTrending();
                } else {
                    stopTrending();
                }
            }
        };
        trendingAction.putValue(Action.SELECTED_KEY, false);
    }

    @Override
    synchronized public void initialize() {
        prefs = new TrendingPreferences(this);
        getConsole().getConsoleLookup().add(prefs);
        getServices().addMenu(trendingAction, "400: CCS Tools :-1:3");
    }

    @Override
    synchronized public void start() {
        plotFactory = (PlotFactory) this.getConsole().getConsoleLookup().lookup(PlotFactory.class);
        if (plotFactory == null) throw new RuntimeException("No plot factory available");
        if (prefs.isAutoStart()) {
            SwingUtilities.invokeLater(this::startTrending);
        }
     }

    @Override
    public void stop() {
        if (prefs.isAutoSave()) {
            SwingUtilities.invokeLater(() -> ((Jas3Console)getConsole()).save());
        }
    }

    @Override
    synchronized public void shutdown() {
        if (treeMan != null) {
            SwingUtilities.invokeLater(this::stopTrending);
        }
    }

    /**
     * Starts trending application.
     * All resource-intensive processes are initialized here.
     * Should be called on the EDT.
     */
    synchronized private void startTrending() {
        
        trendingAction.putValue(Action.SELECTED_KEY, true);
        
        // start executor
        
        ThreadFactory tFactory = new ThreadFactory() {
            final ThreadFactory delegate = Executors.defaultThreadFactory();
            int id = 0;
            @Override
            public Thread newThread(Runnable r) {
                Thread thread = delegate.newThread(r);
                thread.setName("Trending Data Fetcher " + id++);
                return thread;
            }
        };
        dataFetcher = new ThreadPoolExecutor(DATA_FETCHER_MAX_THREADS, DATA_FETCHER_MAX_THREADS, 
                                          DATA_FETCHER_KEEP_ALIVE, TimeUnit.SECONDS, 
                                          new LinkedBlockingQueue<>(), tFactory);
        dataFetcher.allowCoreThreadTimeOut(true);
        
        // Add trending toolbar to GUI
        
        showToolbar();
        
        // Start channel tree manager
        
        if (treeMan == null) treeMan = new TrendingSourcesManager(this);
        treeMan.start();
                
        // Start listening for dataservers in lookup
        
        FreeHEPLookup lookup = getConsole().getConsoleLookup();
        Lookup.Template template = new Lookup.Template(WebResource.class, "dataserver", null);
        Lookup.Result result = lookup.lookup(template);
        handleRestServers(result.allInstances());
        LookupListener ll = e -> handleRestServers(result.allInstances());
        result.addLookupListener(ll);
        
        // Register trending service
        
        final TrendingService trendingService = path -> {
            FTreeProvider treeProvider = (FTreeProvider) lookup.lookup(FTreeProvider.class);
            if (treeProvider != null) {
                final FTree tree = treeProvider.tree("Trending");
                FTreeNode node = tree.findNode(new FTreePath(String.join("/", path)));
                if (node == null) {
                    FTreePath p = new FTreePath(path);
                    getConsole().setStatusMessage("No such trending channel "+p);
                } else {
                    tree.adapterForClass(TrendingChannel.class).doubleClick(node);
                }
            }
        };
        lookup.add(trendingService);
    }
    
    /**
     * Stops trending application.
     * All resources are released here (most, actually - at the moment, some are retained due to limitations of FTree plugin).
     * Should be called on the EDT.
     */
    synchronized private void stopTrending() {
        trendingAction.putValue(Action.SELECTED_KEY, false);
        stopAutoRefresh();
        new ArrayList<>(pages).forEach(page -> page.hidePage());
        treeMan.stop();
        hideToolbar();
        dataFetcher.shutdownNow();
        dataFetcher = null;
    }
//</editor-fold>
    
// -- Getters : ----------------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    TrendingPreferences getPreferences() {
        return prefs;
    }
    
    PlotFactory getPlotFactory() {
        return plotFactory;
    }
    
    TrendingSourcesManager getSourcesManager() {
        return treeMan;
    }
    
    TimeWindowSelector getTimeWindowSelector() {
        return timeWindowBox;
    }
    
    TimeWindow getSelectedTimeWindow() {
        return timeWindowBox.getSelectedTimeWindow();
    }
    
    /**
     * Returns the list of trending pages.
     * Must be called on the EDT.
     * @return The list of trending pages.
     */
    List<TrendingPanel> getPages() {
        return Collections.unmodifiableList(pages);
    }
// </editor-fold>
    
//  -- Plotting : --------------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    /**
     * Plots the channel in the specified region.
     * This method should always be called on the EDT.
     * 
     * @param channel Channel to plot.
     * @param timeWindow Time window. If {@code null}, currently selected time window is used.
     * @param region Region to plot in.
     * @param options The current plot in the specified region is either replaced or added to,
     *                depending on whether the {@code OVERLAY} option is present.
     */
    void plot(TrendingChannel channel, TimeWindow timeWindow, PlotRegion region, TrendPlotter.Option... options) {
        
        // find or create plotter
        
        if (channel == null || region == null) return;
        
        TrendPlotter plotter;
        try {
        plotter = (TrendPlotter) region.currentPlot();
        } catch (ClassCastException x) {
            return;
        }
        if (plotter == null) {
            plotter = create();
            showPlotter(plotter, region);
        }
        if (!plotter.isActive()) return;
        
        // fetch currently selected time window in none is specified
        
        if (timeWindow == null) {
            timeWindow = getSelectedTimeWindow();
            if (timeWindow == null) return;
        }
        
        // plot
        
        Trend trend = new Trend(channel);
        trend.setTimeWindow(timeWindow);
        int mode = !plotter.isEmpty() && options.length != 0 && Arrays.asList(options).contains(TrendPlotter.Option.OVERLAY) ? Plotter.OVERLAY : Plotter.NORMAL;
        plotter.plot(trend, mode);
        
        // launch trend update
        
        final Trend finalTrend = trend;
        dataFetcher.execute(() -> refresh(finalTrend));
        
    }
    
    /**
     * Plots time history for the specified channel.
     * This method should always be called on the EDT.
     * 
     * @param channel Data channel to be plotted.
     * @param timeWindow Time window. If {@code null}, currently selected time window is used.
     * @param options Option that control the selection of target region and other execution parameters.
     */
    void plot(TrendingChannel channel, TimeWindow timeWindow, TrendPlotter.Option... options) {
        
        if (channel == null) return;
        EnumSet<TrendPlotter.Option> opts = options.length == 0 ? EnumSet.noneOf(TrendPlotter.Option.class) : EnumSet.copyOf(Arrays.asList(options));
        
        // fetch currently selected time window in none is specified
        
        if (timeWindow == null) {
            timeWindow = getSelectedTimeWindow();
            if (timeWindow == null) return;
        }
        
        // if using existing plot for the specified channel is requested, find it

        Trend trend = null;
        if (opts.contains(TrendPlotter.Option.EXIST)) {
            trend = findTrend(channel, timeWindow);
        }
        
        // if necessary, create new trend, plotter, region, and page
        
        if (trend == null) {
            PlotPage page = null;
            PlotRegion region = null;
            if (!opts.contains(TrendPlotter.Option.NEWPAGE)) {
                page = plotFactory.currentPage();
            }
            if (page == null) {
                page = createPage(channel.getTitle());
                if (page == null) {
                    getConsole().error("Unable to create a plot page");
                    return;
                } else {
                    region = page.currentRegion();
                }
            }
            if (region == null) {
                if (opts.contains(TrendPlotter.Option.NEWPLOT)) {
                    int n = page.numberOfRegions();
                    for (int i = 0; i < n; i++) {
                        PlotRegion r = page.region(i);
                        if (r.currentPlot() == null) {
                            region = r;
                            break;
                        }
                    }
                } else {
                    region = page.currentRegion();
                }
                if (region == null) {
                    region = page.addRegion();
                }
            }
            trend = new Trend(channel);
            trend.setTimeWindow(timeWindow);
            
            TrendPlotter plotter;
            Plotter pl = region.currentPlot();
            if (pl != null && !(pl instanceof TrendPlotter)) {
                pl.clear();
                region.clear();
                pl = null;
            }
            if (pl == null) {
                plotter = create();
                showPlotter(plotter, region);
            } else {
                plotter = (TrendPlotter) pl;
            }
            int mode = !plotter.isEmpty() && opts.contains(TrendPlotter.Option.OVERLAY) ? Plotter.OVERLAY : Plotter.NORMAL;
            plotter.plot(trend, mode);
        }
        
        // bring to front
        
        trend.getPlotter().toFront();
        
        // launch trend update
        
        final Trend finalTrend = trend;
        dataFetcher.execute(() -> refresh(finalTrend));

    }
    
    /**
     * Refreshes all regions where the specified channel data is plotted.
     * This method should always be called on the EDT.
     * 
     * @param channel Data channel.
     */
    void refresh(TrendingChannel channel) {
        ArrayList<Trend> trends  = findTrends(channel);
        trends.forEach(trend -> {
            if (!trend.getTimeWindow().isFixed()) {
                dataFetcher.execute(() -> refresh(trend.getPlotter().getTrends()));
            }
        });
    }
    
    /**
     * Refreshes all regions on the specified page.
     * This method should always be called on the EDT.
     * 
     * @param page Plot page.
     * @param timeWindow Time window for the new plots. 
     *                   If {@code null}, time windows already associated with plots being refreshed are used.
     */
    void refresh(PlotPage page, TimeWindow timeWindow) {
        int n = page.numberOfRegions();
        try {
            for (int i = 0; i < n; i++) {
                refresh(page.region(i), timeWindow);
            }
        } catch (IndexOutOfBoundsException x) {
            // should not happen but some Jass3 code may modify the page while we're refreshing it
        }
    }
    
    /**
     * Refreshes plots in the specified region.
     * This method should always be called on the EDT.
     * 
     * @param region Plot region.
     * @param timeWindow Time window for the new plots. 
     *                   If {@code null}, time windows already associated with plots being refreshed are used.
     */
    void refresh(PlotRegion region, TimeWindow timeWindow) {
        Plotter plotter = region.currentPlot();
        if (plotter != null && plotter instanceof TrendPlotter) {
            TrendPlotter tp = (TrendPlotter) plotter;
            if (tp.isActive()) {
                if (timeWindow != null) {
                    tp.getTrends().forEach(t -> t.setTimeWindow(timeWindow));
                }
                dataFetcher.execute(() -> refresh(tp.getTrends()));
            }
        }
    }
    
    /**
     * Assigns plotter to a region.
     * This method should always be called on the EDT.
     * 
     * @param plotter Plotter to be displayed.
     * @param region Region where the plotter should be displayed.
     */
    static void showPlotter(TrendPlotter plotter, PlotRegion region) {
        Plotter oldPlotter = region.currentPlot();
        if (oldPlotter != null) {
            oldPlotter.clear();
            region.clear();
        }
        plotter.setRegion(region);
        region.showPlot(plotter);
    }
// </editor-fold>
    
// -- Implementing PlotterProvider : -------------------------------------------

// <editor-fold defaultstate="collapsed">
    @Override
    public boolean supports(Class klass) {
        return Trend.class.isAssignableFrom(klass);
    }

    @Override
    public TrendPlotter create() {
        return new TrendPlotter(this);
    }
// </editor-fold>    
    
// -- Saving/Restoring : -------------------------------------------------------

// <editor-fold defaultstate="collapsed">
    @Override
    public ComponentDescriptor save() {
        
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        if (dataFetcher == null) return desc;
        
        // Save selected time window
        
        TimeWindow tw = getSelectedTimeWindow();
        if (tw.isPersistent()) {
            desc.setTimeWindow(tw.getName());
        } else {
            desc.setTimeWindow(tw.toCompressedString());
        }
        
        // Save auto refresh state
        
        desc.setAutoRefresh(refreshMan != null);
        
        // Save trending panels
        
        int n = pages.size();
        TrendingPanel.Descriptor[] panels = new TrendingPanel.Descriptor[n];
        for (int i=0; i<n; i++) {
            panels[i] = pages.get(i).save();
        }
        desc.setPages(panels);
        
        return desc;
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        if (!(storageBean instanceof Descriptor)) {
            throw new IllegalArgumentException("Illegal descriptor type: "+ storageBean.getClassName());
        }
        Descriptor desc = (Descriptor) storageBean;
        
        // Stop current activities
        
//        stopAutoRefresh();
        autoRefreshMenu.setSelected(false);
        
        ArrayList<TrendingPanel> panels = new ArrayList<>(pages);
        panels.forEach(page -> page.hidePage());
        
        // Restore selected time window
        
        String twString = desc.getTimeWindow();
        TimeWindow selectedTimeWindow = timeWindowBox.getTimeWindow(twString);
        if (selectedTimeWindow != null) {
            timeWindowBox.setSelectedTimeWindow(selectedTimeWindow);
        }
        
        // Restore pages
        
        for (TrendingPanel.Descriptor pageDesc : desc.getPages()) {
            TrendingPanel page = createPage(pageDesc.getTitle());
            page.restore(pageDesc);
        }
        
        // Restore auto-setLoading state
        
        if (desc.isAutoRefresh()) {
            autoRefreshMenu.setSelected(true);
//            startAutoRefresh();
        }
        
        return true;
    }
    
    static public class Descriptor extends ComponentDescriptor {

        private String timeWindow;
        private boolean autoRefresh;
        private TrendingPanel.Descriptor[] pages;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public String getTimeWindow() {
            return timeWindow;
        }

        public void setTimeWindow(String timeWindow) {
            this.timeWindow = timeWindow;
        }

        public TrendingPanel.Descriptor[] getPages() {
            return pages;
        }

        public void setPages(TrendingPanel.Descriptor[] pages) {
            this.pages = pages;
        }

        public TrendingPanel.Descriptor getPages(int index) {
            return this.pages[index];
        }

        public void setPages(int index, TrendingPanel.Descriptor pages) {
            this.pages[index] = pages;
        }

        public boolean isAutoRefresh() {
            return autoRefresh;
        }

        public void setAutoRefresh(boolean autoRefresh) {
            this.autoRefresh = autoRefresh;
        }
        
    }
// </editor-fold>
    
// -- Local methods : ----------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    private void showToolbar() {
        
        Studio app = (Studio) Application.getApplication();
        toolbar = new JToolBar("trending");
        
        TimeWindowSelector box = new TimeWindowSelector(app);
        box.setEnabled(box.getItemCount() > 0);

        JLabel boxLabel = new JLabel("Trending Period: ");
        toolbar.add(boxLabel, 0);
        toolbar.add(box, 1);  
        
        // Apply button
        
        JMenuBar bar = new JMenuBar();
        bar.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(1, 1, 1, 1), BorderFactory.createRaisedBevelBorder()));
        bar.setAlignmentY(Component.CENTER_ALIGNMENT);
        JMenu menu = new JMenu("Apply");
        bar.add(menu);
        JMenuItem item = new JMenuItem("All");
        item.addActionListener(e -> {
            PageManager pm = app.getPageManager();
            PageContext selectedPage = pm.getSelectedPage();
            List pcList = pm.pages();
            for (Object o : pcList) {
                try {
                    PlotPage page = (PlotPage) ((PageContext) o).getPage();
                    if (page != null) {
                        refresh(page, getSelectedTimeWindow());
                    }
                } catch (NullPointerException | ClassCastException x) {
                }
            }
            selectedPage.requestShow();
        });
        menu.add(item);
        item = new JMenuItem("Page"); 
        item.addActionListener(e -> {
            try {
                PlotPage page = (PlotPage) app.getPageManager().getSelectedPage().getPage();
                if (page != null) {
                    refresh(page, getSelectedTimeWindow());
                }
            } catch (NullPointerException | ClassCastException x) {
            }
        });
        menu.add(item);
        item = new JMenuItem("Region");
        item.addActionListener(e -> {
            try {
                PlotPage page = (PlotPage) app.getPageManager().getSelectedPage().getPage();
                refresh(page.currentRegion(), getSelectedTimeWindow());
            } catch (NullPointerException | ClassCastException x) {
            }
        });
        menu.add(item);
        toolbar.add(bar,2);
        
        
        // Resresh button
        
        bar = new JMenuBar();
        bar.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(1, 1, 1, 1), BorderFactory.createRaisedBevelBorder()));
        bar.setAlignmentY(Component.CENTER_ALIGNMENT);
        menu = new JMenu("Refresh");
//        menu.setFont(boxLabel.getFont());
        bar.add(menu);
        item = new JMenuItem("All");
        item.addActionListener(e -> {
                PageManager pm = app.getPageManager();
                PageContext selectedPage = pm.getSelectedPage();
                List pcList = pm.pages();
                for (Object o : pcList) {
                    try {
                        PlotPage page = (PlotPage) ((PageContext)o).getPage();
                        if (page != null) {
                            refresh(page, null);
                        }
                    } catch (NullPointerException|ClassCastException x) {
                    }
                }
                selectedPage.requestShow();
        });
        menu.add(item);
        item = new JMenuItem("Page");
        item.addActionListener(e -> {
            try {
                PlotPage page = (PlotPage) app.getPageManager().getSelectedPage().getPage();
                if (page != null) {
                    refresh(page, null);
                }
            } catch (NullPointerException | ClassCastException x) {
            }
        });
        menu.add(item);
        item = new JMenuItem("Region");
        item.addActionListener(e -> {
            try {
                PlotPage page = (PlotPage) app.getPageManager().getSelectedPage().getPage();
                refresh(page.currentRegion(), null);
            } catch (NullPointerException | ClassCastException x) {
            }
        });
        menu.add(item);
        menu.addSeparator();
        autoRefreshMenu = new JCheckBoxMenuItem("Auto");
        autoRefreshMenu.setSelected(false);
        autoRefreshMenu.addItemListener(e -> {
            if (e.getStateChange() == ItemEvent.SELECTED) {
                startAutoRefresh();
            } else if (e.getStateChange() == ItemEvent.DESELECTED) {
                stopAutoRefresh();
            }
        });
        menu.add(autoRefreshMenu);
        toolbar.add(bar,3);
        
        app.addToolBar(toolbar, toolbar.getName());
        timeWindowBox = box;
    }
    
    private void hideToolbar() {
        Studio app = (Studio) Application.getApplication();
        app.removeToolBar(toolbar);
        toolbar = null;
        timeWindowBox = null;
    }
    
    private void startAutoRefresh() {
        if (refreshMan != null) stopAutoRefresh();
        refreshMan = new AutoRefreshManager(this);
        refreshMan.start();
    }
    
    private void stopAutoRefresh() {
        if (refreshMan == null) return;
        refreshMan.stop();
        refreshMan = null;
    }
    
    private void handleRestServers(Collection<WebResource> servers) {
        // Only one server at a time is currently allowed
        synchronized (restSourceLock) {
            if (restSource != null) {
                getConsole().getConsoleLookup().remove(restSource);
                restSource = null;
            }
            Iterator<WebResource> it = servers.iterator();
            if (it.hasNext()) {
                WebResource server = it.next();
                RestServerSource source = new RestServerSource(server, this);
                restSource = source;
                Runnable r = () -> {
                    if (source.connect()) {
                        synchronized (restSourceLock) {
                            if (restSource == source) {
                                getConsole().getConsoleLookup().add(restSource);
                            }
                        }
                    }
                };
                Thread t = new Thread(r, "REST Connector");
                t.setDaemon(true);
                t.start();
            }
        }
    }
    
    private ArrayList<Trend> findTrends(TrendingChannel channel) {
        String title = channel.getTitle();
        ArrayList<Trend> out = new ArrayList<>();
        pages.forEach(page -> {
            int n = page.numberOfRegions();
            for (int i=0; i<n; i++) {
                PlotRegion region = page.region(i);
                Plotter plotter = region.currentPlot();
                if (plotter != null && plotter instanceof TrendPlotter) {
                    List<Object> data = plotter.getData();
                    for (Object d : data) {
                        Trend trend = (Trend) d;
                        if (trend.getChannel().getTitle().equals(title)) {
                            out.add(trend);
                            break;
                        }
                    }
                }
            }
        });
        return out;
    }
    
    private Trend findTrend(TrendingChannel channel, TimeWindow timeWindow) {
        ArrayList<Trend> all = findTrends(channel);
        Trend best = null;
        int score = 0;
        for (Trend candidate : all) {
            TrendPlotter plotter = candidate.getPlotter();
            if (plotter.getData().size() == 1) {
                int candidateScore = candidate.getTimeWindow() == timeWindow ? 1 : 0;
                candidateScore = candidateScore<<1 + (candidate.getChannel().getPath().equals(channel.getPath()) ? 1 : 0);
                candidateScore = candidateScore<<3 +  plotter.getRegion().getPage().numberOfRegions();
                if (candidateScore > score) {
                    best = candidate;
                    score = candidateScore;
                }
            }
        }
        return best;
    }
    
    /**
     * Fetches data and updates the trend.
     * Should be called on an executor.
     */
    private void refresh(Trend trend) {
        trend.setLoading(true);
        TrendData data = treeMan.fetchData(trend, null);
        if (data != null && data != trend.getData()) {
            trend.setData(data);
        }
    }
    
    /**
     * Fetches data and updates all trends in the specified plotter.
     * Should be called on an executor.
     */
    void refresh(List<Trend> trends) {
        List<TrendData> data = treeMan.fetchData(trends, null);
        Iterator<Trend> it = trends.iterator();
        for (int i = 0; it.hasNext(); i++) {
            Trend trend = it.next();
            TrendData td = data.get(i);
            trend.setData(td);
        }
    }
    
    private TrendingPanel createPage(String name) {
        TrendingPanel panel = new TrendingPanel(this, name);
        pages.add(panel);
        return panel;
    }
    
//    private TrendingPanel createPage(Map<?,?> properties) {
//        TrendingPanel panel = new TrendingPanel(this, name);
//    }
    
    void closePage(TrendingPanel page) {
        pages.remove(page);
    }
// </editor-fold>

// -----------------------------------------------------------------------------
}
