package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Point;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.event.TableModelEvent;
import org.lsst.ccs.gconsole.plugins.monitor.ChannelHandle;

/**
 *
 *
 * @author onoprien
 */
public class SectionedTable extends MonitorTable {

// -- Fields : -----------------------------------------------------------------
    
    private List<MonitorField> columns;
    
    private LinkedHashMap<String, List<Map.Entry<String, ChannelHandle>>> sections;
    private Set<String> expandedSections;

// -- Life cycle : -------------------------------------------------------------
    
    static public SectionedTable getInstance(Collection<Map.Entry<String,ChannelHandle>> channels, List<MonitorField> fields, Descriptor descriptor) {
        SectionedTable instance = new SectionedTable();
        instance.init(channels, fields, descriptor);
        return instance;
    }
    
    private SectionedTable() {
    }
    
    public void init(Collection<Map.Entry<String,ChannelHandle>> channels, List<MonitorField> fields, Descriptor descriptor) {
        
        // Columns and collapsed sections:

        List<ChannelHandle> handles = channels.stream().map(e -> e.getValue()).collect(Collectors.toList());
        columns = trimAbsentFields(fields, handles, null);
        nColumns = columns.size()+1;

        // Sectioned map of channels:
        
        sections = new LinkedHashMap<>();
        List<Map.Entry<String, ChannelHandle>> defSection = new ArrayList<>();
        sections.put("", defSection);

        for (Map.Entry<String, ChannelHandle> e : channels) {
            String dpath = e.getKey();
            int i = dpath.indexOf("//");
            String section;
            String name;
            if (i == -1) {
                defSection.add(e);
            } else {
                section = dpath.substring(0, i);
                name = dpath.substring(i + 2);
                List<Map.Entry<String, ChannelHandle>> cc = sections.get(section);
                if (cc == null) {
                    cc = new ArrayList<>();
                    sections.put(section, cc);
                }
                cc.add(new AbstractMap.SimpleImmutableEntry<>(name, e.getValue()));
            }
        }

        expandedSections = (descriptor == null || descriptor.getExpandedSections() == null) ? new HashSet<>(sections.keySet()) : new HashSet<>(Arrays.asList(descriptor.getExpandedSections()));

        // Build list of rows:
        
        int rowOffset = 0;
        cells = new ArrayList<>();
        for (String section : sections.keySet()) {
            List<ArrayList<MonitorCell>> rows = makeSection(section, rowOffset);
            rowOffset += rows.size();
            cells.addAll(rows);
        }
        cells.trimToSize();
        nRows = cells.size();
    }
    
    @Override
    public void destroy() {
        sections.values().forEach(items -> items.forEach(e -> e.getValue().setTarget(null)));
    }
    
    
// -- Table model : ------------------------------------------------------------
    
    @Override    
    public String getColumnName(int column) {
        return column == 0 ? "Channel" : columns.get(column-1).getTitle();
    }
    

// -- Handling mouse events : --------------------------------------------------
    @Override
    protected void tableMouseClicked(MouseEvent evt) {
        if (evt.getClickCount() == 1) {
            JTable table = (JTable) evt.getSource();
            Point point = evt.getPoint();
            int row = table.rowAtPoint(point);
            int column = table.columnAtPoint(point);
            if ( column == 0 ) {
                MonitorCell cell = cells.get(row).get(0);
                if (cell instanceof SectionTitleCell) {
                    SectionTitleCell c = (SectionTitleCell) cell;
                    String section = c.getTitle();
                    if (c.isCollapsed()) {
                        expandSection(row, section);
                    } else {
                        collapseSection(row, section);
                    }
                }
            }
        }
        super.tableMouseClicked(evt);
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private List<ArrayList<MonitorCell>> makeSection(String section, int rowOffset) {
        List<Map.Entry<String, ChannelHandle>> channels = sections.get(section);
        if (channels == null || channels.isEmpty()) return Collections.emptyList();
        boolean collapsed = !section.isEmpty() && !expandedSections.contains(section);
        if (collapsed) { // make collapsed section row and return a singleton list
            ArrayList<MonitorCell> row = new ArrayList<>(nColumns);
            row.add(new SectionTitleCell(section, true));
            RowUpdater updater = new RowUpdater(rowOffset);
            ArrayList<ChannelHandle> handles = new ArrayList<>(channels.size());
            for (Map.Entry<String, ChannelHandle> e : channels) {
                ChannelHandle ch = e.getValue();
                handles.add(ch);
                ch.setTarget(updater);
            }
            for (MonitorField field : columns) {
                MonitorCell cell = new MonitorCell(handles, field);
                format.format(cell);
                row.add(cell);
            }
            return Collections.singletonList(row);
        } else { // section is not collapsed
            List<ArrayList<MonitorCell>> rows = new ArrayList<>(channels.size()+1);
            if (!section.isEmpty()) { // make expanded section row
                ArrayList<MonitorCell> row = new ArrayList<>(nColumns);
                row.add(new SectionTitleCell(section, false));
                for (MonitorField field : columns) {
                    row.add(MonitorCell.EMPTY);
                }
                rows.add(row);
                rowOffset++;
            }
            for (Map.Entry<String, ChannelHandle> channel : channels) { // make regular row
                ArrayList<MonitorCell> row = new ArrayList<>(nColumns);
                row.add(new MonitorCell(null, null, new FormattedValue(channel.getKey(), SwingConstants.LEFT)));
                List<ChannelHandle> handles = Collections.singletonList(channel.getValue());
                for (MonitorField field : columns) {
                    MonitorCell cell = new MonitorCell(handles, field);
                    format.format(cell);
                    row.add(cell);
                }
                rows.add(row);
                RowUpdater updater = new RowUpdater(rowOffset++);
                channel.getValue().setTarget(updater);
            }
            return rows;
        }
    }
    
    private void expandSection(int row, String section) {
        expandedSections.add(section);
        List<ArrayList<MonitorCell>> sectionRows = makeSection(section, row);
        cells.remove(row);
        cells.addAll(row, sectionRows);
        nRows = cells.size();
        updateUpdaters(row + sectionRows.size() - 1);
        fireTableRowsUpdated(row, row);
        fireTableRowsInserted(row+1, row+sectionRows.size()-1);
        fireChangeEvent(TABLE_EVENT);
    }
    
    private void collapseSection(int row, String section) {
        expandedSections.remove(section);
        List<ArrayList<MonitorCell>> sectionRows = makeSection(section, row);
        cells.set(row, sectionRows.get(0));
        int sectionSize = sections.get(section).size();
        for (int i=0; i<sectionSize; i++) {
            cells.remove(row+1);
        }
        nRows = cells.size();
        updateUpdaters(row);
        fireTableRowsUpdated(row, row);
        fireTableRowsDeleted(row + 1, row + sectionSize);
        fireChangeEvent(TABLE_EVENT);
    }
    
    private void updateUpdaters(int row) {
        while (++row < nRows) {
            MonitorCell cell = cells.get(row).get(1);
            List<ChannelHandle> handles = cell.getChannels();
            for (ChannelHandle h : handles) {
                RowUpdater updater = (RowUpdater) h.getTarget();
                updater.setRow(row);
            }
        }
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class RowUpdater implements Updatable {
        
        int row;
        
        RowUpdater(int row) {
            this.row = row;
        }

        public void setRow(int row) {
            this.row = row;
        }

        @Override
        public List<MonitorField> getAffectedFields(List<String> attributes) {
            return MonitorField.getAffectedFields(attributes, columns);
        }

        @Override
        public void update(ChannelHandle channelHandle, List<MonitorField> fields) {
            for (MonitorField f : fields) {
                int column = columns.indexOf(f);
                if (column != -1) {
                    format.format(cells.get(row).get(column+1), channelHandle);
                }
            }
            fireTableRowsUpdated(row, row);
            fireChangeEvent(CELLS_EVENT);
        }

        @Override
        public void update(ChannelHandle channelHandle) {
            for (int column = 0; column < columns.size(); column++) {
                MonitorField f = columns.get(column);
                if (f.isUpdatable()) {
                    format.format(cells.get(row).get(column+1), channelHandle);
                }
            }
            fireTableRowsUpdated(row, row);
            fireChangeEvent(CELLS_EVENT);
        }
        
    }
    
    private class SectionTitleCell extends MonitorCell {
        
        final String title;
        final boolean collapsed;
        
        SectionTitleCell(String title, boolean collapsed) {
            super(null, null, new FormattedValue("<html> "+ (collapsed ? "&#8853;" : "&#8854;") +"<b> "+ title, null, null, SwingConstants.LEFT, (collapsed ? "Click to expand section" : "Click to collaps section"), false));
            this.title = title;
            this.collapsed = collapsed;
        }

        public boolean isCollapsed() {
            return collapsed;
        }
        
        String getTitle() {
            return title;
        }        
    }
    
    
// -- Saving/Restoring : -------------------------------------------------------
    
    static public class Descriptor implements Serializable {

        private String[] expandedSections;

        public String[] getExpandedSections() {
            return expandedSections;
        }

        public void setExpandedSections(String[] expandedSections) {
            this.expandedSections = expandedSections;
        }
        
    }
    
    @Override
    public Descriptor save() {
        if (expandedSections != null && !expandedSections.isEmpty()) {
//            HashSet<String> expanded = new HashSet<>(expandedSections);
//            expanded.retainAll(sections.keySet());
            Descriptor desc = new Descriptor();
            desc.setExpandedSections(expandedSections.toArray(new String[expandedSections.size()]));
            return desc;
        } else {
            return null;
        }
    }
    
    public void restore(Descriptor desc) {
        // At the moment, there is no use for this method: descriptor is supplied to constructor.
        // If it ever becomes desirable to modify table settings without re-creating the table,
        // we might consider using this method from registerInstance() instead of init().
    }

}
