package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.Color;
import java.awt.Component;
import java.util.*;
import javax.swing.AbstractListModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;
import javax.swing.JTree;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.utilities.structs.TreeBranch;

/**
 *
 * @author emarin
 */
public class CommandListPanel extends JList<DictionaryCommand> implements TreeSelectionListener, ListSelectionListener {
    
    private final AgentPanel.Descriptor descriptor;
    
    private final SortedListModel cmdListModel;
    private final CommandListRenderer renderer;
    
    private Dictionary currentDictionary;
    
    public CommandListPanel(AgentPanel agentPanel){
        descriptor = agentPanel.descriptor;
        cmdListModel= new SortedListModel();
        setModel(cmdListModel);
        
        getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        renderer = new CommandListRenderer();
        setCellRenderer(renderer);
    }
    
    public boolean isVisibleCommandSelected() {
        DictionaryCommand dc = getSelectedValue();
        return dc == null ? false : isCommandVisible(dc);
    }

    /**
     * 
     * @param tse 
     */
    @Override
    public void valueChanged(TreeSelectionEvent tse) {
        TreeBranch<ComponentDictionary> node = (TreeBranch<ComponentDictionary>)( ((JTree)tse.getSource()).getLastSelectedPathComponent());
        // clear the list model and the command description pane
        cmdListModel.clear();
        if (node == null) {
            return;
        }
        
        currentDictionary = node.getContent().dictionary;
        if (currentDictionary == null) {
            return;
        }
        for (DictionaryCommand dc : currentDictionary) {
            if (!descriptor.isHideUnavailableCommands() || isCommandVisible(dc)) {
                cmdListModel.add(dc);
            }
        }
    }

    @Override
    public void valueChanged(ListSelectionEvent e) {
        cmdListModel.clear();
    }

    public void updateRenderer() {
        if (currentDictionary != null) {
            cmdListModel.clear();
            for (DictionaryCommand dc : currentDictionary) {
                if (!descriptor.isHideUnavailableCommands() || isCommandVisible(dc)) {
                    cmdListModel.add(dc);
                }
            }
            this.revalidate();
            this.repaint();
        }
    }
    
    private class CommandListRenderer extends DefaultListCellRenderer {
        
        @Override
        public Component getListCellRendererComponent(JList<?> jlist, Object e, int i, boolean isSelected, boolean cellHasFocus) {
            super.getListCellRendererComponent(jlist, e, i, isSelected, cellHasFocus);
            DictionaryCommand dc = (DictionaryCommand)e;
            setText(dc.getCommandName());
            if (!isCommandVisible(dc)) {
                setForeground(Color.GRAY);
            }
            return this;
        }
    }
    
    private boolean isCommandVisible(DictionaryCommand command) {
        if (currentDictionary != null) {
            return currentDictionary.getLevel() >= command.getLevel() && currentDictionary.getVisibilityForType(command.getType());
        }
        return true;
    }
    
    private class SortedListModel extends AbstractListModel {
        SortedSet<DictionaryCommand> model;
        
        SortedListModel(){
            model = new TreeSet<DictionaryCommand>(new Comparator<DictionaryCommand>(){
                @Override
                public int compare(DictionaryCommand o1, DictionaryCommand o2) {
                    if(isCommandVisible(o2) != isCommandVisible(o1)) {
                        if(isCommandVisible(o1)) {
                            return -1;
                        } else {
                            return 1;
                        }
                    }
                    return o1.getCommandName().compareTo(o2.getCommandName());
                }
            });
        }
        
        @Override
        public int getSize() {
            return model.size();
        }

        @Override
        public DictionaryCommand getElementAt(int index) {
            return (DictionaryCommand)model.toArray()[index];
        }

        public void add(DictionaryCommand element) {
            if (model.add(element)) {
                fireContentsChanged(this, 0, getSize());
            }
        }
        
        public void addAll(DictionaryCommand elements[]) {
            Collection c = Arrays.asList(elements);
            model.addAll(c);
            fireContentsChanged(this, 0, getSize());
        }
        
        public void clear() {
            clearSelection();
            model.clear();
            fireContentsChanged(this, 0, getSize());
        }
        
        public boolean contains(DictionaryCommand element) {
            return model.contains(element);
        }
        
        public Object firstElement() {
            // Return the appropriate element
            return model.first();
        }

        public Iterator iterator() {
            return model.iterator();
        }

        public Object lastElement() {
            // Return the appropriate element
            return model.last();
        }
        
        public boolean removeElement(DictionaryCommand element) {
            boolean removed = model.remove(element);
            if (removed) {
                fireContentsChanged(this, 0, getSize());
            }
            return removed;   
        }        
        
    }
}
