package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.io.Serializable;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.RouteSelectionCommandSet;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.subsystem.shell.ConsoleCommandShell;
import org.openide.util.Exceptions;

/**
 * Handle for a remote {@link Agent}.
 * <ul>
 * <li>Watches the locking state of a target agent, notifies listeners of changes, accepts requests for changes.
 * <li>Is externally notified of changes in connected/disconnected state 
 *     (by a call to {@code updateConnected(...)} method), notifies listeners.
 * </ul>
 * All access to instances of this class should happen on EDT.
 * 
 * @author onoprien
 */
class AgentHandle implements Serializable {
    
// -- Fields : -----------------------------------------------------------------
    
    private final RouteSelectionCommandSet routeSelector = ConsoleCommandShell.createConsoleCommandShell(Console.getConsole()).getConsoleCommandSet();
    
    private final String name;
    private AgentInfo info;
    
    private final AgentLockService lockService;
    private final AgentLockService.AgentLockUpdateListener lockListener;
    private final AgentLockService.AgentLevelListener levelListener;
    
    final Map<String, Dictionary> dictionaries = new HashMap<>();
    final Map<String, CommandSet> commandSets = new HashMap<>();
    
    private AgentLock lock;
    private int level;
    private boolean online;
    private boolean adjusting;
    
    private final CopyOnWriteArrayList<ChangeListener> listeners = new CopyOnWriteArrayList<>();
    private final ChangeEvent changeEvent = new ChangeEvent(this);
    
// -- Life cycle : -------------------------------------------------------------
    
    AgentHandle(String agent) {
        this.name = agent;
        lockService = Console.getConsole().getAgentService(AgentLockService.class);
        lockListener = (agentName, agentLock) -> {
            if (agent.equals(agentName)) {
                ThreadUtil.invokeLater(() -> {
                    lock = agentLock;
                    listeners.forEach(listener -> listener.stateChanged(changeEvent));
                });
            }
        };
        levelListener = (agentName, agentLevel) -> {
            if (agent.equals(agentName)) {
                ThreadUtil.invokeLater(() -> {
                    level = agentLevel;
                    listeners.forEach(listener -> listener.stateChanged(changeEvent));
                });
            }
        };
        online = false;
    }
    
    AgentHandle(AgentInfo agentInfo) {
        this(agentInfo.getName());
        info = agentInfo;
        online = true;
    }
    
    void init() {
        lockService.addAgentLockUpdateListener(lockListener);
        lockService.addAgentLevelListener(levelListener);
    }
    
    void shutdown() {
        lockService.removeAgentLockUpdateListener(lockListener);
        lockService.removeAgentLevelListener(levelListener);
    }
    
// -- Getters : ----------------------------------------------------------------
    
    public String getName() {
        return name;
    }
    
    AgentInfo getInfo() {
        return info;
    }

    AgentLock getLock() {
        return lock;
    }

    int getLevel() {
        return level;
    }

    boolean isAdjusting() {
        return adjusting;
    }

    Map<String, Dictionary> getDictionaries() {
        return dictionaries;
    }

    Map<String, CommandSet> getCommandSets() {
        return commandSets;
    }

    boolean isOnline() {
        return online;
    }
    
    boolean isLockedByMe() {
        return lock != null && lock.getOwnerAgentNames().contains(Console.getConsole().getAgentInfo().getName());
    }

    
// -- Setters : ----------------------------------------------------------------
    
    void setLock(boolean locked) {
        adjusting = true;
        notifyListeners();
        new SwingWorker<Object,Object>() {
            @Override
            protected Object doInBackground() throws Exception {
                if (locked) {
                    lockService.lockAgent(name);
                } else {
                    lockService.unlockAgent(name);
                }
                return null;
            }
            @Override
            protected void done() {
                try {
                    get();
                } catch (InterruptedException x) {
                    Console.getConsole().error("Unable to "+ (locked ? "" : "un") +"lock the "+ name +" subsystem.");
                } catch (ExecutionException x) {
                    try {
                        Console.getConsole().error("Unable to "+ (locked ? "" : "un") +"lock the "+ name +" subsystem.", (Exception) x.getCause());
                    } catch (ClassCastException xx) {
                        Console.getConsole().error("Unable to "+ (locked ? "" : "un") +"lock the "+ name +" subsystem.", x);
                    }
                }
                adjusting = false;
                notifyListeners();
            }
        }.execute();
    }

    public void setLevel(int level) {
        adjusting = true;
        notifyListeners();
        new SwingWorker<Object,Object>() {
            @Override
            protected Object doInBackground() throws Exception {
                lockService.setLevelForAgent(name, level);
                return null;
            }
            @Override
            protected void done() {
                try {
                    get();
                } catch (InterruptedException x) {
                    Console.getConsole().error("Unable to set level "+ level +" for "+ name +" subsystem.");
                } catch (ExecutionException x) {
                    try {
                        Console.getConsole().error("Unable to set level "+ level +" for "+ name +" subsystem.", (Exception) x.getCause());
                    } catch (ClassCastException xx) {
                        Console.getConsole().error("Unable to set level "+ level +" for "+ name +" subsystem.", x);
                    }
                }
                adjusting = false;
                notifyListeners();
            }
        }.execute();
    }


// -- Updates from external sources : ------------------------------------------
    
    public void updateDictionaries() {
        commandSets.clear();
        dictionaries.clear();
        if (online) {
            Map<String, CommandSet> commands = routeSelector.getCommandSetsForRoute(name);
            if (commands.isEmpty()) { // FIXME this ugly code works around lack of notification when dictionary is ready
                new SwingWorker<Boolean,Object>() {
                    @Override
                    protected Boolean doInBackground() throws Exception {
                        int attempt = 0;
                        Map<String, CommandSet> c = Collections.emptyMap();
                        while (attempt < 100 && c.isEmpty()) {
                            Thread.sleep(attempt++ < 10 ? 200 : 1000);
                            c = routeSelector.getCommandSetsForRoute(name);
                        }
                        return !c.isEmpty();
                    }
                    @Override
                    protected void done() {
                        try {
                            if (get()) updateDictionaries();
                        } catch (InterruptedException | ExecutionException ex) {
                        }
                    }
                }.execute();
                return;
            } else {
                for (Map.Entry<String, CommandSet> entry : commands.entrySet()) {
                    commandSets.put(entry.getKey(), entry.getValue());
                    dictionaries.put(entry.getKey(), entry.getValue().getCommandDictionary());
                }
            }
        }
        notifyListeners();
    }
    
    public void updateOnline(AgentInfo agentInfo) {
        this.online = agentInfo != null;
        if (online) {
            info = agentInfo;
        }
        updateDictionaries();
    }
    
    
// -- Handling listeners : -----------------------------------------------------
    
    public boolean addListener(ChangeListener listener) {
        return listeners.addIfAbsent(listener);
    }
    
    public boolean removeListener(ChangeListener listener) {
        return listeners.remove(listener);
    }
    
    public boolean removeAllListeners() {
        boolean out = ! listeners.isEmpty();
        listeners.clear();
        return out;
    }
    
    private void notifyListeners() {
        listeners.forEach(listener -> listener.stateChanged(changeEvent));
    }
    
    
// -- Overriding Object : ------------------------------------------------------

    @Override
    public int hashCode() {
        return name.hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true;
        if (obj instanceof AgentHandle) {
            return name.equals(((AgentHandle)obj).name);
        } else {
            return false;
        }
    }
    
    
}
