package org.lsst.ccs.gconsole.agent;

import java.time.Instant;
import java.util.*;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.DataProviderInfo.Attribute;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.gconsole.agent.AgentChannel.Key;
import org.lsst.ccs.gconsole.base.Console;

/**
 * Handles status aggregator data for a specific {@code Agent}.
 * 
 * Implementation notes:<ul>
 * <li>
 * A handle exists for any channel present on the buses.
 * <li>
 * Unwatched agents will have an empty list of listeners. For such agents, all non-final fields except {@code state} are {@code null}.
 * <li>
 * When the agent transfers to the watched status, the {@code channels} map is set to either {@code null} if no listener is interested
 * in channels from this agent, or to an empty list. In the letter case, the dictionary is requested.
 * <li>
 * All methods except getters are executed on the status aggregator thread. Listeners are notified on that thread as well.
 * <li>
 * Once the dictionary is received, it is used to populate {@code channels}.
 * {@code rejectedPaths} remains {@code null} if there are no templates in listeners, otherwise it is initialized to an empty set.
 * The listeners receive "configuration" event at this time.
 * </ul>
 */
class AgentHandle {
    
// -- Fields : -----------------------------------------------------------------
    
    private final AgentInfo agentInfo;
    private final AgentStatusAggregator aggregator;
    private final ArrayList<ListenerHandle> listeners = new ArrayList<>(0); // listeners interested in this agent
    
    private volatile StateBundle state; // last recordered agent state
    
    /* Map of inner paths to channels; null if there are no listeners interested in channels;
       empty if there are no watched channels at the moment, but there are unsatisfied templates.*/
    private LinkedHashMap<String, MutableAgentChannel> channels;
    
    /* Cached rejections; null if there are no expected channels or templates in any of the listener handles */
    private HashSet<String> rejectedPaths;

    private volatile AgentChannelsDictionary dictionary;
    private volatile Instant dictTime; // time when the dictionary inputs were last requested; null if never or the dictionary is already installed
    private volatile DataProviderDictionary dataDictionary;
    private ConfigurationInfo configInfo;


// -- Life cycle : -------------------------------------------------------------
    
    /** Constructs an instance */
    AgentHandle(AgentInfo agentInfo, AgentStatusAggregator aggregator) {
        this.aggregator = aggregator;
        this.agentInfo = agentInfo;
    }
    
    /**
     * Called immediately after construction.
     * @param message Message that triggered creation of this agent handle (first message received from the agent).
     */
    void onConnect(StatusMessage message) {
        
        // Save state:
        
        state = message.getState();
        
        // Create local handles for interested listeners:
        
        aggregator.listeners.forEach(globalHandle -> {
            ListenerHandle localHandle = new ListenerHandle(globalHandle);
            if (localHandle.init()) {
                listeners.add(localHandle);
                if (localHandle.isInterestedInChannels()) {
                    if (channels == null) channels = new LinkedHashMap<>();
                }
            }
        });
        listeners.trimToSize();
        
        // Notify listeners
        
        AgentStatusEvent event = new AgentStatusEvent(aggregator, agentInfo);
        listeners.forEach(lh -> lh.fireConnect(event));
    }
    
    /**
     * Called when the agent disconnects, and this agent handle needs to be discarded.
     * @param mess Message that informed us of disconnection, or {@code null} if the disconnection was detected by other means.
     */
    void onDisconnect(StatusMessage mess) {
        List<AgentChannel> removed = channels == null ? Collections.emptyList() : new ArrayList<>(channels.values());
        AgentStatusEvent event = new AgentStatusEvent(aggregator, agentInfo, Collections.emptyMap(), Collections.emptyList(), removed);
        listeners.forEach(lh -> {
            AgentStatusEvent e = event.filter(lh.paths);
            lh.global.listener.disconnect(e);
        });
    }


// -- Getters : ----------------------------------------------------------------
    
    /** Returns the agent descriptor. */
    AgentInfo getAgent() {
        return agentInfo;
    }
    
    /** Returns the channel specified by the given local path, or {@code null} if there is no watched channel with this path. */
    synchronized MutableAgentChannel getChannel(String innerPath) {
        return channels == null ? null : channels.get(innerPath);
    }
    
    /** Returns the last recorded state. */
    synchronized StateBundle getState() {
        return state;
    }
    
    /**
     * Returns a collection of watched channels. 
     * An empty collection is returned if there are no channels, regardless of whether there
     * are any templates that can result in channels added once they appear in status messages.
     */
    synchronized Collection<MutableAgentChannel> getChannels() {
        return channels == null ? Collections.emptyList() : channels.values();
    }
    
    
// -- Processing messages : ----------------------------------------------------
    
    void onMessage(StatusMessage mess) {
        
        StateBundle newState = mess.getState();
        boolean isLatest = ! state.getLastModified().isAfter(newState.getLastModified());
        if (isLatest) state = newState;
        
        // No channels are being watched - ignore:
        
        if (channels == null) return;        
        
        // No dictionary yet
        
        if (dictionary == null) {
            if (mess instanceof StatusDataProviderDictionary) {
                dataDictionary = ((StatusDataProviderDictionary) mess).getDataProviderDictionary();
            } else if (mess instanceof StatusConfigurationInfo) {
                configInfo = ((StatusConfigurationInfo)mess).getConfigurationInfo();
            }
            updateDictionary();
            return;
        }

        // Detect status changes and accumulate into event:
        
        MutableAgentStatusEvent event = new MutableAgentStatusEvent(aggregator, mess);

        synchronized (this) { // keep lock while modifying data

            // Agent and channel states:

            if (mess instanceof StatusStateChangeNotification || !isLatest) {
                
                StateBundle diff = isLatest ? ((StatusStateChangeNotification)mess).getNewState().diffState(((StatusStateChangeNotification)mess).getOldState()) : state;

                Map<String, String> current = getAllStates(diff);
                for (Map.Entry<String, String> e : current.entrySet()) {
                    String innerPath = "state/" + e.getKey();
                    MutableAgentChannel channel = channels.get(innerPath);
                    if (channel == null) {
                        channel = createChannel(innerPath);
                        if (channel != null) {
                            channel.set(e.getValue());
                            event.addAddedChannel(channel);
                        }
                    } else {
                        boolean changed = channel.set(e.getValue());
                        if (changed) {
                            event.addChange(channel, AgentChannel.Key.VALUE);
                        }
                    }
                }
                
//                Map<String, DataProviderState> change = diff.getComponentsWithState(DataProviderState.class);
//                for (Map.Entry<String, DataProviderState> e : change.entrySet()) {
//                    String innerPath = dictionary.getPathFromTrendingKey(e.getKey());
//                    MutableAgentChannel channel = channels.get(innerPath);
//                    if (channel != null) {
//                        boolean changed = channel.set(AgentChannel.Key.STATE, e.getValue());
//                        if (changed) {
//                            event.addChange(channel, AgentChannel.Key.STATE);
//                        }
//                    }
//                }
                
            }

            // Trending data :
            
            Object o = mess.getEncodedData();
            if (o instanceof KeyValueDataList) {
                KeyValueDataList encodedData = (KeyValueDataList) o;
                for (KeyValueData d : encodedData) {
                    KeyValueData.KeyValueDataType type = d.getType();
                    String attrKey = null;
                    String innerPath = null;
                    Object value = d.getValue();
                    switch (type) {
                        case KeyValueTrendingData:
                            innerPath = dictionary.getPathFromTrendingKey(d.getKey());
                            attrKey = AgentChannel.Key.VALUE;
                            break;
                        case KeyValueMetaData:
                            innerPath = d.getKey();
                            int lastIndex = innerPath.lastIndexOf('/');
                            attrKey = innerPath.substring(lastIndex + 1);
                            innerPath = innerPath.substring(0, lastIndex);
                            if (AgentChannel.Key.STATE.equals(attrKey) && value != null) { // convert states to enum
                                try {
                                    value = DataProviderState.valueOf(value.toString());
                                } catch (IllegalArgumentException x) {
                                    attrKey = null; // ignore
                                }
                            }
                            break;
                    }
                    if (attrKey != null) {
                        MutableAgentChannel channel = channels.get(innerPath);
                        if (channel != null) {
                            boolean changed = channel.set(attrKey, value);
                            if (changed) {
                                event.addChange(channel, attrKey);
                            }                            
                        } else if (attrKey.equals(AgentChannel.Key.VALUE)) {
                            channel = createChannel(innerPath);
                            if (channel != null) {
                                channel.set(attrKey, value);
                                event.addAddedChannel(channel);
                            }
                        }
                    }
                }
            }
            
            // Configuration :
            
            if (mess instanceof StatusConfigurationInfo) {
                ConfigurationInfo config = ((StatusConfigurationInfo)mess).getConfigurationInfo();
                for (ConfigurationParameterInfo conf : config.getLatestChanges()) {
                    String innerPath = dictionary.getPathFromTrendingKey(conf.getComponentName());
                    MutableAgentChannel channel = channels.get(innerPath);
                    if (channel != null) {
                        String attributeName = conf.getParameterName();
                        channel.set(attributeName, conf);
                        event.addChange(channel, attributeName);
                    }
                    innerPath = "configuration/"+ conf.getPathName();
                    channel = channels.get(innerPath);
                    if (channel == null) {
                        channel = createChannel(innerPath);
                        if (channel != null) {
                            channel.set(conf);
                            event.addAddedChannel(channel);
                        }
                    } else {
                        channel.set(conf);
                        event.addChange(channel, AgentChannel.Key.VALUE);
                    }
                }
            }
            
            // Time stamp:
            
            {
                Instant time = Instant.ofEpochMilli(mess.getTimeStamp());
                String innerPath = "runtimeInfo/current time";
                MutableAgentChannel channel = channels.get(innerPath);
                if (channel == null) {
                    channel = createChannel(innerPath);
                    if (channel != null) {
                        channel.set(time);
                        event.addAddedChannel(channel);
                    }
                } else {
                    boolean changed = channel.set(time);
                    if (changed) {
                        event.addChange(channel, AgentChannel.Key.VALUE);
                    }
                }
            }
        
        } // finished mofifying data, release lock on localAgent handle
        
        // Recompile the set of rejected paths if necessary
        
        if (!(event.getAddedChannels().isEmpty() && event.getRemovedChannels().isEmpty())) {
            resetRejectedPaths();
        }
        
        // Notify listeners:
        
        if (!event.isEmpty()) {
            listeners.forEach(lh -> lh.fireChange(event));
            if (!event.getAddedChannels().isEmpty()) { // if new channels, force publication of initial values
                Console.getConsole().sendCommand(agentInfo.getName() +"/publishDataProviderDictionary");
            }
        }
        
    }
    

// -- Adding/removing listeners : ----------------------------------------------
    
    /**
     * Called to add a listener.
     * Creates a local (agent-specific) handle for the listener, notifies 
     * 
     * @param global Global handle for the listener.
     */
    void addListener(AgentStatusAggregator.GlobalListenerHandle global) {
        ListenerHandle localHandle = new ListenerHandle(global);
        if (localHandle.init()) {
            listeners.ensureCapacity(listeners.size()+1);
            listeners.add(localHandle);
            localHandle.fireConnect(null);
            if (localHandle.isInterestedInChannels()) {
                if (channels == null) channels = new LinkedHashMap<>();
                if (dictionary == null) {
                    updateDictionary();
                } else {
                    localHandle.processDictionary(); // config event is sent from here
                    resetRejectedPaths();
                }
            }
        }
    }
    
    void removeListener(AgentStatusAggregator.GlobalListenerHandle global) {
        
        // remove local listener handles
        
        Iterator<ListenerHandle> it = listeners.iterator();
        boolean changed = false;
        while (it.hasNext()) {
            ListenerHandle local = it.next();
            if (local.global == global) {
                it.remove();
                changed = true;
            }
        }
        
        // if any handles have been removed, update {@code channels} and {@code rejectedPaths}.
        
        if (changed) {
            HashSet<String> paths = null;
            for (ListenerHandle lh : listeners) {
                if (lh.paths != null) {
                    if (paths == null) paths = new HashSet<>();
                    paths.addAll(lh.paths);
                }
            }
            synchronized (this) {
                if (paths == null) {
                    channels = null;
                    dictionary = null;
                } else {
                    Iterator<Map.Entry<String, MutableAgentChannel>> iter = channels.entrySet().iterator();
                    while (iter.hasNext()) {
                        Map.Entry<String, MutableAgentChannel> e = iter.next();
                        if (!paths.contains(e.getKey())) {
                            iter.remove();
                        }
                    }
                }
            }
            resetRejectedPaths();
        }
    }
    
    
// -- Requesting and processing dictionary: ------------------------------------
    
    /**
     * Creates dictionary of all inputs are available
     * Otherwise, checks if any inputs should be requested.
     */
    void updateDictionary() {
        
        // If inputs are ready, create dictionary and notify listeners
        
        if (dataDictionary != null && configInfo != null) {
            
            String prefix = agentInfo.getName() +"/";
            ArrayList<MutableAgentChannel> allChannels = new ArrayList<>();
            List<ConfigurationParameterInfo> configList = configInfo.getAllParameterInfo();
            Map<String, DataProviderState> channelStates = state.getComponentsWithState(DataProviderState.class);
            
            // From published dictionary
            
            String section = null;
            for (DataProviderInfo dc : dataDictionary.getDataProviderInfos()) {
                
                // Temporary - ignore non-monitoring entries -------------------
                String dataType = dc.getAttributeValue(Attribute.DATA_TYPE);
                if (dataType != null && !dataType.equals(DataProviderInfo.Type.MONITORING.name())) continue;
                // -------------------------------------------------------------
                
                String localPath = dc.getPath();
                MutableAgentChannel channel = new BasicChannel(prefix + localPath, agentInfo);
                String trendKey = dc.getKey();
                channel.set(Key.TRENDING, trendKey);
                DataProviderState channelState = channelStates.get(trendKey);
                if (channelState != null) {
                    channel.set(AgentChannel.Key.STATE, channelState);
                }
                for (Attribute att : dc.getAttributes()) {
                    Object attValue = dc.getAttributeValue(att);
                    if (att == Attribute.DESCRIPTION) {
                        String descr = attValue.toString();
                        int ii = descr.indexOf("\\");
                        if (ii != -1) {
                            section = descr.substring(0, ii);
                            attValue = descr.substring(ii + 1);
                        }
                        if (section != null) {
                            channel.set(Key.SECTION, section);
                        }
                    }
                    channel.set(att.getName(), attValue);
                }
                configList.forEach(config -> {
                    if (trendKey.equals(config.getComponentName())) {
                        channel.set(config.getParameterName(), config);
                    }
                });
                allChannels.add(channel);
            }
            
            // From state
            
            getAllStates(state).forEach((key, value) -> {
                String path = prefix +"state/"+ key;
                MutableAgentChannel channel = new BasicChannel(path, agentInfo);
                channel.set(value);
                allChannels.add(channel);
            });
            
            // From configuration
            
            configList.forEach(config -> {
                String path = prefix +"configuration/"+ config.getPathName();
                MutableAgentChannel channel = new BasicChannel(path, agentInfo);
                channel.set(config);
                allChannels.add(channel);
            });
            
            // Clean up and create dictionary 
            
            allChannels.trimToSize();
            dictionary = new AgentChannelsDictionary(allChannels);
            dataDictionary = null;
            configInfo = null;
            dictTime = null;
            
            // Update listener handles

            listeners.forEach(lh -> lh.processDictionary());  // Notify listeners:
            resetRejectedPaths();            
            return;
        }
        
        // If in waiting period - do nothing and return
        
        if (dictTime != null && dictTime.isAfter(Instant.now())) {
            return;
        }
        
        // Request missing inputs
        
        if (dataDictionary == null) {
            Console.getConsole().sendCommand(agentInfo.getName() +"/publishDataProviderDictionary");
        }
        if (configInfo == null) {
            Console.getConsole().sendCommand(agentInfo.getName() +"/publishConfigurationInfo");
        }
        int delay = 60000 + new Random().nextInt(5000);
        dictTime = Instant.now().plusMillis(delay);
        
    }

    
// -- Local methods : ----------------------------------------------------------

    /**
     * Called when data is encountered in status message that is not associated with any existing channel in AgentHandle.channels.
     * Returns {@code null} if there are no listeners interested in that channel, or a newly created channel.
     */
    MutableAgentChannel createChannel(String innerPath) {
        if (rejectedPaths == null || rejectedPaths.contains(innerPath)) return null;
        boolean accept = false;
        for (ListenerHandle lh : listeners) {
            if (lh.paths != null) {
                if (lh.paths.contains(innerPath)) {
                    accept = true;
                } else {
                    for (String template : lh.templates) {
                        if (innerPath.startsWith(template)) {
                            accept = true;
                            lh.paths.add(innerPath);
                            break;
                        }
                    }
                }
            }
        }
        if (accept) {
            MutableAgentChannel channel = new BasicChannel(agentInfo.getName() + "/" + innerPath, agentInfo);
            channels.put(innerPath, channel);
            dictionary.add(channel);
            return channel;
        } else {
            rejectedPaths.add(innerPath);
            return null;
        }
    }
    
    private void resetRejectedPaths() {
        rejectedPaths = null;
        if (channels == null) return;
        for (ListenerHandle lh : listeners) {
            if (lh.paths != null) {
                if (lh.templates.isEmpty()) {
                    for (String localPath : lh.paths) {
                        if (!channels.containsKey(localPath)) {
                            rejectedPaths = new HashSet<>();
                            return;
                        }
                    }
                } else {
                    rejectedPaths = new HashSet<>();
                    return;
                }
            }
        }
    }
    
    private static Map<String,String> getAllStates(StateBundle bundle) {
        Map<String,String> out = bundle.getAllStatesAsStrings();
        for (String component : bundle.getComponentsWithStates()) {
            addAllStates(bundle.getComponentStateBundle(component), component +"/", out);
        }
        return out;
    }
    
    private static void addAllStates(StateBundle bundle, String prefix, Map<String,String> out) {
        bundle.getInternalStates().forEach((key, value) -> {
            out.put(prefix + key, value);
        });
        bundle.getDecodedStates().forEach((key, value) -> {
            if (value != null && !(value instanceof DataProviderState)) {
                out.put(prefix + key, value.toString());
            }
        });
        bundle.getComponentsWithStates().forEach(component -> {
            addAllStates(bundle.getComponentStateBundle(component), prefix + component +"/", out);
        });
    }
    
    
// -- Local classes : ----------------------------------------------------------

    /**
     * Local (agent specific) handle for a listener.
     * Created by {@code AgentHandle.createLocalListenerHangle(AgentStatusAggregator.GlobalListenerHandle)}. 
     */
    private class ListenerHandle {

        /** Global listener handle. */
        final AgentStatusAggregator.GlobalListenerHandle global;
        
        /**
         * Explicit inner paths of watched channels;
         * {@code null} if this listener is only interested in agent connects and disconnects.
         */
        Set<String> paths;
        
        /** Unsatisfied path templates; may be empty, never {@code null} once initialized. */
        List<String> templates; // inner path templates
        
        ListenerHandle(AgentStatusAggregator.GlobalListenerHandle global) {
            this.global = global;
        }
        
        /**
         * Initializes the listener handle.
         * @return True is the listener is actually interested in the agent of this AgentHandle.
         */
        boolean init() {
            if (global.agents == null || global.agents.contains(agentInfo.getName())) {
                if (global.channels == null) {
                    paths = new HashSet<>();
                } else {
                    String prefix = agentInfo.getName() +"/";
                    for (String s : global.channels) {
                        if (isSelector(s)) {
                            String a = parseSelector(s).get("agent.name");
                            if (a == null || agentInfo.getName().equals(a)) {
                                paths = new HashSet<>();
                                break;
                            }
                        } else if (s.startsWith(prefix) || s.startsWith("/")) {
                            paths = new HashSet<>();
                            break;
                        }
                    }
                }
                return true;
            } else {
                if (global.channels != null) {
                    String prefix = agentInfo.getName() +"/";
                    for (String s : global.channels) {
                        if (isSelector(s)) {
                            String a = parseSelector(s).get("agent.name");
                            if (agentInfo.getName().equals(a)) {
                                paths = new HashSet<>();
                                break;
                            }
                        } else if (s.startsWith(prefix)) {
                            paths = new HashSet<>();
                            break;
                        }
                    }
                }
                return paths != null;
            }
        }
        
        void processDictionary() {
            
            if (paths == null) return;
            
            // Set paths and templates
            
            templates = new ArrayList<>();
            boolean inAgentList = global.agents == null || global.agents.contains(agentInfo.getName());
            String prefix = agentInfo.getName() + "/";
            List<MutableAgentChannel> allChannels = dictionary.getAllChannels();
            for (String s : global.channels) {
                if (isSelector(s)) {
                    for (MutableAgentChannel channel : allChannels) {
                        if (acceptSelector(channel, s)) {
                            paths.add(channel.getLocalPath());
                        }
                    }
                } else {
                    if (s.startsWith(prefix)) {
                        s = s.substring(prefix.length());
                    } else if (inAgentList && s.startsWith("/")) {
                        s = s.substring(1);
                    } else {
                        continue;
                    }
                    if (s.endsWith("/") || s.isEmpty()) {
                        templates.add(s);
                    } else {
                        paths.add(s);
                    }
                }
            }
            
            // Update channel map in agent handle
            
            ArrayList<AgentChannel> added = new ArrayList<>();
            synchronized (AgentHandle.this) {
                for (MutableAgentChannel channel : allChannels) {
                    String localPath = channel.getLocalPath();
                    if (paths.contains(localPath)) {
                        channels.put(localPath, channel);
                        added.add(channel);
                    } else {
                        for (String template : templates) {
                            if (localPath.startsWith(template)) {
                                paths.add(localPath);
                                channels.put(localPath, channel);
                                added.add(channel);
                                break;
                            }
                        }
                    }
                }
            }
            
            // Clean up
            
            if (templates.isEmpty()) {
                templates = Collections.emptyList();
            } else {
                ((ArrayList)templates).trimToSize();
            }
            paths = new HashSet<>(paths);
            
            // Notify the listener
            
            AgentStatusEvent event = new AgentStatusEvent(aggregator, agentInfo, null, added, null);
            global.listener.configure(event);
        }
        
        void fireConnect(AgentStatusEvent event) {
            if (event == null) event = new AgentStatusEvent(aggregator, agentInfo);
            global.listener.connect(event);
        }
        
        void fireChange(AgentStatusEvent event) {
            AgentStatusEvent e = event.filter(paths);
            if (!e.isEmpty()) {
                global.listener.statusChanged(e);
            }
        }
        
        boolean isInterestedInChannels() {
            return paths != null;
        }
        
    }
    
// -- Utility methods : --------------------------------------------------------
    
    /** Returns {@code true} if the specified string is in a channel selector format (as opposed to explicit path or template). */
    static boolean isSelector(String s) {
        if (s.contains("=")) return true;
        return ! s.contains("/");
    }
    
    /**
     * Applies selector string to a channel.
     * Selector is in
     * {@code [localAgent.getKey=value&][localAgent.type=value&][localAgent.key[=value]&�&localAgent.key[=value]&][key[=value]&�&key[=value]]}
     * format.
     * 
     * @param channel Channel to test.
     * @param selector Selector.
     * @return {@code true} is the channel satisfies the selector.
     */
    static boolean acceptSelector(AgentChannel channel, String selector) {
        for (String condition : selector.split("&")) {
            String[] ss = condition.split("=");
            String key = ss[0];
            Object value;
            if (key.startsWith("agent.")) {
                key = key.substring("agent.".length());
                switch (key) {
                    case "name":
                        value = channel.getAgentName(); break;
                    case "type":
                        value = channel.getAgent().getType(); break;
                    default:
                        value = channel.get(key);
                }
            } else {
                value = channel.get(key);
            }
            if (ss.length == 1) {
                if (value == null) return false;
            } else {
                if (value == null) {
                    if (!"null".equals(ss[1])) return false;
                } else {
                    if (!value.toString().equals(ss[1])) return false;
                }
            }
        
        }
        return true;
    }
    
    static Map<String,String> parseSelector(String selector) {
        String[] ss = selector.split("&");
        HashMap<String,String> out = new LinkedHashMap<>(ss.length*2);
        for (String s : ss) {
            String[] tt = s.split("=");
            out.put(tt[0], tt.length == 1 ? null : tt[1]);
        }
        return out;
    }
    
    static boolean matchTemplate(String path, String template) {
        if (template.startsWith("/")) {
            path = path.substring(path.indexOf("/"));
        }
        return template.endsWith("/") ? path.startsWith(template) : path.equals(template);
    }
    
    static boolean matchLocalTemplate(String localPath, String localTemplate) {
        if (localTemplate.isEmpty()) return true;
        return localTemplate.endsWith("/") ? localPath.startsWith(localTemplate) : localPath.equals(localTemplate);
    }
    
    

}
