package org.lsst.ccs.gconsole.plugins.tracer.filters;

import java.awt.Color;
import java.time.Instant;
import java.util.*;
import java.util.logging.Level;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RuntimeInfo;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.messages.StatusRuntimeInfo;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.plugins.tracer.FilteredMessage;

/**
 * Standard configurable filter.
 *
 * @author onoprien
 */
@ConsoleLookup(
        name = "Standard Configurable Filter",
        path = "BuiltIn/Standard Configurable",
        description = "Generic filter that can be configured to accept messages based on bus type, message type, log level, etc. The filter applies default formatting.<p>"
                + "In order to be accepted, a message should satisfy all criteria described below.<p>"
                + "<b>Optional parameters:</b>"
                + "<dl>"
                + "<dt>name</dt><dd>Filter name.</dd>"
                + "<dt>subsystems</dt><dd>Comma-separated list of subsystems accepted by this filter.</dd>"
                + "<dt>properties</dt><dd>Comma-separated list of properties the subsystem must have to be accepted.</dd>"
                
                + "<dt>commandMessages</dt><dd>If true, messages received on the command bus are accepted.</dd>"
                + "<dt>statusMessages</dt><dd>If true, messages received on the status bus are accepted.</dd>"
                + "<dt>logMessages</dt><dd>If true, messages received on the log bus are accepted.</dd>"
                
                + "<dt>heartBeat</dt><dd>If true, heartbeat status messages are accepted.</dd>"
                + "<dt>jvmInfo</dt><dd>If true, JVM information status messages are accepted.</dd>"
                + "<dt>stateChange</dt><dd>If true, subsystem state change status messages are accepted.</dd>"
                + "<dt>alert</dt><dd>If true, alert status messages are accepted.</dd>"
                + "<dt>data</dt><dd>If true, status messages with trending data are accepted.</dd>"
                + "<dt>service</dt><dd>If true, service status messages are accepted.</dd>"
                + "<dt>others</dt><dd>If true, all other status messages are accepted.</dd>"
                
                + "<dt>logLevel</dt><dd>Log messages below the specified level are rejected.</dd>"
                + "</dl>"
)
public class StandardMessageFilter extends AbstractMessageFilter {

// -- Fields : -----------------------------------------------------------------
    
    static public final Color COLOR_ERROR = new Color(200, 0, 0);
    static public final Color COLOR_WARNING = new Color(0, 0, 255);
    static public final Color COLOR_GOOD = new Color(0, 200, 0);
    static public final Color COLOR_FINE = new Color(100, 100, 100);
    
    private final String name;
    private final List<String> agents;
    private final List<Map.Entry<String,String>> properties;
    
    private final boolean commandMessages;
    private final boolean statusMessages;
    private final boolean logMessages;
    
    private final boolean heartBeat;
    private final boolean jvmInfo;
    private final boolean stateChange;
    private final boolean alert;
    private final boolean data;
    private final boolean service;
    private final boolean others;
    
    private final int logLevel;

// -- Life cycle : -------------------------------------------------------------
    
    public StandardMessageFilter(String name, String subsystems, String subsystemProperties,
            boolean commandMessages, boolean statusMessages, boolean logMessages,
            boolean heartBeat, boolean jvmInfo, boolean stateChange, boolean alert, boolean trending, boolean service, boolean others,
            String logLevel) {

        this.name = name;
        
        if (subsystems == null || subsystems.trim().isEmpty()) {
            agents = null;
        } else {
            String[] ss = subsystems.split(",");
            agents = new ArrayList<>(ss.length);
            for (String s : ss) {
                this.agents.add(s.trim());
            }
        }
        
        if (subsystemProperties == null || subsystemProperties.trim().isEmpty()) {
            properties = null;
        } else {
            String[] ss = subsystemProperties.split(",");
            properties = new ArrayList<>(ss.length);
            for (String s : ss) {
                String[] kv = s.split("=");
                if (kv.length == 1) {
                    properties.add(new AbstractMap.SimpleEntry(kv[0].trim(), null));
                } else {
                    properties.add(new AbstractMap.SimpleEntry(kv[0].trim(), kv[1].trim()));
                }
            }
        }
        
        this.commandMessages = commandMessages;
        this.statusMessages = statusMessages;
        this.logMessages = logMessages;
        
        this.heartBeat = heartBeat;
        this.jvmInfo = jvmInfo;
        this.stateChange = stateChange;
        this.alert = alert;
        this.data = trending;
        this.service = service;
        this.others = others;
        
        this.logLevel = logLevelIntValue(logLevel);
    }
    
    
// -- Getters : ----------------------------------------------------------------

    @Override
    public String getPath() {
        return getClass().getAnnotation(ConsoleLookup.class).path();
    }

    @Override
    public String getDescription() {
        return "Generic folter that can be configured to accept messages based on bus type, message type, log level, etc. The filter applies default formatting.";
    }
    
    
// -- Filtering : --------------------------------------------------------------

    @Override
    public FilteredMessage test(FilteredMessage filteredMessage) {
        
        BusMessage bm = filteredMessage.getBusMessage();
        
        if (bm instanceof LogMessage) {
            if (!logMessages) return null;
        } else if (bm instanceof StatusMessage) {
            if (!statusMessages) return null;
        } else {
            if (!commandMessages) return null;
        }
        
        AgentInfo agent = bm.getOriginAgentInfo();
        String agentName = agent.getName();
        if (agents != null && !agents.contains(agentName)) return null;
        
        if (properties != null) {
            for (Map.Entry<String,String> e : properties) {
                String v = agent.getAgentProperty(e.getKey());
                if (v == null || (e.getValue() != null && !e.getValue().equals(v))) return null;
            }
        }
        
        StringBuilder sb = new StringBuilder();
        if (agents == null || agents.size() > 1) {
            sb.append(agentName).append(" ");
        }
        
        Instant instant = bm.getCCSTimeStamp().getUTCInstant();
        sb.append(Const.DEFAULT_T_FORMAT.format(instant)).append(": ");
        
        if (bm instanceof LogMessage) { // log bus
            
            LogMessage lm = (LogMessage)bm;
            String s = lm.getFormattedDetails().trim();
            if (s == null || s.isEmpty() || s.contains("java.util.logging.LogRecord@")) return null;  // FIXME (ask Max to fix)
            sb.append(s);
            
            int level = logLevelIntValue(lm.getLevel());
            if (level < logLevel) return null;
            if (level >= Level.SEVERE.intValue()) {
                filteredMessage.setColor(COLOR_ERROR);
            } else if (level >= Level.WARNING.intValue()) {
                filteredMessage.setColor(COLOR_WARNING);
            } else if (level <= Level.FINE.intValue()) {
                filteredMessage.setColor(COLOR_FINE);
            }
            
        } else if (bm instanceof StatusMessage) { // status bus
            
            StatusMessage sm = (StatusMessage)bm;
            
            if (sm instanceof StatusHeartBeat) {
                
                if (!heartBeat) return null;
                sb.append("Heart Beat.");
                
            } else if (sm instanceof StatusSubsystemData) {
                
                if (!data) return null;
                try {
                    sb.append(sm.getEncodedData());
                } catch (Exception x) {
                    sb.append(sm);
                }
                if (sb.length() > 200) sb.delete(200, Integer.MAX_VALUE).append("...");
                
                
            } else if (sm instanceof StatusStateChangeNotification) {
                
                if (!stateChange) return null;
                StatusStateChangeNotification m = (StatusStateChangeNotification) sm;
                StateBundle change = m.getNewState().diffState(m.getOldState());
                sb.append("State change: ").append(change);
                
            } else if (sm instanceof StatusRuntimeInfo) {
                
                if (!jvmInfo) return null;
                RuntimeInfo rt = ((StatusRuntimeInfo) sm).getRuntimeInfo();
                sb.append("JVM info: ").append(rt.getNThreads()).append(" threads, ");
                sb.append(rt.getFreeMemory()/1000000).append(" MB free, ");
                sb.append(Math.round(100.*rt.getSystemCpuLoad())).append("% CPU load.");
                
            } else if (sm instanceof StatusRaisedAlert) {
                
                if (!alert) return null;
                StatusRaisedAlert m = (StatusRaisedAlert) sm;
                AlertState level = m.getRaisedAlertSummary().getAlertState();
                switch (level) {
                    case ALARM: filteredMessage.setColor(COLOR_ERROR); break;
                    case WARNING: filteredMessage.setColor(COLOR_WARNING); break;
                    case NOMINAL: filteredMessage.setColor(COLOR_GOOD); break;
                }
                sb.append(level).append(" ");
                Alert alarm = m.getRaisedAlert();
                sb.append(alarm.getAlertId()).append(" ").append(alarm.getDescription()).append("\n");
                sb.append("Cause: ").append(m.getCause());
                
            } else if (sm instanceof StatusClearedAlert) {
                
                if (!alert) return null;
                StatusClearedAlert m = (StatusClearedAlert) sm;
                sb.append("Cleared alerts: ");
                for (String id : m.getClearAlertIds()) {
                    sb.append(id).append(", ");
                }
                sb.delete(sb.length()-2, sb.length());
                filteredMessage.setColor(COLOR_GOOD);
                
            } else if (sm instanceof StatusDataProviderDictionary) {
                
                if (!service) return null;
                sb.append("Data channels dictionary received.");
                filteredMessage.setColor(COLOR_FINE);
                
            } else if (sm instanceof StatusConfigurationInfo) {
                
                if (!service) return null;
                StatusConfigurationInfo m = (StatusConfigurationInfo) sm;
                sb.append("Configuration received: ").append(m.getConfigurationInfo().getConfigurationName());
                filteredMessage.setColor(COLOR_FINE);
                
            } else {
                
                if (!others) return null;
                try {
                    sb.append(sm.getEncodedData());
                } catch (Exception x) {
                    sb.append(sm);
                }
                if (sb.length() > 200) sb.delete(200, Integer.MAX_VALUE).append("...");
            }
            
        } else { // command bus

            if (bm instanceof CommandRequest) {
                CommandRequest cr = (CommandRequest) bm;
                sb.append("Command ").append(cr.getBasicCommand().getCommand()).append(" to ").append(cr.getDestination()).append(".");
            } else if (bm instanceof CommandNack) {
                CommandNack cn = (CommandNack) bm;
                sb.append("Command not accepted. Reason: ").append(cn.getReason());
                filteredMessage.setColor(COLOR_WARNING);
            } else if (bm instanceof CommandAck) {
                sb.append("Command accepted.");
            } else if (bm instanceof CommandResult) {
                CommandResult cr = (CommandResult) bm;
                if (cr.wasSuccessful()) {
                    sb.append("Command executed");
                    Object out = cr.getResult();
                    if (out == null) {
                        sb.append(".");
                    } else {
                        sb.append(": ").append(out);
                    }
                } else {
                    sb.append("Command failed: ").append(cr.getResult());
                    filteredMessage.setColor(COLOR_ERROR);
                }
            } else {
                sb.append(bm);
            }

        }
        
        filteredMessage.setText(sb.toString());
        return filteredMessage;
    }
    
// -- Static utility methods : -------------------------------------------------
    
    static public int logLevelIntValue(String level) {
        if (level == null) return Level.ALL.intValue();
        switch (level.toUpperCase()) {
            case "OFF": case "O":
                return Level.OFF.intValue();
            case "SEVERE": case "FATAL": case "ERROR": case "S": case "E":
                return Level.SEVERE.intValue();
            case "WARNING": case "WARN": case "W":
                return Level.WARNING.intValue();
            case "INFO": case "I":
                return Level.INFO.intValue();
            case "CONFIG": case "C":
                return Level.CONFIG.intValue();
            case "FINE": case "DEBUG": case "F": case "D":
                return Level.FINE.intValue();
            case "FINER":
                return Level.FINER.intValue();
            case "FINEST": case "TRACE": case "T":
                return Level.FINEST.intValue();
            case "ALL": case "A": default:
                return Level.ALL.intValue();
        }
    }
    
// -- Specific subclasses : ----------------------------------------------------
    
    @ConsoleLookup(
        name = "Standard Formatter",
        path = "BuiltIn/Formatters/Standard",
        description = "The filter applies default formatting.")
    public static class All extends StandardMessageFilter {
        
        public All() {
            super("All", null, null, true, true, true, true, true, true, true, true, true, true, null);
        }
        
        public All(String subsystem) {
            super(subsystem, subsystem, null, true, true, true, true, true, true, true, true, true, true, null);
        }
        
    }
    
    @ConsoleLookup(
        name = "Default Filter",
        path = "BuiltIn/Default",
        description = "This filter selects messages that are relevant in most circumstances, and applies default formatting.")
    public static class Default extends StandardMessageFilter {
        
        public Default() {
            super("Default", null, null, false, true, true, false, false, true, true, true, false, true, "INFO");
        }
        
        public Default(String subsystem) {
            super(subsystem, subsystem, null, false, true, true, false, false, true, true, true, false, true, "INFO");
        }
        
    }

}
