package org.lsst.ccs.gconsole.plugins.dictionary;

import java.awt.Color;
import java.awt.Component;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import javax.swing.JTree;
import javax.swing.SwingWorker;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.utilities.structs.TreeBranch;

/**
 * Represent the dictionary of the selected subsystem as a tree.
 * @author emarin
 */
public class DictionaryTreePanel extends JTree {
    private static final long serialVersionUID = -3173309107169865621L;
    
    private TreeBranch<ComponentDictionary> dictionaryTree;
    private final CommandTreeRenderer renderer;
    
    DictionaryTreePanel() {
        super();
        this.setModel(new DefaultTreeModel(dictionaryTree));
        
        getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        
        renderer = new CommandTreeRenderer();
        setCellRenderer(renderer);
    }
    
    void updateData(AgentDictionaries dict){
        new BuildDictionaryWorker(dict).execute();
    }
    
    public void updateRenderer(){
        this.revalidate();
        this.repaint();
    }
    
    private class BuildDictionaryWorker extends SwingWorker<TreeBranch<ComponentDictionary>, String> {
        
        private final AgentDictionaries dict;
        
        BuildDictionaryWorker(AgentDictionaries dict) {
            this.dict = dict;
        }
        
        @Override
        protected TreeBranch<ComponentDictionary> doInBackground() {
            try {
                
                // Extract the command sets for the requested destination
                Map<String, Dictionary> dictionaries = new HashMap<>(dict.dictionaries);
                Map<String, CommandSet> commandSets = dict.commandSets;
                TreeBranch<ComponentDictionary> dictionaryTree = null;
                
                String rootName = dict.ai.getName();
                // First node building
                Dictionary rootDictionary = dictionaries.remove(rootName);
                dictionaryTree = new TreeBranch<ComponentDictionary>(new ComponentDictionary(rootName, rootName, rootDictionary, commandSets.get(rootName)));
                TreeBranch<ComponentDictionary> treeWalker = dictionaryTree;
                // Tree building
                for (Map.Entry<String,Dictionary> entry : dictionaries.entrySet()){
                    treeWalker = dictionaryTree;
                    String[] pathTab = entry.getKey().split("/");
                    boolean found;
                    for (int i = 1;i < pathTab.length;i++){
                        found = false;
                        if (treeWalker.getChildren() != null){
                            Iterator<TreeBranch<ComponentDictionary>> it = treeWalker.getChildIterator();
                            while(it.hasNext() && !found){
                                TreeBranch<ComponentDictionary> currIt = it.next();
                                if (currIt.getContent().name.equals(pathTab[i])){
                                    treeWalker = currIt;
                                    found = true;
                                    if (i == pathTab.length - 1){
                                        // The node exists but its dictionary has not been set
                                        treeWalker.setContent(new ComponentDictionary(entry.getKey(), pathTab[i],entry.getValue(), commandSets.get(entry.getKey())));
                                    }
                                }
                            }
                        }
                        if (!found){
                            treeWalker = new TreeBranch(treeWalker, new ComponentDictionary(entry.getKey(), pathTab[i],entry.getValue(), commandSets.get(entry.getKey())));
                        }
                    }
                }
                
                TreeBranch<ComponentDictionary> root = dictionaryTree;
                removeLeafsWithEmptyDicts(root);
                return dictionaryTree;
            } catch (Exception ex) {
                ex.printStackTrace();
                throw ex;
            }
        }
        
        private void removeLeafsWithEmptyDicts(TreeBranch<ComponentDictionary> node) {
            Iterator<TreeBranch<ComponentDictionary>> it = node.getChildIterator();
            while(it.hasNext()) {
                TreeBranch<ComponentDictionary> currIt = it.next();
                removeLeafsWithEmptyDicts(currIt);
                if(currIt.getChildCount()==0 && !currIt.getContent().dictionary.iterator().hasNext()) {
                    it.remove();
                }
            }
        }
        
        @Override
        public void done() {
            try {
                dictionaryTree = get();
                if (dictionaryTree != null){
                    setModel(new DefaultTreeModel(dictionaryTree));
                    getSelectionModel().setSelectionPath(new TreePath(getModel().getRoot()));
                }
                // revalidate();
                //repaint();
                
            } catch (InterruptedException | ExecutionException ex) {
                ex.printStackTrace();
            }
        }
        
    }
    
    private class CommandTreeRenderer extends DefaultTreeCellRenderer{
        
        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean selected, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            super.getTreeCellRendererComponent(tree, value, selected, expanded, leaf, row, hasFocus);
            setForeground(greyOut((TreeBranch<ComponentDictionary>)value) ? Color.GRAY : Color.BLACK);
            
            return this;
        }
        
        private boolean greyOut(TreeBranch<ComponentDictionary> tree) {
            for(TreeBranch<ComponentDictionary> child : tree.getChildren()) {
                if (! greyOut(child)) {
                    return false;
                }
            }
            Dictionary d = tree.getContent().dictionary;
            for(DictionaryCommand dc : d) {
                if(dc.getLevel() <= d.getLevel() && d.getVisibilityForType(dc.getType())) {
                    return false;
                }
            }
            return true;
        }
        
        
    }
    
}