package org.lsst.ccs.gconsole.plugins.dictionary;

import java.awt.Component;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.ArrayList;
import java.util.List;
import javax.swing.ImageIcon;
import javax.swing.JCheckBox;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.RowFilter;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableRowSorter;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.command.RouteSelectionCommandSet;

/**
 * Lists available agents, and their attached locking information.
 * @author emarin
 */
public class AgentDictionaryTable extends JTable implements ItemListener {
    
    private static final String[] COLUMN_NAMES = new String[] {"Agent name", "Lock", "Level"};
    
    private AgentDefaultTableModel tableModel = new AgentDefaultTableModel(COLUMN_NAMES); 
    
    private TableRowSorter<AgentDefaultTableModel> sorter = new TableRowSorter<AgentDefaultTableModel>(tableModel); 
    
    private boolean[] checkSelected = new boolean[AgentInfo.AgentType.values().length]; 
    
    private final RouteSelectionCommandSet rsc;
    
    public AgentDictionaryTable(RouteSelectionCommandSet commandSet) {
        setModel(tableModel);
        setRowSorter(sorter);
        getColumn(COLUMN_NAMES[0]).setCellRenderer(new DefaultTableCellRenderer()  {

            @Override
            public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
                super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
                if (value != null){
                    AgentDictionaries ds = (AgentDictionaries)value;
                    setText(ds.ai.getName());
                    setToolTipText(ds.ai.getName());
                }
                return this;
            }
        });
                
        getColumn(COLUMN_NAMES[1]).setCellRenderer(new DefaultTableCellRenderer()  {
            
            @Override
            public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
                setHorizontalAlignment(CENTER);
                if((boolean)value) {
                    setIcon(new ImageIcon(this.getClass().getResource("ic_lock_black_18dp.png"),"locked"));
                } else {
                    setIcon(new ImageIcon(this.getClass().getResource("ic_lock_open_black_18dp.png"), "unlocked"));
                }
                return this;
            }
        });
        
        getColumn(COLUMN_NAMES[0]).setMinWidth(150);
        
        setRowHeight(27);

        setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
//        setCellSelectionEnabled(true);
        setShowGrid(false);
        this.rsc = commandSet;
    }
    
   

    @Override
    public void itemStateChanged(ItemEvent e) {
        JCheckBox checkBox = (JCheckBox)e.getSource();
        String name = checkBox.getName();
        AgentInfo.AgentType type = AgentInfo.AgentType.valueOf(name);
        checkSelected[type.ordinal()]=checkBox.isSelected();

        // Create array of filters
        ArrayList<RowFilter<AgentDefaultTableModel,Integer>> filterList = new ArrayList<RowFilter<AgentDefaultTableModel,Integer>>();
        
        for (AgentInfo.AgentType agentType : AgentInfo.AgentType.values()){

                if (checkSelected[agentType.ordinal()]){
                    filterList.add(new RowFilter<AgentDefaultTableModel,Integer>(){
                        
                        @Override
                        public boolean include(RowFilter.Entry<? extends AgentDefaultTableModel, ? extends Integer> entry) {
                            AgentInfo agent = ((AgentDictionaries)entry.getModel().getValueAt(entry.getIdentifier(), 0)).ai;
                            if (agent.getType().equals(agentType)){
                                return true;
                            }
                            else return false;
                        }
                        
                    });
                }
        }
        
        // updateAgentsList Filter
        RowFilter filter = RowFilter.orFilter(filterList);
        sorter.setRowFilter(filter);
    }
    
    AgentDictionaries getSelectedAgentDictionaries() {
        if(getSelectedRow() >=0){ // If an agent is selected
            AgentDictionaries ds = (AgentDictionaries)getValueAt(getSelectedRow(), 0);
            ds.update(rsc);
            return ds;
        } else return null;
    }
    
    public void addAgentToList(AgentInfo ai) {
        ((DefaultTableModel)getModel()).addRow(new Object[]{new AgentDictionaries(ai)});
    }
    
    public void removeAgentFromList(AgentInfo ai) {
        ((AgentDefaultTableModel)getModel()).removeRow(ai);
    }
    
    public void updateAgentsList(List<AgentInfo> ais) {
        Object[][] data = new Object[ais.size()][1];
        for (int i = 0; i < ais.size(); i++) {
            AgentInfo ai = ais.get(i);
            data[i][0] = new AgentDictionaries(ai);
        }
        (((AgentDefaultTableModel)getModel())).update(data);
    }
    
    void updateLevelForAgent(String agentName, int level) {
        (((AgentDefaultTableModel)getModel())).updateLevelForAgent(agentName, level);
        
    }
    
    void updateLockForAgent(String agentName, boolean locked) {
        (((AgentDefaultTableModel)getModel())).updateLockForAgent(agentName, locked);
        
    }
}