package org.lsst.ccs.gconsole.plugins.trending.timeselection;

import java.awt.Dimension;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.CopyOnWriteArrayList;
import javafx.application.Platform;
import javafx.embed.swing.JFXPanel;
import javafx.scene.Scene;
import javafx.scene.control.DatePicker;
import javafx.scene.paint.Color;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JFormattedTextField;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerDateModel;
import javax.swing.SwingUtilities;
import javax.swing.text.DefaultFormatter;
import javax.swing.text.NumberFormatter;

/**
 * A GUI component that allows choosing zoned date and time (an instant on a time line).
 *
 * @author onoprien
 */
public class JTimeDateChooser extends JPanel {

// -- Fields : -----------------------------------------------------------------
    
    private final int VSPACE = 5;
    private final int HSPACE = 10;
    
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();
    private boolean armed = true;
    private boolean armedFX = true;
    
    private final JFXPanel fxPanel;
    private DatePicker datePicker;
    private LocalDate date;
    private ZoneId zone = ZoneId.systemDefault();
    
    private final JSpinner timeSpinner;
    private final SpinnerDateModel timeModel;

// -- Life cycle : -------------------------------------------------------------
    
    static JTimeDateChooser getInstance() {
        return JTimeDateChooser.getInstance(null);
    }

    static JTimeDateChooser getInstance(ZonedDateTime time) {
        Platform.setImplicitExit(false);
        JTimeDateChooser instance = new JTimeDateChooser();
        instance.init();
        if (time != null) {
            instance.set(time);
        }
        return instance;
    }
    
    private JTimeDateChooser() {
        setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
    
        fxPanel = new JFXPanel();
//        fxPanel.setMaximumSize(fxPanel.getPreferredSize());
        add(fxPanel);
        
        add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        
        timeModel = new SpinnerDateModel(new Date(0), new Date(0), new Date(Long.MAX_VALUE/2L), Calendar.HOUR);
        timeSpinner = new JSpinner(timeModel);
        JSpinner.DateEditor editor = new JSpinner.DateEditor(timeSpinner, "HH:mm:ss");
        timeSpinner.setEditor(editor);
//        JFormattedTextField tf = editor.getTextField();
//        try {
//            DefaultFormatter f = (DefaultFormatter) tf.getFormatter();
//            f.setCommitsOnValidEdit(true);
//            f.setAllowsInvalid(false);
//        } catch (ClassCastException x) {}



//        timeSpinner.setMaximumSize(timeSpinner.getPreferredSize());
        timeSpinner.addChangeListener(e -> fire());
        add(timeSpinner);
        
        add(Box.createHorizontalGlue());
    }
    
    private void init() {
        Platform.runLater(() -> {
            datePicker = new DatePicker();
            datePicker.setOnAction(e -> {
                if (armedFX) {
                    date = datePicker.getValue();
                    SwingUtilities.invokeLater(this::fire);
                }
            });
            Scene scene = new Scene(datePicker, Color.ALICEBLUE);
            fxPanel.setScene(scene);
        });
    }
    
    
// -- Getters and setters : ----------------------------------------------------
    
    /**
     * Returns the currently selected date and time.
     * @return Selected time.
     */
    public Instant get() {
        ZonedDateTime time = timeModel.getDate().toInstant().atZone(ZoneId.systemDefault());
        LocalTime localTime = time.toLocalTime();
        LocalDateTime localDateTime = localTime.atDate(date);
        return localDateTime.atZone(zone).toInstant();
    }
    
    public void set(Instant instant) {
        ZonedDateTime dt = instant.atZone(zone);
        armed = false;
        timeModel.setValue(new Date(instant.toEpochMilli()));
        date = dt.toLocalDate();
        Platform.runLater(() -> {
            armedFX = false;
            datePicker.setValue(date);
            armedFX = true;
        });
        armed = true;
    }
    
    public void set(ZonedDateTime time) {
        set(time.toInstant());
    }
    
    public ZoneId getZone() {
        return zone;
    }
    
    public void set(Date date) {
        set(Instant.ofEpochMilli(date.getTime()));
    }
    
    public void setZone(ZoneId zone) {
        Instant dt = get();
        this.zone = zone;
        set(dt);
    }

    @Override
    public void setEnabled(boolean enabled) {
        Platform.runLater(() -> {
            datePicker.setDisable(!enabled);
        });
        timeSpinner.setEnabled(enabled);
        super.setEnabled(enabled);
    }

// -- Handling listeners : -----------------------------------------------------
    
    public interface Listener {
        void changed(JTimeDateChooser source);
    }
    
    public void addListener(Listener listener) {
        listeners.addIfAbsent(listener);
    }
    
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }
    
    protected void fire() {
        if (armed) {
            listeners.forEach(listener -> listener.changed(this));
        }
    }

}
