package org.lsst.ccs.gconsole.plugins.trending;

import java.util.*;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;

/**
 * Combines several {@link TrendingSource}s into a single source.
 * <p>
 * If a channel is available from more than one constituent source, there is no guarantees
 * regarding the choice of the source that will be used to retrieve {@code TrendData} for
 * that channel.
 *
 * @author onoprien
 */
public class CompoundTrendingSource extends TrendingSource {

// -- Fields : -----------------------------------------------------------------
    
    private final Set<TrendingSource> sources = Collections.newSetFromMap(new IdentityHashMap<>());
    private Map<TrendingChannel,TrendingSource> chan2source = new HashMap<>();
    private final TrendingSource.Listener forwarder;

// -- Life cycle : -------------------------------------------------------------
    
    public CompoundTrendingSource() {
        forwarder = e -> {
            ArrayList<TrendingChannel> removed, added;
            synchronized (this) {
                Set<TrendingChannel> before = chan2source.keySet();
                chan2source = new HashMap<>();
                sources.forEach(source -> {
                    source.getChannels().forEach(channel -> {
                        chan2source.put(channel, source);
                    });
                });
                Set<TrendingChannel> after = new HashSet<>(chan2source.keySet());
                after.removeAll(before);
                added = new ArrayList<>(after);
                before.removeAll(chan2source.keySet());
                removed = new ArrayList<>(before);
            }
            fireEvent(removed, added);
        };
    }

// -- Implement TrendingSource -------------------------------------------------

    @Override
    synchronized public ArrayList<TrendingChannel> getChannels() {
        return new ArrayList<>(chan2source.keySet());
    }

    @Override
    public TrendData get(TrendingChannel channel, long begin, long end, EnumSet<Trend.Meta> extra, TrendData data) {
        TrendingSource source;
        synchronized (this) {
            source = chan2source.get(channel);
        }
        return source == null ? null : source.get(channel, begin, end, extra, data);
    }
    
    
// -- Modifications : ----------------------------------------------------------
    
    public boolean add(TrendingSource source) {
        return add(Collections.singletonList(source));
    }
    
    public boolean add(List<TrendingSource> sourcesToAdd) {
        ArrayList<TrendingChannel> addedChannels = new ArrayList<>();
        synchronized (this) {
            sourcesToAdd.forEach(source -> {
                if (sources.add(source)) {
                    List<TrendingChannel> channels = source.getChannels();
                    channels.forEach(channel -> {
                        TrendingSource other = chan2source.put(channel, source);
                        if (other == null) addedChannels.add(channel);
                    });
                    source.addListener(forwarder);
                }
            });
        }
        if (!addedChannels.isEmpty()) {
            fireEvent(null, addedChannels);
            return true;
        }
        return false;
    }
    
    public boolean remove(TrendingSource source) {
        return remove(Collections.singletonList(source));
    }
    
    public boolean remove(List<TrendingSource> sourcesToRemove) {
        ArrayList<TrendingChannel> removedChannels = new ArrayList<>();
        synchronized (this) {
            sourcesToRemove.forEach(source -> {
                if (sources.remove(source)) {
                    List<TrendingChannel> channels = source.getChannels();
                    channels.forEach(channel -> {
                        chan2source.remove(channel);
                    });
                    source.removeListener(forwarder);
                }
            });
            if (!removedChannels.isEmpty()) {
                sources.forEach(source -> {
                    source.getChannels().forEach(channel -> {
                        if (removedChannels.remove(channel)) {
                            chan2source.put(channel, source);
                        }
                    });
                });
            }
        }
        if (!removedChannels.isEmpty()) {
            fireEvent(null, removedChannels);
            return true;
        }
        return false;
    }
    
    public boolean set(TrendingSource source) {
        return set(Collections.singletonList(source));
    }
    
    public boolean set(List<TrendingSource> sourcesToSet) {
        ArrayList<TrendingChannel> addedChannels, removedChannels;
        synchronized (this) {
            removedChannels = getChannels();
            sources.forEach(source -> source.removeListener(forwarder));
            sources.clear();
            chan2source.clear();
            addedChannels = new ArrayList<>();
            sourcesToSet.forEach(source -> {
                if (sources.add(source)) {
                    List<TrendingChannel> channels = source.getChannels();
                    channels.forEach(channel -> {
                        TrendingSource other = chan2source.put(channel, source);
                        if (other == null) addedChannels.add(channel);
                    });
                    source.addListener(forwarder);
                }
            });
        }
        if (!(addedChannels.isEmpty() && removedChannels.isEmpty())) {
            fireEvent(removedChannels, addedChannels);
            return true;
        }
        return false;
    }
    
    public boolean clear() {
        ArrayList<TrendingChannel> removedChannels;
        synchronized (this) {
            removedChannels = getChannels();
            sources.forEach(source -> source.removeListener(forwarder));
            sources.clear();
            chan2source.clear();
        }
        if (!removedChannels.isEmpty()) {
            fireEvent(removedChannels, null);
            return true;
        }
        return false;
    }

}
