package org.lsst.ccs.gconsole.plugins.dictionary;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.net.URL;
import java.util.List;
import java.util.concurrent.ExecutionException;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.tree.DefaultTreeModel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.command.RouteSelectionCommandSet;
import org.lsst.ccs.command.TokenizedCommand;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.logging.StackTraceFormats;
import org.lsst.ccs.utilities.structs.TreeBranch;

/**
 *
 * @author emarin
 */
public class DictionaryTreeMainPanel extends JPanel implements ListSelectionListener, AgentPresenceListener, ActionListener, AgentLockService.AgentLockUpdateListener, AgentLockService.AgentLevelListener {

    private final Logger logger = Logger.getLogger("org.lsst.ccs.plugin.jas3.dictionary");
    
    private final AgentMessagingLayer messagingAccess;
    
    private final AgentLockService agentLockService;
    
    private CommandListPanel commandListPanel;

    private final AgentDictionaryTable agentTable;

    private final DictionaryTreePanel dictionaryTree;
    private final ArgInputPanel argInputPanel;
    private final HTMLTextPane resultPane;
    
    //Optionally set the look and feel.
    DictionaryTreeMainPanel(AgentMessagingLayer messagingAccess, RouteSelectionCommandSet rsc, AgentLockService als) {
        super(new BorderLayout());
        
        this.messagingAccess = messagingAccess;
        this.agentLockService = als;

        // Agent Table creation
        agentTable = new AgentDictionaryTable(rsc);
        agentTable.getSelectionModel().addListSelectionListener(this);
        
        // Dictionary Tree creation
        dictionaryTree = new DictionaryTreePanel();

        // Create a scroll pane and adds the subsystem list to it
        JScrollPane agentsView = new JScrollPane(agentTable);
        agentTable.setFillsViewportHeight(true);

        //Create the scroll pane and add the dictionaryJTree to it. 
        JScrollPane treeView = new JScrollPane(dictionaryTree);

        // Create the command list viewing pane.
        commandListPanel = new CommandListPanel();
        dictionaryTree.addTreeSelectionListener(commandListPanel);
        //agentTable.getSelectionModel().addListSelectionListener(cmdList);

        commandListPanel.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                if (e.getClickCount() == 2) {
                    DictionaryCommand cmd = commandListPanel.getSelectedValue();
                    // Double click is only enabled for command without arguments
                    if (cmd.getArguments().length == 0) {
                        sendCommand();
                    }
                }
            }
        });

        JScrollPane cmdListView = new JScrollPane(commandListPanel);

        //Add the views to the TOP panel
        JPanel topPane = new JPanel(new GridLayout(1, 3));

        // Add subsystemview to the main Panel (not the top panel)
        JPanel leftPane = new JPanel(new BorderLayout());
        leftPane.setPreferredSize(new Dimension(300,800));

        // Refresh agents button
        URL url = this.getClass().getResource("refresh_24dp.png");
        JButton refreshAgentsButton = new JButton();
        refreshAgentsButton.setIcon(new ImageIcon(url));
        refreshAgentsButton.setToolTipText("Refresh");
        refreshAgentsButton.addActionListener(this);

        // JCheckBox panel
        JCheckBox [] checkBoxArray = new JCheckBox[AgentInfo.AgentType.values().length];
        JPanel checkBoxPanel = new JPanel(new GridLayout(4,2));
        for (AgentInfo.AgentType type : AgentInfo.AgentType.values()){
            checkBoxArray[type.ordinal()]= new JCheckBox(type.displayName());
            checkBoxArray[type.ordinal()].addItemListener(agentTable);
            checkBoxArray[type.ordinal()].setName(type.toString());
            checkBoxPanel.add(checkBoxArray[type.ordinal()]);
        }
        
        JPanel leftPanelBottom = new JPanel(new BorderLayout()) ;
        leftPanelBottom.add(new ConsoleBuiltInsPanel(), BorderLayout.NORTH);
        leftPanelBottom.add(checkBoxPanel,BorderLayout.CENTER);
        leftPanelBottom.add(refreshAgentsButton,BorderLayout.SOUTH);

        leftPane.add(agentsView, BorderLayout.CENTER);
        leftPane.add(leftPanelBottom, BorderLayout.SOUTH);

        add(leftPane, BorderLayout.WEST);

        // Add TreeView to the top panel
        topPane.add(treeView);

        //Add command list view to the top panel
        topPane.add(cmdListView);

        // Result Panel
        // create and add the argInputPanel to the top panel
        argInputPanel = new ArgInputPanel();
        commandListPanel.addListSelectionListener(argInputPanel);

        JPanel buttonPane = new JPanel();
        buttonPane.setLayout(new FlowLayout(FlowLayout.RIGHT));
        JButton sendCmdButton = new JButton(new ImageIcon(this.getClass().getResource("ic_send_black_24dp.png")));
        sendCmdButton.setText("Send");
        sendCmdButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand();

            }
        });

        buttonPane.add(sendCmdButton);
        add(buttonPane, BorderLayout.SOUTH);

        JPanel topPaneRightPanel = new JPanel(new BorderLayout());
        topPaneRightPanel.add(argInputPanel, BorderLayout.CENTER);
        topPaneRightPanel.add(buttonPane, BorderLayout.SOUTH);

        topPane.add(topPaneRightPanel);

        // Console Pane
        resultPane = new HTMLTextPane();
        JScrollPane resultPaneView = new JScrollPane(resultPane);

        // Split Panel creation
        JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        splitPane.setTopComponent(topPane);
        splitPane.setBottomComponent(resultPaneView);
        splitPane.setContinuousLayout(true);

        add(splitPane, BorderLayout.CENTER);
        
        // set the worker checkbox to true
        checkBoxArray[AgentInfo.AgentType.WORKER.ordinal()].setSelected(true);
    }
    
    // Agent selected has changed
    @Override
    public void valueChanged(ListSelectionEvent e) {
        dictionaryTree.setModel(new DefaultTreeModel(null));
        argInputPanel.clearForm();
        AgentDictionaries dict = agentTable.getSelectedAgentDictionaries();
        if(dict != null) {
            dictionaryTree.updateData(dict);
        } 
    }

    @Override
    public void connecting(AgentInfo agent) {
        SwingUtilities.invokeLater(
                () -> {
                    agentTable.addAgentToList(agent);
                    agentTable.updateLockForAgent(agent.getName(), agentLockService.getLockForAgent(agent.getName()) != null);
                    agentTable.updateLevelForAgent(agent.getName(), agentLockService.getLevelForAgent(agent.getName()));
                }
        );
    }
    
    @Override
    public void disconnecting(AgentInfo agent) {
        agentTable.removeAgentFromList(agent);
    }

    public void sendCommand() {
        TreeBranch<ComponentDictionary> destinationNode = (TreeBranch<ComponentDictionary>) dictionaryTree.getLastSelectedPathComponent();
        ComponentDictionary commandContext = destinationNode.getContent();
        String s = " ";
        DictionaryCommand cmd = commandListPanel.getSelectedValue();
        if (cmd != null) {
            s += argInputPanel.getFormattedArgumentValues();
            new SendCommandWorker(cmd.getCommandName() + s, commandContext.commandSet, commandContext.path).execute();
        }
    }

    /**
     * Refresh agents performed
     * @param e 
     */
    @Override
    public void actionPerformed(ActionEvent e) {
        new SwingWorker<List<AgentInfo>, String>() {
            
            @Override
            protected List<AgentInfo> doInBackground() throws Exception {
                
                return messagingAccess.getAgentPresenceManager().listConnectedAgents();
            }
            
            @Override
            public void done() {
                try {
                    List<AgentInfo> agentInfos = get();
                    agentTable.updateAgentsList(agentInfos);
                } catch (Exception ex) {
                    logger.error(ex);
                }
            }
        }.execute();
    }

    @Override
    public void onAgentLockUpdate(String agentName, AgentLock lock) {
        agentTable.updateLockForAgent(agentName, agentLockService.getLockForAgent(agentName) != null);
        commandListPanel.updateRenderer();
        dictionaryTree.updateRenderer();
    }

    @Override
    public void onAgentLevelChange(String agentName, int level) {
        agentTable.updateLevelForAgent(agentName, level);
        commandListPanel.updateRenderer();
        dictionaryTree.updateRenderer();
    }
    
    public class SendCommandWorker extends SwingWorker<String, String> {
        
        private final String cmd;
        private final CommandSet destination;
        private final String path;

        public SendCommandWorker(String cmdName, CommandSet dest, String path) {
            this.cmd = cmdName;
            this.destination = dest;
            this.path = path;
        }

        @Override
        protected String doInBackground() throws Exception {
            resultPane.insertHTML("invoke " + path + " " + cmd +" ...");
            try {
                Object res = destination.invoke(new TokenizedCommand(cmd));
                // in fact if this is an exception should mark as error!
                String strRes = null;
                if (res instanceof Exception) {
                    strRes = StackTraceFormats.toString((Exception) res, 1);
                } else if (res == null) {
                    strRes = "ok : VOID";
                } else {
                    strRes = String.valueOf(res);
                }
                return strRes;
            } catch (Exception e) {
                return "<html> <font color=\"#FF0000\"> Exception caught : " + StackTraceFormats.toString(e, 1) + "</font></html>";
            }
        }

        @Override
        public void done() {
            try {
                String result = get();
                resultPane.insertHTML(result);
            } catch (InterruptedException ex) {
                logger.error(ex);
            } catch (ExecutionException ex) {
                logger.error(ex);
            }
        }
    }
    
    private class ConsoleBuiltInsPanel extends JPanel {
    
    private final JButton setLevelButton = new JButton("setLevel");
    private final JButton unlockButton = new JButton("Unlock");
    private final JButton lockButton = new JButton("Lock");
    private final HintTextField levelInputTF = new HintTextField("level");
    
        ConsoleBuiltInsPanel() {
            super(new GridLayout(2,2));
            add(lockButton);
            add(unlockButton);
            add(levelInputTF);
            add(setLevelButton);
            
            setLevelButton.addActionListener((e) -> { 
                try {
                    AgentDictionaries dc = agentTable.getSelectedAgentDictionaries();
                    if (dc == null){
                        return;
                    }
                    String selectedAgentName = dc.ai.getName();
                    int level = Integer.valueOf(levelInputTF.getText());
                    agentLockService.setLevelForAgent(selectedAgentName, level);
                    levelInputTF.setHint();
                } catch (NumberFormatException ex) {
                    // ignored
                } catch(Exception ex) {
                    JOptionPane.showMessageDialog(this, ex.getMessage());
                }
            });
            
            lockButton.addActionListener((e) -> { 
                try {
                    AgentDictionaries dc = agentTable.getSelectedAgentDictionaries();
                    if (dc == null){
                        return;
                    }
                    String selectedAgentName = dc.ai.getName();
                    agentLockService.lockAgent(selectedAgentName);
                    levelInputTF.setHint();
                } catch (NumberFormatException ex) {
                    // ignored
                } catch(Exception ex) {
                    JOptionPane.showMessageDialog(this, ex.getMessage());
                }
            });
            
            unlockButton.addActionListener((e) -> { 
                try {
                    AgentDictionaries dc = agentTable.getSelectedAgentDictionaries();
                    if (dc == null){
                        return;
                    }
                    String selectedAgentName = dc.ai.getName();
                    agentLockService.unlockAgent(selectedAgentName);
                    levelInputTF.setHint();
                } catch (NumberFormatException ex) {
                    // ignored
                } catch(Exception ex) {
                    JOptionPane.showMessageDialog(this, ex.getMessage());
                }
            });

        }
    
}
    
   
}