package org.lsst.ccs.gconsole.plugins.alert;

import java.awt.event.ActionEvent;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.plugins.tracer.LsstTracerPlugin;
import org.lsst.ccs.gconsole.plugins.tracer.Tracer;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.services.alert.AlertService;

/**
 * Graphical console plugin that handles alerts received from CCS subsystems.
 *
 * @author onoprien
 */
@Plugin(name="LSST Alert Notification Plugin",
        id="alert",
        description="LSST CCS alert notification service.")
public class LsstAlertPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    static private final String OPT_AUTO_START_VIEWER = "autoStartViewer";
    static private final String OPT_AUTO_START_TRACER = "autoStartTracer";
    static final String OPT_HISTORY = "history";
    static final String OPT_MUTE = "mute";
    static final String OPT_TOFRONT = "toFront";
    static final String OPT_SELECT = "selectLast";
    
    private AlertService alertService;
    
    private Action viewerAction;
    private volatile AlertViewer viewer; // modified on EDT only

    
// -- Lifecycle : --------------------------------------------------------------

    @Override
    public void initialize() {
        
        alertService = getConsole().getAlertService();
        
        // Define plugin settings and preferences
        
        getServices().addProperty(OPT_AUTO_START_VIEWER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, "Startup", "${"+ OPT_AUTO_START_VIEWER +"} Start viewer on console sturtup.");
        getServices().addProperty(OPT_AUTO_START_TRACER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, "Startup", "${"+ OPT_AUTO_START_TRACER +"} Start tracer on console sturtup.");
        
        getServices().addProperty(OPT_HISTORY, false);
        getServices().addProperty(OPT_MUTE, false);
        getServices().addProperty(OPT_TOFRONT, true);
        getServices().addProperty(OPT_SELECT, true);
        
        // Create menus for launching alert viewer and tracer
        
        viewerAction = new AbstractAction("Viewer") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean) getValue(Action.SELECTED_KEY)) {
                    onNewAlertViewer(null);
                } else {
                    stopAlertViewer();
                }
                
            }
        };
        viewerAction.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(viewerAction, "400: CCS Tools :-1:4", "Alerts:1");
        
        Action act  = new AbstractAction("Tracer") {
            @Override
            public void actionPerformed(ActionEvent e) {
                onNewAlertTracer();
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:4", "Alerts:2");
    }

    @Override
    public void start() {
        if ((boolean) getServices().getProperty(OPT_AUTO_START_VIEWER)) {
            onNewAlertViewer(null);
        }
        if ((boolean) getServices().getProperty(OPT_AUTO_START_TRACER)) {
            onNewAlertTracer();
        }
    }
  
    @Override
    public void shutdown() {
        // At the moment, nothing meaningfull can be done on plugin shutdown, since it might be called from JVM shutdown hook.
        // That's OK - standard tools are never shut down.
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    public void onNewAlertViewer(AlertViewer.Descriptor config) {
        if (SwingUtilities.isEventDispatchThread()) {
            if (viewer == null) {
                viewerAction.putValue(Action.SELECTED_KEY, true);
                viewer = new AlertViewer(this, config);
                alertService.addListener(viewer);
            }
        } else {
            SwingUtilities.invokeLater(() -> onNewAlertViewer(config));
        }
    }
    
    public void stopAlertViewer() {
        if (SwingUtilities.isEventDispatchThread()) {
            if (viewer != null) {
                viewerAction.putValue(Action.SELECTED_KEY, false);
                alertService.removeListener(viewer);
                viewer.stop();
                viewer = null;
            }
        } else {
            SwingUtilities.invokeLater(this::stopAlertViewer);
        }
    }
    
    public void onNewAlertTracer() {
        LsstTracerPlugin tracerPlugin = (LsstTracerPlugin) getConsole().getConsoleLookup().lookup(LsstTracerPlugin.class);
        Tracer tracer = tracerPlugin.createTracer("Alerts", "BuiltIn/Alerts");
        tracer.setFilter(filteredMessage -> {
            BusMessage bm = filteredMessage.getBusMessage();
            if (bm instanceof StatusRaisedAlert) {
                StatusRaisedAlert am = (StatusRaisedAlert) bm;
                Alert alert = am.getRaisedAlert();
                String id = alert.getAlertId();
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getTimeStamp())).append(" : ");
                sb.append(am.getRaisedAlertSummary().getRaisedAlert(id).getLatestAlertState());
                sb.append(" from ").append(am.getOriginAgentInfo().getName()).append(".\n");
                sb.append("ID: ").append(id).append(".  Description:\n");
                sb.append(alert.getDescription()).append("\n");
                filteredMessage.setMessage(sb.toString());
                return filteredMessage;
            } else if (bm instanceof StatusClearedAlert) {
                StatusClearedAlert am = (StatusClearedAlert) bm;
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getTimeStamp())).append(" : ");
                sb.append(am.getOriginAgentInfo().getName()).append(" cleared alerts ");
                for (String id : am.getClearAlertIds()) {
                    sb.append(id).append(", ");
                }
                sb.delete(sb.length()-2, sb.length()).append(".\n");
                filteredMessage.setMessage(sb.toString());
                return filteredMessage;
            } else {
                return null;
            }
        });
    }
    
    public void clearAlerts(Map<String,String[]> alarms) {
        AgentMessagingLayer messenger = getConsole().getMessagingAccess();
        CommandOriginator originator = new CommandOriginator() {
            public void processAck(CommandAck ack) {}
            public void processResult(CommandResult result) {}
            public void processNack(CommandNack nack) {}
        };
        alarms.forEach((source, ids) -> {
            if (getConsole().getMessagingAccess().getAgentPresenceManager().agentExists(source)) {
                CommandRequest request;
                if (ids == null) {
                    request = new CommandRequest(source, "clearAllAlerts");
                } else {
                    request = new CommandRequest(source, "clearAlerts", (Object) ids);
                }
                messenger.sendCommandRequest(request, originator);
            } else {
                if (ids == null) {
                    alertService.clearAllAlertsForSource(source);
                } else {
                    alertService.clearAlertsForSource(source, ids);
                }
            }
        });
    }

// -- Saving/restoring : -------------------------------------------------------
    
    @Override
    public ComponentDescriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        desc.setHistory((boolean) getServices().getProperty(OPT_HISTORY));
        desc.setMute((boolean) getServices().getProperty(OPT_MUTE));
        desc.setToFront((boolean) getServices().getProperty(OPT_TOFRONT));
        desc.setSelectLast((boolean) getServices().getProperty(OPT_SELECT));
        if (viewer != null) {
            if (SwingUtilities.isEventDispatchThread()) {
                desc.setViewerConfig(viewer.save());
            } else {
                try {
                    SwingUtilities.invokeAndWait(() -> {
                        if (viewer != null) {
                            desc.setViewerConfig(viewer.save());
                        }
                    });
                } catch (InterruptedException|InvocationTargetException x) {
                }
            }
        }
        return desc;
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        
        if (!(storageBean instanceof Descriptor)) {
            throw new IllegalArgumentException("Illegal descriptor type: "+ storageBean.getClassName());
        }
        Descriptor desc = (Descriptor) storageBean;
        
        getServices().setProperty(OPT_HISTORY, desc.isHistory());
        getServices().setProperty(OPT_MUTE, desc.isMute());
        getServices().setProperty(OPT_TOFRONT, desc.isToFront());
        getServices().setProperty(OPT_SELECT, desc.isSelectLast());
        
        AlertViewer.Descriptor viewerConfig = desc.getViewerConfig();
        if (viewerConfig == null) {
            if (viewer != null) {
                stopAlertViewer();
            }
        } else {
            stopAlertViewer();
            onNewAlertViewer(viewerConfig);
        }
        return true;
    }
    
    static public class Descriptor extends ComponentDescriptor {

        private boolean selectLast;
        private boolean toFront;
        private boolean mute;
        private boolean history;
        private AlertViewer.Descriptor viewerConfig;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public boolean isSelectLast() {
            return selectLast;
        }

        public void setSelectLast(boolean selectLast) {
            this.selectLast = selectLast;
        }

        public boolean isToFront() {
            return toFront;
        }

        public void setToFront(boolean toFront) {
            this.toFront = toFront;
        }

        public boolean isMute() {
            return mute;
        }

        public void setMute(boolean mute) {
            this.mute = mute;
        }

        public boolean isHistory() {
            return history;
        }

        public void setHistory(boolean history) {
            this.history = history;
        }

        public AlertViewer.Descriptor getViewerConfig() {
            return viewerConfig;
        }

        public void setViewerConfig(AlertViewer.Descriptor viewerConfig) {
            this.viewerConfig = viewerConfig;
        }
        
    }
        
}
