package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Color;
import java.math.BigInteger;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.SwingConstants;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.python.google.common.base.Objects;

/**
 * Formatter for monitoring data. This class may be used as is or subclassed to modify formatting logic.
 * <p>
 * All convenience methods that take {@link MonitorCell} or lists of data channels
 * (as {@link AgentChannel} or {@link ChannelHandle}) eventually forward the call to
 * {@link #format(MonitorField field, List channels)}.
 * Methods that take a single channel (again, as {@link AgentChannel} or {@link ChannelHandle}),
 * eventually forward the call to {@link #format(MonitorField field, AgentChannel channel)}.
 *
 * @author onoprien
 */
public class MonitorFormat {
    
// -- Static constants : -------------------------------------------------------
    
    static public final MonitorFormat DEFAULT = new MonitorFormat();
    
    static public final Color COLOR_FG = Color.BLACK; // default foreground
    static public final Color COLOR_BG = Color.WHITE; // default background
    
    static public final Color COLOR_CHANGED = Color.BLUE;
    
    static public final Color COLOR_GOOD = new Color(160, 255, 160);
    static public final Color COLOR_WARN = new Color(255, 255, 100);
    static public final Color COLOR_ERR = new Color(255, 160, 160);
    static public final Color COLOR_OFF = new Color(160, 200, 255);
    static public final Color COLOR_POPUP = new Color(255, 255, 160);
    static public final Color COLOR_MULTI = Color.LIGHT_GRAY;
    static public final Color COLOR_NA = COLOR_BG;
    
    static public final EnumMap<DataProviderState,Color> COLOR_STATE = new EnumMap(DataProviderState.class);
    static {
        COLOR_STATE.put(DataProviderState.OFF_LINE, COLOR_OFF);
        COLOR_STATE.put(DataProviderState.DISABLED, new Color(160, 255, 200));
        COLOR_STATE.put(DataProviderState.NOT_VALIDATED, COLOR_BG);
        COLOR_STATE.put(DataProviderState.NOMINAL, COLOR_GOOD);
        COLOR_STATE.put(DataProviderState.WARNING, COLOR_WARN);
        COLOR_STATE.put(DataProviderState.ALARM, COLOR_ERR);
    }
    
    static public final String TEXT_MULTI = "---";
    static public final int HA_MULTI = SwingConstants.CENTER;
    
// -- Fields : -----------------------------------------------------------------
    
    protected String DEFAULT_FLOAT_FORMAT = "%14.5f";
    protected String DEFAULT_INT_FORMAT = "%10d";
    protected String DEFAULT_DT_FORMAT = "%1$tF %1$tT";
    protected int LENGTH_LIMIT = 40;
    
    
// -- Formatting : -------------------------------------------------------------
    
    /**
     * Produces formatted value, given the field and the list of contributing channels. <br>
     * The implementation provided by this class calls {@link #format(MonitorField field, AgentChannel channel)}
     * for each channel, then merges the results using {@link #merge(MonitorField field, List formattedValues)}.
     * 
     * @param field Monitored field.
     * @param channels Contributing channels.
     * @return Formatted value.
     */
    public FormattedValue format(MonitorField field, List<AgentChannel> channels) {
        
        // Ask the field format itself based on the list of channels:
        
        FormattedValue fv = field.format(channels);
        
        // If the field cannot, ask it to format each channel separately, then merge:
        
        if (fv == null) {
            List<FormattedValue> fvList = channels.stream().map(ch -> format(field, ch)).collect(Collectors.toList());
            fv = merge(field, fvList);
        }
        
        // If the formatted value is incomplete, finish formatting:
        
        if (!fv.isValid()) {
            fv = format(fv);
        }
        
        // If the formatted value is still incomplete, return indication that the data is currently unavailable:
        
        if (!fv.isValid()) {
            fv = FormattedValue.NA;
        }
        
        return fv;
    }
    
    /**
     * Produces formatted value, given the monitored field and the data channel whose value needs to be formatted. <br>
     * The implementation provided by this class first tries to call {@link MonitorField.format(AgentChannel channel)}.
     * If {@code null} is returned, it extracts the unformatted value with a call to 
     * {@link getValue(MonitorField field, AgentChannel channel)}, then formats it with
     * {@link format(Object unformattedValue, MonitorField field, AgentChannel channel)}.
     * 
     * @param field Monitored field.
     * @param channel Data channel.
     * @return Formatted value.
     */
    public FormattedValue format(MonitorField field, AgentChannel channel) {
        
        // Ask the field format itself:
        
        FormattedValue fv = field.format(channel);
        if (fv.isValid()) return fv;
        
        // If the formatted value is incomplete, finish formatting:
        
        fv = format(fv);
        if (fv.isValid()) return fv;
        
        // Otherwise, return indication that the data is currently unavailable
        
        return FormattedValue.NA;
    }

    /**
     * Completes formatting of a partially formatted value.
     * The instance passed to this method should have its {@code value} and optionally {@code format}
     * fields set to the unformatted value and the suggested format string, respectively.
     * 
     * @param fv Partially formatted value.
     * @return Formatted value.
     */
    public FormattedValue format(FormattedValue fv) {
        
        if (fv.isValid()) return fv;
        
        // If configurable parameter, color foreground:
        
        if (fv.value instanceof ConfigurationParameterInfo) {
            ConfigurationParameterInfo conf = (ConfigurationParameterInfo) fv.value;
            fv.value = conf.getCurrentValue();
            if (fv.fgColor == null) fv.fgColor = conf.isDirty() ? COLOR_CHANGED : COLOR_FG;
            if (fv.editable == null) fv.editable = !conf.isFinal();
            if (fv.toolTip == null) fv.toolTip = conf.getDescription();
            return format(fv);
        }
        
        // Format raw value:
        
        if (fv.text == null) {
            
            // if format is known, try to use it
            
            if (fv.format != null) {
                
//                if (fv.format.contains("%f")) {              // FIXME
//                    fv.format = DEFAULT_FLOAT_FORMAT;        // FIXME
//                }                                            // FIXME
                
                char conv = fv.format.charAt(fv.format.length() - 1);
                try {
                    switch (conv) {
                        case 'd': case 'o': case 'x': case 'X':  // integer
                            if (!(fv.value instanceof Integer || fv.value instanceof Long || fv.value instanceof Short || fv.value instanceof Byte || fv.value instanceof BigInteger)) {
                                fv.value = Long.parseLong(fv.value.toString());
                            }
                            break;
                        case 'f': case 'e': case 'E': case 'g': case 'G':
                            if (!(fv.value instanceof Double || fv.value instanceof Float || fv.value instanceof BigInteger)) {
                                fv.value = Double.parseDouble(fv.value.toString());
                            }
                            break;
                    }
                    fv.text = String.format(fv.format, fv.value);
                } catch (NumberFormatException | IllegalFormatException x) {
                }
            }
            
            // otherwise, if a number, try to use default format
            
            if (fv.text == null) {
                if (fv.value instanceof Double || fv.value instanceof Float) {
                    try {
                        fv.text = String.format(DEFAULT_FLOAT_FORMAT, fv.value);
                    } catch (IllegalFormatException x) {
                    }
                } else if (fv.value instanceof Integer) {
                    try {
                        fv.text = String.format(DEFAULT_INT_FORMAT, fv.value);
                    } catch (IllegalFormatException x) {
                    }
                } else if (fv.value instanceof Instant) {
                    try {
                        fv.text = String.format(DEFAULT_DT_FORMAT, ZonedDateTime.ofInstant((Instant)fv.value, ZoneId.systemDefault()));
                    } catch (IllegalFormatException | DateTimeException x) {
                    }
                } else if (fv.value instanceof ZonedDateTime) {
                    try {
                        fv.text = String.format(DEFAULT_DT_FORMAT, fv.value);
                    } catch (IllegalFormatException x) {
                    }
                }
            }

            // otherwise, convert to String:
            
            if (fv.text == null) {
                fv.text = fv.value.toString();
            }
            
            // limit text length:
            
            if (fv.text.length() > LENGTH_LIMIT) {
                if (fv.toolTip == null) {
                    fv.toolTip = fv.text;
                } else {
                    fv.toolTip = "<html><b>"+ fv.text +"</b><br>"+ fv.toolTip;
                }
            }

        }
        
        fv.value = null;        
        return fv;
    }
    
    
// -- Convenience methods: -----------------------------------------------------
    
    /**
     * Convenience method that updates {@link FormattedValue} associated with the specified cell. <br>
     * The actual formatting is done by {@link #format(MonitorField field, List channels)}.
     * 
     * @param cell Cell to be formatted.
     * @return True if the formatted value associated with the cell has changed as a result of this call.
     */
    public boolean format(MonitorCell cell) {
        List<AgentChannel> channels = cell.getChannels().stream().map(h -> h.getChannel()).collect(Collectors.toList());
        FormattedValue fv = format(cell.getField(), channels);
        if (Objects.equal(fv, cell.getFormattedValue())) {
            return false;
        } else {
            cell.setFormattedValue(fv);
            return true;
        }
    }
    
    /**
     * Convenience method that updates {@link FormattedValue} associated with the specified cell,
     * assuming that the only possible reason for a change are changes in the specified channel. <br>
     * The implementation provided by this class forwards the call to {@link #format(MonitorCell cell)}, 
     * ignoring the {@code channelHandle} argument. Subclasses might implement more efficient formatting
     * for cells that depend on multiple channels.
     * 
     * @param cell Cell to be formatted.
     * @param channelHandle Handle of the data channel that might have changed.
     * @return  True if the formatted value associated with the cell has changed as a result of this call.
     */
    public boolean format(MonitorCell cell, ChannelHandle channelHandle) {
        return format(cell);
    }
    
    /**
     * Convenience method that produces formatted value, given the field and the channel handle. <br>
     * The actual formatting is done by {@link #format(MonitorField field, AgentChannel channel)}.
     * 
     * @param field Monitored field.
     * @param channelHandle Handle of a data channel whose value needs to be formatted.
     * @return Formatted value.
     */
    public FormattedValue format(MonitorField field, ChannelHandle channelHandle) {
        return format(field, channelHandle.getChannel());
    }
    
    
// -- Utility methods : --------------------------------------------------------
    
    /**
     * Merges zero or more formatted values.
     * 
     * @param field Monitored field.
     * @param values Formatted values to merge.
     * @return Merged formatted value.
     */
    public FormattedValue merge(MonitorField field, List<FormattedValue> values) {
        switch (values.size()) {
            case 0:
                return new FormattedValue(field.getTitle(), COLOR_FG, COLOR_BG, SwingConstants.CENTER, null, false);
            case 1:
                return values.get(0);
            default:
                Iterator<FormattedValue> it = values.iterator();
                FormattedValue first = it.next();
                String text = first.getText();
                Color fgColor = first.getFgColor();
                Color bgColor = first.getBgColor();
                int horizontalAlignment = first.getHorizontalAlignment();
                String toolTip = first.getToolTip();
                while (it.hasNext()) {
                    FormattedValue other = it.next();
                    if (!Objects.equal(text, other.getText())) {
                        text = TEXT_MULTI;
                    }
                    fgColor = mergeConfigColor(fgColor, other.getFgColor());
                    bgColor = mergeStateColor(bgColor, other.getBgColor());
                    if (horizontalAlignment != other.getHorizontalAlignment()) {
                        horizontalAlignment = HA_MULTI;
                    }
                    if (!Objects.equal(toolTip, other.getToolTip())) {
                        toolTip = null;
                    }
                }
                return new FormattedValue(text, fgColor, bgColor, horizontalAlignment, toolTip, false);
        }
    }
    
    /**
     * Merges colors used to indicate a state of a configurable parameter. <br>
     * Used internally by {@link #merge(MonitorField field, List formattedValues)}.
     * 
     * @param c1 First color.
     * @param c2 Second color.
     * @return Color of the merged value.
     */
    public Color mergeConfigColor(Color c1, Color c2) {
        return COLOR_FG.equals(c1) ? c2 : c1; 
    }
    
    /**
     * Merges colors used to indicate the channel state. <br>
     * Used internally by {@link #merge(MonitorField field, List formattedValues)}.
     * 
     * @param c1 First color.
     * @param c2 Second color.
     * @return Color of the merged value.
     */
    public Color mergeStateColor(Color c1, Color c2) {
        for (Color c : COLOR_STATE.values()) {
            if (c.equals(c1)) {
                return c2;
            } else if (c.equals(c2)) {
                return c1;
            }
        }
        return c1;
    }
    
    /**
     * Extracts unformatted value corresponding to the specified field from a channel. <br>
     * The implementation provided by this class forwards the call to {@code fielf.getValue(AgentChannel channel)}.
     * 
     * @param field Monitored field.
     * @param channel Data channel.
     * @return Unformatted value,or {@code null} if the value cannot be extracted.
     */
    public Object getValue(MonitorField field, AgentChannel channel) {
        return field.getValue(channel);
    }
    
    
// -- Test : -------------------------------------------------------------------
    
    static public void main(String... arg) {
        
//        System.out.println(String.format("%s", new Double("2.7723786437894628")));
        System.out.println(String.format("%f", 2.7723786437894628));
        System.out.println(String.format("%1$tF %1$tT", ZonedDateTime.now()));
        System.out.println(String.format("%10.4f", 2.7723786437894628));
        System.out.println(" ");
        
        
    }
    
}
