package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.List;
import org.lsst.ccs.gconsole.agent.AgentChannel;

/**
 * Data channel handle used by monitoring data views.
 * The handle provides access to the underlying channel, its display path, and the 
 * {@link Updatable} target component that displays it and should be notified of changes.
 * 
 * @author onoprien
 */
public interface ChannelHandle {

    /**
     * Returns data channel object wrapped by this handle.
     * @return Data channel.
     */
    AgentChannel getChannel();

    /**
     * Returns the display path associated with the underlying data channel in the monitoring view.
     * @return Display path.
     */
    String getPath();

    /**
     * Returns the object that handles an updatable display of the data channel.
     * @return Target display for the underlying channel.
     */
    Updatable getTarget();

    /**
     * Sets data channel object wrapped by this handle.
     * @param channel 
     */
    void setChannel(AgentChannel channel);

    /**
     * Sets the display path associated with the underlying data channel.
     * @param path Display path.
     */
    void setPath(String path);

    /**
     * Set the object that handles an updatable display of the data channel.
     * @param target Target display for the underlying channel.
     */
    void setTarget(Updatable target);
    
    /**
     * Updates the target.
     * The default implementation converts the list of attributes into the list of monitored fields using the target's
     * {@code getAffectedFields(...)} method, then calls the appropriate {@code update(...)} method of the target.
     * Most classes implementing this interface will not override this method.
     * 
     * @param attributes List of attributes whose values might have changed, or {@code null} if all attribute values might have changed.
     */
    default void update(List<String> attributes) {
        Updatable target = getTarget();
        if (target != null) {
            if (attributes == null) {
                target.update(this);
            } else {
                List<MonitorField> fields = target.getAffectedFields(attributes);
                if (fields == null) {
                    target.update(this);
                } else if (!fields.isEmpty()) {
                    target.update(this, fields);
                }
            }
        }
    }

}
