package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import java.util.stream.Collectors;
import javax.swing.SwingConstants;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.base.Console;
import static org.lsst.ccs.gconsole.plugins.monitor.MonitorFormat.COLOR_STATE;

/**
 * Descriptor of a field that can be displayed by monitoring views.
 * Static constants of this class enumerate standard types of monitored values.
 * Additional instances may be created at run time.
 * <p>
 * Instances can be registered as default handlers for data channel attribute keys.
 * <p>
 * Equality and hash code are based on the object identity.
 *
 * @author onoprien
 */
public class MonitorField {
    
    static private final HashMap<String,MonitorField> fields = new HashMap<>();
    
// -- Pre-defined standard fields : --------------------------------------------
    
    static public final MonitorField NULL = new MonitorField(null, "", null);
    
    static public final MonitorField NAME = new MonitorField("name", "Name", null);
    static {registerInstance(NAME);}
    
    static public final MonitorField VALUE = new MonitorField(AgentChannel.Key.VALUE, 
            "Value",
            new FormattedValue(null, null, null, SwingConstants.RIGHT, null, null),
            Collections.unmodifiableSet(new HashSet(Arrays.asList(new String[] {AgentChannel.Key.VALUE, AgentChannel.Key.STATE}))));
    static {registerInstance(VALUE);}
    
    static public final MonitorField UNITS = new MonitorField(AgentChannel.Key.UNITS, "Units", new FormattedValue(null, null, null, SwingConstants.LEFT, null, false));
    static {registerInstance(UNITS);}
    
    static public final MonitorField LOW_ALARM = new Limit(AgentChannel.Key.LOW_ALARM, "Low Limit");
    static {registerInstance(LOW_ALARM);}
    
    static public final MonitorField LOW_WARN = new Limit(AgentChannel.Key.LOW_WARN, "Low Warn") {
        private final Set<String> keys = Collections.unmodifiableSet(new HashSet(Arrays.asList(new String[] {AgentChannel.Key.LOW_WARN, AgentChannel.Key.LOW_ALARM})));
        @Override
        public Set<String> getKeys() {
            return keys;
        }
        @Override
        public FormattedValue format(AgentChannel channel) {
            try {
                FormattedValue fv = super.format(channel);
                Object o = fv.value;
                if (o instanceof ConfigurationParameterInfo) {
                    o = ((ConfigurationParameterInfo)o).getCurrentValue();
                } 
                if (!(o instanceof Double)) {
                    o = Double.valueOf(o.toString());
                }
                double d = (Double)o;
                o = channel.get(AgentChannel.Key.LOW_ALARM);
                if (o instanceof ConfigurationParameterInfo) {
                    o = ((ConfigurationParameterInfo)o).getCurrentValue();
                } 
                if (!(o instanceof Double)) {
                    o = Double.valueOf(o.toString());
                }
                d = (Double)o + d;
                fv.text = String.format(fv.format, d);
                return fv;
            } catch (NullPointerException | NumberFormatException | IllegalFormatException x) {
                return FormattedValue.NA;
            }
        }
        @Override
        public void setValue(Object value, AgentChannel channel) {
            try {
                if (!(value instanceof Double)) {
                    value = Double.valueOf(value.toString());
                }
                double d = (Double)value;
                Object o = channel.get(AgentChannel.Key.LOW_ALARM);
                if (o instanceof ConfigurationParameterInfo) {
                    o = ((ConfigurationParameterInfo)o).getCurrentValue();
                } 
                if (!(o instanceof Double)) {
                    o = Double.valueOf(o.toString());
                }
                d = d - (Double)o;
                value = String.valueOf(d);
                ConfigurationParameterInfo conf = (ConfigurationParameterInfo) getValue(channel);
                Console.getConsole().sendCommand(channel.getAgentName() + "/change", conf.getComponentName(), conf.getParameterName(), value);
            } catch (ClassCastException | NullPointerException x) {
            }
        }
    };
    static {registerInstance(LOW_WARN);}
    
    static public final MonitorField ALERT_LOW = new Alert(AgentChannel.Key.ALERT_LOW);
    static {registerInstance(ALERT_LOW);}

    static public final MonitorField HIGH_ALARM = new Limit(AgentChannel.Key.HIGH_ALARM, "High Limit");
    static {registerInstance(HIGH_ALARM);}
    
    static public final MonitorField HIGH_WARN = new Limit(AgentChannel.Key.HIGH_WARN, "High Warn") {
        private final Set<String> keys = Collections.unmodifiableSet(new HashSet(Arrays.asList(new String[] {AgentChannel.Key.HIGH_WARN, AgentChannel.Key.HIGH_ALARM})));
        @Override
        public Set<String> getKeys() {
            return keys;
        }
        @Override
        public FormattedValue format(AgentChannel channel) {
            try {
                FormattedValue fv = super.format(channel);
                Object o = fv.value;
                if (o instanceof ConfigurationParameterInfo) {
                    o = ((ConfigurationParameterInfo)o).getCurrentValue();
                } 
                if (!(o instanceof Double)) {
                    o = Double.valueOf(o.toString());
                }
                double d = (Double)o;
                o = channel.get(AgentChannel.Key.HIGH_ALARM);
                if (o instanceof ConfigurationParameterInfo) {
                    o = ((ConfigurationParameterInfo)o).getCurrentValue();
                } 
                if (!(o instanceof Double)) {
                    o = Double.valueOf(o.toString());
                }
                d = (Double)o - d;
                fv.text = String.format(fv.format, d);
                return fv;
            } catch (NullPointerException | NumberFormatException | IllegalFormatException x) {
                return FormattedValue.NA;
            }
        }
        @Override
        public void setValue(Object value, AgentChannel channel) {
            try {
                if (!(value instanceof Double)) {
                    value = Double.valueOf(value.toString());
                }
                double d = (Double)value;
                Object o = channel.get(AgentChannel.Key.HIGH_ALARM);
                if (o instanceof ConfigurationParameterInfo) {
                    o = ((ConfigurationParameterInfo)o).getCurrentValue();
                } 
                if (!(o instanceof Double)) {
                    o = Double.valueOf(o.toString());
                }
                d = d - (Double)o;
                value = String.valueOf(d);
                ConfigurationParameterInfo conf = (ConfigurationParameterInfo) getValue(channel);
                Console.getConsole().sendCommand(channel.getAgentName() + "/change", conf.getComponentName(), conf.getParameterName(), value);
            } catch (ClassCastException | NullPointerException x) {
            }
        }
    };
    static {registerInstance(HIGH_WARN);}
    
    static public final MonitorField ALERT_HIGH = new Alert(AgentChannel.Key.ALERT_HIGH);
    static {registerInstance(ALERT_HIGH);}
    
    static public final MonitorField DESCR = new MonitorField("description", "Description", new FormattedValue(null, null, null, SwingConstants.LEFT, null, false));
    static {registerInstance(DESCR);}


// -- Fields : -----------------------------------------------------------------
    
    protected final String key;
    protected final String title;
    protected final FormattedValue template;
    protected final Set<String> keys;

// -- Life cycle and registration : --------------------------------------------
    
    /**
     * Constructs an instance.
     * 
     * @param key Attribute key. Unless overridden, {@code getKey()} and {@code getKeys()}
     *            methods will return this key and a singleton set containing this key, respectively.
     *            The {@code getValue(channel)} method will return the value of the data channel attribute
     *            identified by this key.
     * @param title Human readable title.
     * @param template Formatted value template.
     */
    public MonitorField(String key, String title, FormattedValue template) {
        this(key, title, template, null);
    }
    
    /**
     * Constructs an instance.
     * 
     * @param key Attribute key. Unless overridden, {@code getKey()} will return this key. The {@code getValue(channel)}
     *            method will return the value of the data channel attribute identified by this key.
     * @param title Human readable title.
     * @param template Formatted value template.
     * @param keys Keys of attributes that might affect this field. If {@code null}, the field is assumed to be affected
     *             by the attribute specified by the {@code key} argument. If that argument is {@code null} as well,
     *             the field is not affected by any parameters.
     */
    public MonitorField(String key, String title, FormattedValue template, Set<String> keys) {
        this.key = key;
        this.title = title;
        this.template = template == null ? new FormattedValue() : new FormattedValue(template);
        this.keys = keys == null? (key == null ? Collections.emptySet() : Collections.singleton(key)) : keys;
    }
    
    /**
     * Deep copy constructor.
     * @param other Field to be cloned.
     */
    public MonitorField(MonitorField other) {
        this.key = other.key;
        this.title = other.title;
        this.template = other.template;
        this.keys = other.keys;
    }
    
    /**
     * Registers the specified instance as the default handler for the attribute
     * identified by the key returned by its {@code getKey()} method.
     * 
     * @param field Instance to register.
     * @throws IllegalArgumentException if an instance with this {@code key} already registered.
     */
    static public synchronized void registerInstance(MonitorField field) {
        String key = field.getKey();
        if (fields.containsKey(key)) throw new IllegalArgumentException("Field with key "+ key +" is already registered.");
        fields.put(key, field);
    }
    
    /**
     * Returns an instance registered as the default handler for the specified attribute key.
     * If a default handler for the specified key is already registered, it is returned;
     * otherwise, if a standard field with the specified name (static field in this class) exists, it is returned;
     * otherwise, a new instance is created with default parameters:<br>
     * {@code new MonitorField(key, key, null)}.
     * 
     * @param key Attribute key.
     * @return Registered instance for the specified key.
     */
    static public synchronized MonitorField getInstance(String key) {
        MonitorField field = valueOf(key);
        if (field == null) {
            try {
                field =  (MonitorField) MonitorField.class.getField(key).get(null);
            } catch (NoSuchFieldException | SecurityException | IllegalArgumentException | IllegalAccessException | NullPointerException x) {
            }
        }
        if (field == null) {
            field = new MonitorField(key, key, null);
            registerInstance(field);
        }
        return field;
    }
    
    
// -- Statis utilities : -------------------------------------------------------
    
    /**
     * Returns an instance registered as the default handler for the attribute identified
     * by the specified key, or {@code null} if no such instance exists.
     * 
     * @param key Attribute key.
     * @return Registered instance, or {@code null} if no such instance exists.
     */
    static public synchronized MonitorField valueOf(String key) {
        return fields.get(key);
    }
    
    static public synchronized List<MonitorField> getDefaultFields(Collection<String> keys) {
        return keys.stream().map(k -> valueOf(k)).filter(f -> f != null).collect(Collectors.toList());
    }
    
    static public List<MonitorField> getAffectedFields(Collection<String> keys, Collection<MonitorField> fields) {
        if (keys == null) return null;
        return fields.stream().filter(field -> {
            Set<String> att = field.getKeys();
            return keys.stream().anyMatch(k -> att.contains(k));
        }).collect(Collectors.toList());
    }
    
    static public List<MonitorField> getAffectedFields(Collection<String> keys) {
        if (keys == null) return null;
        synchronized (MonitorField.class) {
            return getAffectedFields(keys, fields.values());
        }
    }
    
    
// -- Retrieving current value : -----------------------------------------------
    
    /**
     * Returns the current value of this field in the specified data channel, or {@code null}
     * if the channel does not contain a value for this field.
     * 
     * @param channel Data channel.
     * @return Current value of this field.
     */
    public Object getValue(AgentChannel channel) {
        return (channel == null || key == null) ? null : channel.get(key);
    }
    
    public void setValue(Object value, AgentChannel channel) {
        try {
            ConfigurationParameterInfo conf = (ConfigurationParameterInfo) getValue(channel);
            Console.getConsole().sendCommand(channel.getAgentName() +"/change", conf.getComponentName(), conf.getParameterName(), value);
        } catch (ClassCastException | NullPointerException x) {
        }
    }

    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns a unique getKey of this field.
     * @return Name of this field.
     */
    public String getKey() {
        return key;
    }
    
    /**
     * Returns human readable title associated with this field.
     * @return Title of this field.
     */
    public String getTitle() {
        return title;
    }
    
    /**
     * Returns a set of keys of data channel attributes that might effect the value of this field or its formatting.
     * For fields that are not updated throughout lifetime of a subsystem, this method returns an empty set.
     * 
     * @return Set of attribute keys that might affect this field.
     */
    public Set<String> getKeys() {
        return keys;
    }
    
    public boolean isUpdatable() {
        return !getKeys().isEmpty();
    }
    
    
// -- Formatting : -------------------------------------------------------------

    /**
     * Returns the current formatted value of this field in the specified data channel,
     * or {@code null} if this field does not define custom formatting. In the latter case,
     * clients are expected to format the raw value returned by the {@link #getValue(AgentChannel)}
     * method, possibly taking into account values returned by other getters.
     * 
     * @param channel Data channel.
     * @return Fully or partially formatted value of this field.
     */
    public FormattedValue format(AgentChannel channel) {
        if (channel == null) {
            return FormattedValue.NA;
        } else {
            FormattedValue fv = new FormattedValue(template);
            if (fv.bgColor == null && keys.contains(AgentChannel.Key.STATE)) {
                DataProviderState state = channel.get(AgentChannel.Key.STATE);
                if (state != null) fv.bgColor = COLOR_STATE.get(state);                
            }
            if (fv.format == null) {
                fv.format = channel.get(AgentChannel.Key.FORMAT);
            }
            fv.value = getValue(channel);
            return fv;
        }
    }

    /**
     * Returns the current formatted value of this field for the specified list of channels,
     * or {@code null} if this field does not define formatting for multiple channels.
     * <p>
     * The default implementation returns {@code null}.
     * 
     * @param channels List of channels.
     * @return Fully or partially formatted value.
     */
    public FormattedValue format(List<AgentChannel> channels) {
        return null;
    }

// -- Overriding Object : ------------------------------------------------------

    /**
     * Returns the human readable title of this field.
     * @return Title of this field.
     */
    @Override
    public String toString() {
        return title;
    }
    
    
// -- Specialized subclusses : -------------------------------------------------
    
    static private class Alert extends MonitorField {
        
        Alert(String key) {
            super(key, "Al.", null);
        }
        
        @Override
        public FormattedValue format(AgentChannel channel) {
            String text;
            String alertName;
            Object value = channel.get(key);
            if (value instanceof String && !((String)value).trim().isEmpty()) {
                alertName = (String) value;
                text = "  \u2713  ";
            } else {
                alertName = "No alert";
                text = " ";
            }
            return new FormattedValue(text, MonitorFormat.COLOR_FG, MonitorFormat.COLOR_BG, SwingConstants.CENTER, alertName, false);
        }
        
    }
    
    static private class Limit extends MonitorField {
        
        Limit(String key, String title) {
            super(key, title, null);
        }

        @Override
        public FormattedValue format(AgentChannel channel) {
            if (channel == null) {
                return FormattedValue.NA;
            } else {
                FormattedValue fv = new FormattedValue();
                fv.value = channel.get(key);
                fv.horizontalAlignment = SwingConstants.RIGHT;
                Object f = channel.get(AgentChannel.Key.FORMAT);
                if (f instanceof String) {
                    fv.format = (String)f;
                } else {
                    fv.format = "%10.4f";
                }
                return fv;
            }
        }
        
    }
    
}
