package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Color;
import java.util.*;
import javax.swing.SwingConstants;

/**
 * Formatted content of a graphic component that displays a single value.
 * The value may correspond to any {@link MonitorField}.
 *
 * @author onoprien
 */
public class FormattedValue {

// -- Fields : -----------------------------------------------------------------

    /** Indicates that the data is currently unavailable. */
    static public final FormattedValue NA = new FormattedValue("", MonitorFormat.COLOR_FG, MonitorFormat.COLOR_NA, SwingConstants.CENTER, "Unknown", false);
    
    /** Indicates a permanently empty cell. */
    static public final FormattedValue EMPTY = new FormattedValue("", MonitorFormat.COLOR_FG, MonitorFormat.COLOR_BG, SwingConstants.CENTER, null, false);
    
    public String text;
    public Color fgColor;
    public Color bgColor;
    public Integer horizontalAlignment;
    public String toolTip;
    public Boolean editable;
    
    public Object value;
    public String format;


// -- Life cycle : -------------------------------------------------------------

    /**
     * Default constructor.
     * All fields are null.
     */
    public FormattedValue() {
    }

    /**
     * Full constructor.
     * All fields are set to specified values.
     * 
     * @param text Text to be displayed.
     * @param fgColor Text color.
     * @param bgColor Background color.
     * @param horizontalAlignment Horizontal alignment, as defined in {@link SwingConstants}.
     * @param toolTip Tool tip.
     * @param isEditable True if the cell should be editable by the console user.
     */
    public FormattedValue(String text, Color fgColor, Color bgColor, Integer horizontalAlignment, String toolTip, Boolean isEditable) {
        this.text = text;
        this.fgColor = fgColor;
        this.bgColor = bgColor;
        this.horizontalAlignment = horizontalAlignment;
        this.toolTip = toolTip;
        this.editable = isEditable;
    }
    
    /**
     * Copy constructor.
     * @param other Instance to be copied.
     */
    public FormattedValue(FormattedValue other) {
        this.text = other.text;
        this.fgColor = other.fgColor;
        this.bgColor = other.bgColor;
        this.horizontalAlignment = other.horizontalAlignment;
        this.toolTip = other.toolTip;
        this.editable = other.editable;
    }
    
    /**
     * Constructs an instance with specified text and alignment.
     * All other fields are left at default values.
     * 
     * @param text Text to be displayed.
     * @param horizontalAlignment Horizontal alignment, as defined in {@link SwingConstants}.
     */
    public FormattedValue(String text, Integer horizontalAlignment) {
        this(text, null, null, horizontalAlignment, null, null);
    }
    
    
// -- Getters : ----------------------------------------------------------------

    public String getText() {
        return text == null ? "" : text;
    }

    public Color getFgColor() {
        return fgColor == null ? MonitorFormat.COLOR_FG : fgColor;
    }

    public Color getBgColor() {
        return bgColor == null ? MonitorFormat.COLOR_BG : bgColor;
    }

    public int getHorizontalAlignment() {
        return horizontalAlignment == null ? SwingConstants.CENTER : horizontalAlignment;
    }

    public String getToolTip() {
        return toolTip;
    }
    
    public boolean isEditable() {
        return editable == null ? false : editable;
    }
    

// -- Overriding Object : ------------------------------------------------------

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof FormattedValue)) return false;
        FormattedValue other = (FormattedValue) obj;
        return Objects.equals(text, other.text) && 
                Objects.equals(fgColor, other.fgColor) && 
                Objects.equals(bgColor, other.bgColor) && 
                Objects.equals(horizontalAlignment, other.horizontalAlignment) &&
                Objects.equals(toolTip, other.toolTip) &&
                Objects.equals(editable, other.editable);
    }

    @Override
    public int hashCode() {
        final int PRIME = 31;
        int result = 1;
        result = result * PRIME + (text == null ? 0 : text.hashCode());
        result = result * PRIME + (fgColor == null ? 0 : fgColor.hashCode());
        result = result * PRIME + (bgColor == null ? 0 : bgColor.hashCode());
        result = result * PRIME + (horizontalAlignment == null ? 0 : horizontalAlignment.hashCode());
        return result;
    }

    @Override
    public String toString() {
        return text;
    }

}
