package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Color;
import java.awt.Dimension;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;

/**
 * Monitoring data table to be displayed as a part of an {@link ImageView}.
 * <p>
 * The implementation provided by this class is optimized for handling small tables
 * (typically 1-4 cells) displayed on top of graphical representation of equipment
 * by {@link ImageView}. If this class is ever used for large tables, it will have to
 * be subclassed and at least the {@code update(...)} method will have to be overridden.
 *
 * @author onoprien
 */
public class CellTableView extends JTable implements MonitorView, AgentChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final int nRows, nColumns;
    private final ArrayList<Cell> cells;
    
    private final Dimension maxSize = new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);

// -- Life cycle : -------------------------------------------------------------
    
    public CellTableView(int rows, int columns, Cell... content) {
        if (rows*columns != content.length) throw new IllegalArgumentException("Incorrect number of cell descriptors");
        nRows = rows;
        nColumns = columns;
        cells = new ArrayList<>(rows*columns);
        for (Cell cell : content) {
            cells.add(cell);
        }
        setOpaque(true);
        setBorder(BorderFactory.createLineBorder(Color.BLACK));
        setShowGrid(true);
        setRowSelectionAllowed(false);
        setColumnSelectionAllowed(false);
        MonitorTableCellRenderer renderer = new MonitorTableCellRenderer();
        renderer.setEnsureWidth(true);
        renderer.setEnsureHeight(true);
        setDefaultRenderer(Object.class, renderer);
        setModel(new Model());
    }

// -- Implementing MonitorView : -----------------------------------------------
    
    @Override
    public String getName() {
        return "";
    }
    
    @Override
    public void setName(String name) {
    }

    @Override
    public JComponent getPanel() {
        return this;
    }

    @Override
    public AgentChannelsFilter getFilter() {
        return this;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
    }

    @Override
    public Descriptor save() {
        return MonitorView.super.save();
    }

    @Override
    public void restore(Descriptor descriptor) {
        MonitorView.super.restore(descriptor);
    }
    
    
// -- Implementing AgentStatusListener : ---------------------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            update(event);
        });
    }

    @Override
    public void configure(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            update(event);
        });
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            update(event);
        });
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            update(event);
        });
    }
    
    /**
     * Updates the table.
     * The implementation is efficient for small tables used on image views.
     */
    private void update(AgentStatusEvent event) {
        for (AgentChannel channel : event.getAddedChannels()) {
            String path = channel.getPath();
            for (Cell cell : cells) {
                if (cell.path.equals(path)) {
                    cell.channel = channel;
                    format(cell);
                    break;
                }
            }
        }
        for (AgentChannel channel : event.getRemovedChannels()) {
            String path = channel.getPath();
            for (Cell cell : cells) {
                if (cell.path.equals(path)) {
                    cell.channel = null;
                    format(cell);
                    break;
                }
            }
        }
        for (AgentChannel channel : event.getStatusChanges().keySet()) {
            String path = channel.getPath();
            for (Cell cell : cells) {
                if (cell.path.equals(path)) {
                    cell.channel = channel;
                    format(cell);
                    break;
                }
            }
        }
        ((Model)getModel()).fireTableDataChanged();
    }
    
    private void format(Cell cell) {
        if (cell.path == null) return;
        FormattedValue fv = MonitorFormat.DEFAULT.format(MonitorField.VALUE, cell.channel);
        if (cell.format != null) {
            fv.text = String.format(cell.format, fv.text);
        }
        cell.formattedValue = fv;
    }
    
    
// -- Implementing Filter : ----------------------------------------------------
    
    @Override
    public List<String> getOriginChannels() {
        return cells.stream().map(cell -> cell.path).collect(Collectors.toList());
    }

    @Override
    public List<String> getDisplayChannels() {
        return getOriginChannels();
    }

    @Override
    public String getOriginPath(String displayPath) {
        return displayPath;
    }

    @Override
    public List<String> getDisplayPath(String originPath) {
        return Collections.singletonList(originPath);
    }

    @Override
    public List<String> getFields(boolean compact) {
        return Collections.singletonList(AgentChannel.Key.VALUE);
    }
    
    
// -- Overriding JTable : ------------------------------------------------------

    @Override
    public Dimension getMinimumSize() {
        return super.getPreferredSize();
    }

    @Override
    public Dimension getMaximumSize() {
        maxSize.height = getPreferredSize().height;
        return maxSize;
    }
    
    
// -- Cell descriptor class : --------------------------------------------------
    
    public static class Cell {
        
        private final String path;
        private final String format;
        
        private AgentChannel channel;
        private FormattedValue formattedValue;
        
        public Cell(String path) {
            this(path, null);
        }
        
        public Cell(String path, String format) {
            this.path = path;
            this.format = format;
        }
        
        public Cell(FormattedValue formattedValue) {
            this.path = null;
            this.format = null;
            this.formattedValue = formattedValue;
        }
        
    }
    
    
// -- Auxiliary classes : ------------------------------------------------------
    
    private class Model extends AbstractTableModel {

        @Override
        public int getRowCount() {
            return nRows;
        }

        @Override
        public int getColumnCount() {
            return nColumns;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            return cells.get(rowIndex * nColumns + columnIndex).formattedValue;
        }
        
    } 
    
    
// -- Testing : ----------------------------------------------------------------
    
    static public void main(String... args) {
        System.out.println(String.format("xxx%2sxxx%nyyy", "11"));
    }
    
}
