package org.lsst.ccs.gconsole.services.optpage;

import java.util.*;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.ConsoleService;
import org.lsst.ccs.gconsole.base.OptionalPage;

/**
 * Service plugin that provides support for optional pages utility.
 *
 * @author onoprien
 */
@Plugin(name = "Optional Page Service Plugin",
        id="opt-page",
        description = "Support for optional pages utility.")
public class LsstOptionalPageService extends ConsolePlugin implements ConsoleService {

// -- Fields : -----------------------------------------------------------------
    
    private final ArrayList<OptionalPageHandler> optionalPages = new ArrayList<>(0);

// -- Life cycle : -------------------------------------------------------------
    
// -- Registration : -----------------------------------------------------------
    
    /**
     * Registers an optional page descriptor with the console.
     * Once registered, the descriptor will trigger creation of console pages whenever
     * compatible subsystems are discovered on the buses. 
     * See {@link OptionalPage} documentation for details.
     * The demo subsystem GUI provides an example of use: {@code org.lsst.ccs.subsystem.demo.gui.plugins.OptionalPagePlugin}.
     * 
     * @param descriptor Page descriptor.
     */
    public void add(OptionalPage descriptor) {
        if (SwingUtilities.isEventDispatchThread()) {
            OptionalPageHandler handler = new OptionalPageHandler(descriptor);
            optionalPages.add(handler);
            optionalPages.trimToSize();
            handler.start();            
        } else {
            SwingUtilities.invokeLater(() -> add(descriptor));
        }
    }
    
    /**
     * Removes a previously registered optional page with the specified path.
     * See {@link #add(OptionalPage)}.
     * 
     * @deprecated Use removeOptionalPage(OptionalPage descriptor) instead.
     * @param path Page path.
     */
    @Deprecated
    public void remove(String path) {
        if (SwingUtilities.isEventDispatchThread()) {
            Iterator<OptionalPageHandler> it = optionalPages.iterator();
            while (it.hasNext()) {
                OptionalPageHandler handler = it.next();
                if (handler.getDescriptor().getPath().equals(path)) {
                    it.remove();
                    handler.stop();
                    break;
                }
            }            
        } else {
            SwingUtilities.invokeLater(() -> LsstOptionalPageService.this.remove(path));
        }
    }
    
    /**
     * Removes a previously registered optional page.
     * See {@link #add(OptionalPage)}.
     * 
     * @param descriptor Descriptor of the page to be removed
     */
    public void remove(OptionalPage descriptor) {
        if (SwingUtilities.isEventDispatchThread()) {
            Iterator<OptionalPageHandler> it = optionalPages.iterator();
            while (it.hasNext()) {
                OptionalPageHandler handler = it.next();
                if (handler.getDescriptor() == descriptor) {
                    it.remove();
                    handler.stop();
                    break;
                }
            }            
        } else {
            SwingUtilities.invokeLater(() -> remove(descriptor));
        }
    }
    
    
// -- Programmatically opening/closing pages: ----------------------------------
    
    public void setOpen(OptionalPage descriptor, AgentInfo agent, boolean isOpen) {
        OptionalPageHandler h = getHandler(descriptor);
        if (h != null) {
            h.setOpen(agent, isOpen);
        }
    }
    
    
// -- Saving/restoring individual optional pages : -----------------------------
    
    public HashMap<String,Boolean> save(OptionalPage descriptor) {
        OptionalPageHandler h = getHandler(descriptor);
        if (h != null) {
            return h.save();
        }
        return null;
    }
    
    public void restore(OptionalPage descriptor, HashMap<String,Boolean> storageBean) {
        OptionalPageHandler h = getHandler(descriptor);
        if (h != null) {
            h.restore(storageBean);
        }
    }
    
    
// -- Saving/restoring pages managed by this plugin : --------------------------

    @Override
    public ComponentDescriptor save() {
        
        HashMap<String, Boolean> open = new HashMap<>();
        optionalPages.forEach(h -> {
            if (h.getDescriptor().isAutoSave()) {
                open.putAll(h.save());
            }
        });
        
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        desc.setOpen(open);
        return desc;
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        try {
            Descriptor desc = (Descriptor) storageBean;
            HashMap<String, Boolean> open = desc.getOpen();
            optionalPages.forEach(h -> {
                if (h.getDescriptor().isAutoSave()) {
                    h.restore(open);
                }
            });
        } catch (ClassCastException | NullPointerException x) {
        }
        return true;
    }
    
    static public class Descriptor extends ComponentDescriptor {

        private HashMap<String, Boolean> open;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public HashMap<String, Boolean> getOpen() {
            return open;
        }

        public void setOpen(HashMap<String, Boolean> open) {
            this.open = open;
        }
        
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    OptionalPageHandler getHandler(OptionalPage descriptor) {
        for (OptionalPageHandler h : optionalPages) {
            OptionalPage d = h.getDescriptor();
            if (d == descriptor) {
                return h;
            }
        }
        return null;
    }

}
