package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.Color;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.gconsole.plugins.tracer.FilteredMessage.Flag;
import static org.lsst.ccs.gconsole.plugins.tracer.TracerFilter.Mode.*;
import static org.lsst.ccs.gconsole.plugins.tracer.TracerFilter.Target.*;
import static org.lsst.ccs.gconsole.plugins.tracer.TracerFilter.Method.*;

/**
 * Tracer-tailored implementation of {@link MessageFilter}.
 *
 * @author onoprien
 */
public class TracerFilter implements MessageFilter {
    
    /** Enumeration of filter application modes. */
    public enum Mode {
        
        /** Reject messages that do not satisfy this filter. */
        ON("Reject messages that do not satisfy this filter", "On"),
        /** Accept all messages, change properties of those that satisfy this filter. */
        PASS("Accept all messages, change properties of those that satisfy this filter", "Pass"),
        /** Ignore this filter. */
        OFF("Ignore this filter", "Off");
        
        private final String toolTip;
        private final String hr;
        
        Mode(String toolTip, String humanReadable) {
            this.toolTip = toolTip;
            hr = humanReadable;
        }
        
        public String getToolTip() {
            return toolTip;
        }
        
        @Override
        public String toString() {
            return hr;
        }
    }
    
    /** Enumeration of targets to which the filter can be applied. */
    public enum Target {
        /** The filter is applied to the BusMessage object. If the {@link Method} of the filter requires 
         * a String as a target, the BusMessage is converted to String by calling its toString() method. */
        MESSAGE("The filter is applied to the bus message object. If the Method of<br>"
             + " the filter requires a string as a target, the message is<br>"
             + " converted to a string.",
             "Message"),
        /** The filter is applied to the BusMessage object. If the {@link Method} of the filter requires 
         * a String as a target, the BusMessage is converted to String by calling its toString() method. */
        OBJECT("The filter is applied to the subsystem-provided object embedded<br>"
             + " into the bus message. If the Method of the filter requires<br>"
             + " a string as a target, the object is converted to a string.",
             "Object"),
        /** The filter is applied to the name of the BusMessage origin. */
        SOURCE("The filter is applied to the name of the BusMessage origin.", "Source"),
        /** The filter is applied to the string produced by formatters embedded in previously applied filters.
         * If none of the previously applied filters formatted the message, the BusMessage is converted 
         * to String by calling its toString() method. */
        TEXT("The filter is applied to the string produced by formatters embedded<br>"
           + " in previously applied filters. If none of the previously applied<br>"
           + " filters formatted the message, the BusMessage is converted to<br>"
           + " String by calling its toString() method.",
           "Text"),
        /** The filter is applied to the comma-separated list of flags attached to the message. */
        FLAG("The filter is applied to the comma-separated list of flags attached<br>"
           + " to the message.",
           "Flag"),
        /**
         * The filter definition string is expected to be in <tt>Template==Definition</tt> format.
         * The filter is applied to the string produced by parsing the message with the template. 
         * If parsing fails, the filter is not satisfied. The template may include place holders
         * in the <tt>${name}</tt> format, where <tt>name</tt> is a name of the message property, 
         * or one of the following:
         * <dl>
         * <dt>SOURCE</dt><dd>Name of message origin.</dd>
         * <dt>TEXT</dt><dd>String formatted by previous filters.</dd>
         * <dt>DATE</dt><dd>Message date in 'yyyy-MM-dd' format.</dd>
         * <dt>TIME</dt><dd>Message time in 'HH:mm:ss' format.</dd>
         * <dt>DT</dt><dd>Message date and time in 'yyyy-MM-dd HH:mm:ss' format.</dd>
         * <dt>LOGGER</dt><dd>The name of the logger.</dd>
         * <dt>LEVEL</dt><dd>The level of the log message.</dd>
         * <dt>MESSAGE</dt><dd>The formatted message associated with the bus message.</dd>
         * <dt>STATE</dt><dd>The subsystem state embedded in the status message.</dd>
         * <dt>DESTINATION</dt><dd>The destination of the command message.</dd>
         * </dl>
         */
        TEMPLATE("The filter definition string is expected to be in <br>"
               + "<tt>Template==Definition</tt> format. The filter is applied to<br>"
               + " the string produced by parsing the message with the template.<br>"
               + " If parsing fails, the filter is not satisfied. The template<br>"
               + " may include place holders in the <tt>${name}</tt> format,<br>"
               + " where <tt>name</tt> is a name of the message property, or one<br>"
               + " of the following:<br>"
               + " <b>SOURCE:</b> Name of message origin<br>"
               + " <b>TEXT</b> String formatted by previous filters<br>"
               + " <b>DATE</b> Message date in 'yyyy-MM-dd' format<br>"
               + " <b>TIME</b> Message time in 'HH:mm:ss' format<br>"
               + " <b>DT</b> Message date and time in 'yyyy-MM-dd HH:mm:ss' format<br>"
               + " <b>LOGGER</b> Name of the logger<br>"
               + " <b>LEVEL</b> Level of the log message<br>"
               + " <b>MESSAGE</b> Formatted message associated with the bus message<br>"
               + " <b>STATE</b> Subsystem state embedded in the status message<br>"
               + " <b>DESTINATION</b> Destination of the command message<br>"
               ,"Template");
        
        private final String toolTip;
        private final String hr;
        
        Target(String toolTip, String humanReadable) {
            this.toolTip = toolTip;
            hr = humanReadable;
        }
        
        public String getToolTip() {
            return toolTip;
        }
        
        @Override
        public String toString() {
            return hr;
        }
    }
    
    /**
     * Enumeration of methods that determine how the filter definition string is interpreted 
     * to decide whether the target satisfies the filter.
     */
    public enum Method {
        /** The definition string is a regular expression the target should match. */
        REGEX("The definition string is a regular expression the target should match.", "Reg Ex"),
        /** The definition string is a Unix-style wildcard the target should match. */
        WILDCARD("The definition string is a Unix-style wildcard the target should match.", "Wildcard"),
        /** The target should contain the definition string. */
        CONTAINS("The target should contain the definition string.", "Contains"),
        /** The target should be equal to the definition string. */
        EQUALS("The target should be equal to the definition string.", "Equals"),
        /** The target should be an instance of the class specified by the definition string, or of its subclass. */
        CLASS("The target should be an instance of the class specified by the<br>"
             + " definition string, or of its subclass.", "Class"),
        /** The definition string is the name of the filter that should be applied to the target. */
        NAME("The definition string is the name of the filter that should be<br>"
             + " applied to the target.", "Name");
        
        private final String toolTip;
        private final String hr;
        
        Method(String toolTip, String humanReadable) {
            this.toolTip = toolTip;
            hr = humanReadable;
        }
        
        public String getToolTip() {
            return toolTip;
        }
        
        @Override
        public String toString() {
            return hr;
        }
    }

// -- Fields : -----------------------------------------------------------------
    
    /** Delimeter between target and pattern parts of the filter definition code string. */
    static final public String T_P_DELIMETER = "<=>";
  
    static final private Pattern pPar = Pattern.compile("\\$\\{([^}]+)\\}");
    
    static final private SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
    static final private SimpleDateFormat timeFormat = new SimpleDateFormat("HH:mm:ss");
    static final private SimpleDateFormat dtFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    static final private HashMap<String,Function<FilteredMessage,String>> key2func;
    static {
        key2func = new HashMap<>();
        key2func.put("SOURCE", m -> m.getBusMessage().getOriginAgentInfo().getName());
        key2func.put("TEXT", m -> m.getMessage());
        key2func.put("DATE", m -> dateFormat.format(new Date(m.getBusMessage().getTimeStamp())));
        key2func.put("TIME", m -> timeFormat.format(new Date(m.getBusMessage().getTimeStamp())));
        key2func.put("DT", m -> dtFormat.format(new Date(m.getBusMessage().getTimeStamp())));
        key2func.put("LOGGER", m -> ((LogMessage)m.getBusMessage()).getLoggerName());
        key2func.put("LEVEL", m -> ((LogMessage)m.getBusMessage()).getLevel());
        key2func.put("MESSAGE", m -> ((LogMessage)m.getBusMessage()).getFormattedDetails());
        key2func.put("STATE", m -> ((StatusMessage)m.getBusMessage()).getState().toString());
        key2func.put("DESTINATION", m -> ((CommandMessage)m.getBusMessage()).getDestination());
    }
    
    private String name;
    
    private Mode mode;
    private boolean invert;
    private boolean format;
    private Target target;
    private Method method;
    private String code;
    private Color color;
    private Flag action;
    
    private Class clazz;
    private Predicate<String> tester;
    private MessageFilter delegate;
    private String pattern;
    private String[] template;
    
// -- Construction and initialization : ----------------------------------------
    
    public TracerFilter(Mode mode, boolean invert, boolean format, Target target, Method method, String code, Color color, Flag action, FilterRegistry registry) {

        this.mode = mode;
        this.invert = invert;
        this.format = format;
        this.target = target;
        this.method = method;
        this.action = action;
        this.color = color;
        this.code = code;
        
        try {
            switch (target) {
                case TEMPLATE:
                    String[] tokens = code.split(TracerFilter.T_P_DELIMETER);
                    String temp = tokens[0];
                    Matcher m = pPar.matcher(temp);
                    ArrayList<String> t = new ArrayList<>();
                    int s = 0;
                    while (m.find()) {
                        t.add(temp.substring(s, m.start()));
                        t.add(m.group(1));
                        s = m.end();
                    }
                    if (s < temp.length()) t.add(temp.substring(s));
                    template = t.toArray(new String[t.size()]);
                    pattern = tokens.length == 1 ? "" : tokens[1];
                    break;
                default:
                    pattern = code;
            }
        } catch (IndexOutOfBoundsException x) {
            throw new IllegalArgumentException("Unable to create tracer filter", x);
        }
        
        try {
            switch (method) {
                case REGEX:
                    tester = Pattern.compile(pattern).asPredicate();
                    break;
                case WILDCARD:
                    pattern = wildcardToRegex(pattern);
                    tester = Pattern.compile(pattern).asPredicate();
                    break;
                case CONTAINS:
                    tester = message -> message.contains(pattern);
                    break;
                case EQUALS:
                    tester = message -> message.equals(pattern);
                    break;
                case CLASS:
                    clazz = Class.forName(pattern);
                    break;
                case NAME:
                    if (registry == null) throw new IllegalArgumentException("Filter registry is not specified");
                    delegate = registry.createFilter(pattern);
//                    delegate = registry.getFilter(pattern);
                    if (delegate == null) throw new IllegalArgumentException("Unknown filter: " + code);
                    break;
                default:
                    throw new IllegalArgumentException("Unknown filtering method: " + method);
            }
        } catch (ClassNotFoundException|IllegalArgumentException x) {
            throw new IllegalArgumentException("Unable to create tracer filter", x);
        }
        
    }
    
// -- Getters/Setters : --------------------------------------------------------
    
    public Mode getMode() {return mode;}
    
    public boolean isInverted() {return invert;}
    
    public boolean isFormatting() {return format;}
    
    public Target getTarget() {return target;}
    
    public Method getMethod() {return method;}
    
    public String getCode() {return code;}
    
    public Color getColor() {return color;}
    
    public FilteredMessage.Flag getAction() {return action;}
    
    public void setName(String name) {
        this.name = name;
    }

    
// -- Implementing MessageFilter : ---------------------------------------------
    
    @Override
    public String getPath() {
        return name;
    }

    @Override
    public FilteredMessage test(FilteredMessage filteredMessage) {
        
        if (mode == OFF) return filteredMessage;
        
        // prepare target to which the filtering will be applied
        
        Object targetObject;
        try {
            switch (target) {
                case MESSAGE:
                    targetObject = filteredMessage.getBusMessage(); break;                    
                case OBJECT:
                    targetObject = filteredMessage.getBusMessage().getObject(); break;
                case SOURCE:
                    targetObject = filteredMessage.getBusMessage().getOriginAgentInfo().getName(); break;
                case TEXT:
                    targetObject = filteredMessage.getText();
                    if (targetObject == null) targetObject = filteredMessage.getBusMessage().toString();
                    break;
                case FLAG:
                    EnumSet<Flag> flags = filteredMessage.getFlags();
                    if (flags == null) {
                        targetObject = "";
                    } else {
                        StringBuilder sb = new StringBuilder();
                        flags.forEach(flag -> sb.append(",").append(flag));
                        targetObject = sb.substring(1);
                    }
                    break;
                case TEMPLATE:
                    targetObject = expand(filteredMessage); break;
                default:
                   targetObject = null; // assert known target
            }
        } catch (Throwable t) {
            targetObject = null; // failed to construct target
        }
        
        // reject message if target cannot be identified
        
        boolean accept = (targetObject != null);
        
        // if the message is not already rejected, apply method to target and set acceptance flag
        
        if (accept) {
            switch (method) {
                case REGEX:
                case WILDCARD:
                case CONTAINS:
                case EQUALS:
                    String s = targetObject.toString();
                    accept = tester.test(targetObject.toString());
                    break;
                case CLASS:
                    accept = clazz.isAssignableFrom(targetObject.getClass());
                    break;
                case NAME:
                    FilteredMessage fm = new TracerMessage(filteredMessage);
                    if (targetObject instanceof String) {
                        fm.setText((String) targetObject);
                    }
                    fm = delegate.test(fm);
                    if (fm == null) {
                        accept = false;
                    } else {
                        accept = true;
                        targetObject = fm;
                    }
                    break;
            }
        }
                
        // invert acceptance flag if necessary
        
        if (invert) accept = !accept;
        
        // modify the message if accepted
        
        if (accept) {
            if (format) {
                if (targetObject instanceof FilteredMessage) {
                    filteredMessage = (FilteredMessage) targetObject;
                } else if (targetObject != null) {
                    filteredMessage.setMessage(targetObject.toString());
                }
            }
            if (color != null) {
                filteredMessage.setColor(color);
            }
            if (action != null) {
                filteredMessage.addFlag(action);
            }
            return filteredMessage;
        } else {
            return mode == ON ? null : filteredMessage;
        }
        
    }
    
                
// -- Local methods : ----------------------------------------------------------
    
    private String expand(FilteredMessage message) {
        StringBuilder sb = new StringBuilder();
        for (int i=0; i<template.length; ) {
            sb.append(template[i++]);
            if (i<template.length) sb.append(expand(template[i++], message));
        }
        return sb.toString();
    }
    
    private String expand(String key, FilteredMessage message) {
        String out = key2func.get(key).apply(message);
        if (out == null) {
            Object o = message.getProperty(key);
            if (o != null) out = o.toString();
        }
        return out;
    }
    
    static private String wildcardToRegex(String wildcard) {
        StringBuilder s = new StringBuilder();
//        s.append('^');
        for (int i = 0, is = wildcard.length(); i < is; i++) {
            char c = wildcard.charAt(i);
            switch (c) {
                case '*':
                    s.append(".*");
                    break;
                case '?':
                    s.append(".");
                    break;
                case '(':
                case ')':
                case '[':
                case ']':
                case '$':
                case '^':
                case '.':
                case '{':
                case '}':
                case '|':
                case '\\':
                    s.append("\\");
                    s.append(c);
                    break;
                default:
                    s.append(c);
                    break;
            }
        }
//        s.append('$');
        return (s.toString());
    }


}
