package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.Arrays;
import java.util.concurrent.ExecutionException;
import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.SwingWorker;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.command.CommandInvocationException;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.command.TokenizedCommand;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.util.session.Savable;
import org.lsst.ccs.messaging.CommandRejectedException;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.logging.StackTraceFormats;
import org.lsst.ccs.utilities.structs.TreeBranch;

/**
 * Graphic panel that provides command dictionary interface to a specific subsystem.
 *
 * @author onoprien
 */
public final class AgentPanel extends JSplitPane implements Savable, ChangeListener {

// -- Fields : -----------------------------------------------------------------
    
    private final Logger logger = Logger.getLogger("org.lsst.ccs.plugin.jas3.dictionary");
    
    private AgentHandle agent;
    
    private final DictionaryTreePanel dictionaryTree;
    private final CommandListPanel commandListPanel;
    private final ArgInputPanel argInputPanel;
    private final HTMLTextPane resultPane;
    
    Descriptor descriptor;

// -- Life cycle : -------------------------------------------------------------
    
    public AgentPanel() {
        
        super(VERTICAL_SPLIT);
        setContinuousLayout(true);
        descriptor = new Descriptor();

        JPanel topPane = new JPanel(new GridLayout(1, 3));
        setTopComponent(topPane);
        resultPane = new HTMLTextPane();
        setBottomComponent(new JScrollPane(resultPane));
        
        dictionaryTree = new DictionaryTreePanel(this);
        topPane.add(new JScrollPane(dictionaryTree));

        commandListPanel = new CommandListPanel(this);
        topPane.add(new JScrollPane(commandListPanel));
        dictionaryTree.addTreeSelectionListener(commandListPanel);
        commandListPanel.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                if (e.getClickCount() == 2) {
                    DictionaryCommand cmd = commandListPanel.getSelectedValue();
                    // Double click is only enabled for command without arguments
                    if (cmd.getArguments().length == 0) {
                        sendCommand();
                    }
                }
            }
        });

        JPanel topPaneRightPanel = new JPanel(new BorderLayout());
        topPane.add(topPaneRightPanel);        
        
        argInputPanel = new ArgInputPanel();
        topPaneRightPanel.add(argInputPanel, BorderLayout.CENTER);
        commandListPanel.addListSelectionListener(argInputPanel);

        Box buttonPane = Box.createHorizontalBox();
        topPaneRightPanel.add(buttonPane, BorderLayout.SOUTH);
        
        JCheckBox cb = new JCheckBox("Hide locked");
        buttonPane.add(cb);
        cb.setToolTipText("Hide commands unavailable due to agent lock state");
        cb.setSelected(descriptor.hideUnavailableCommands);
        cb.addActionListener(e -> {
            descriptor.setHideUnavailableCommands(cb.isSelected());
            dictionaryTree.updateData(agent);
        });
        
        buttonPane.add(Box.createRigidArea(Const.HDIM));
        buttonPane.add(Box.createHorizontalGlue());
        
        JButton sendCmdButton = new JButton(new ImageIcon(this.getClass().getResource("ic_send_black_24dp.png")));
        buttonPane.add(sendCmdButton);
        sendCmdButton.setText("Send");
        sendCmdButton.setEnabled(false);
        sendCmdButton.addActionListener(e -> sendCommand());

        commandListPanel.addListSelectionListener(e -> {
            boolean en = commandListPanel.isVisibleCommandSelected();
            sendCmdButton.setEnabled(en);
            argInputPanel.setEnabled(en);
        });
        
    }

    
// -- Receiving external events : ----------------------------------------------
    
    void setAgent(AgentHandle agent) {
        if (this.agent != null) {
            this.agent.removeListener(this);
        }
        this.agent = agent;
        if (agent != null) {
            if (agent.getDictionaries().isEmpty()) {
                agent.updateDictionaries();
            }
            agent.addListener(this);
        }
        dictionaryTree.updateData(agent);
    }

    /**
     * Called in response to changes in agent data.
     * @param e Event
     */
    @Override
    public void stateChanged(ChangeEvent e) {
        if (agent == null || !agent.isAdjusting()) {
            dictionaryTree.updateData(agent);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------

    private void sendCommand() {
        TreeBranch<ComponentDictionary> destinationNode = (TreeBranch<ComponentDictionary>) dictionaryTree.getLastSelectedPathComponent();
        ComponentDictionary commandContext = destinationNode.getContent();
        String s = " ";
        DictionaryCommand cmd = commandListPanel.getSelectedValue();
        if (cmd != null) {
            s += argInputPanel.getFormattedArgumentValues();
            new SendCommandWorker(cmd.getCommandName() + s, commandContext.commandSet, commandContext.path).execute();
        }
    }
    
    public class SendCommandWorker extends SwingWorker<String, String> {
        
        private final String cmd;
        private final CommandSet destination;
        private final String path;

        public SendCommandWorker(String cmdName, CommandSet dest, String path) {
            this.cmd = cmdName;
            this.destination = dest;
            this.path = path;
        }

        @Override
        protected String doInBackground() throws Exception {
            resultPane.insertHTML("<b>> " + path + " " + cmd +" ...</b>");
            try {
                Object res = destination.invoke(new TokenizedCommand(cmd));
                if (res instanceof Exception) {
                    throw (Exception)res;
                } else if (res == null) {
                    return "Done.";
                } else {
                    if (res.getClass().isArray()) {
                        try {
                            res = Arrays.deepToString((Object[])res);
                        } catch (ClassCastException x) {
                        }
                    }
                    return res.toString().replaceAll("(\\r\\n|\\n)", "<br />");
                }
            } catch (CommandInvocationException x) {
                StringBuilder sb = new StringBuilder("<font color=\"#FF0000\">");
                Throwable t = x.getCause();
                if (t instanceof CommandRejectedException) {
                    CommandNack nack = ((CommandRejectedException)t).getCommandNack();
                    sb.append("Command rejected");
                    if (nack != null) {
                        String reason = nack.getReason();
                        if (reason != null && !reason.isEmpty()) {
                            sb.append(": ").append(reason);
                        }
                    }
                } else if (t == null) {
                    String s = x.getMessage();
                    s = s.replace("No handler found for command", "No such command:");
                    sb.append(s);
                } else {
                    sb.append(t.getMessage());
                }
                return sb.append("</font>").toString();
            } catch (Exception e) {
                return "<html> <font color=\"#FF0000\"> Exception caught : " + StackTraceFormats.toString(e, 1) + "</font></html>";
            }
        }

        @Override
        public void done() {
            try {
                String result = get();
                resultPane.insertHTML(result);
            } catch (InterruptedException ex) {
                logger.error(ex);
            } catch (ExecutionException ex) {
                logger.error(ex);
            }
        }
    }
    
// -- Saving/restoring : -------------------------------------------------------    
    
    @Override
    public void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
        } else if (descriptor == null) {
            this.descriptor = new Descriptor();
        }
    }

    @Override
    public Descriptor save() {
        return descriptor;
    }
    
    static public class Descriptor implements Serializable {

        private boolean hideUnavailableCommands;

        public boolean isHideUnavailableCommands() {
            return hideUnavailableCommands;
        }

        public void setHideUnavailableCommands(boolean hideUnavailableCommands) {
            this.hideUnavailableCommands = hideUnavailableCommands;
        }
        
    }
    
}
