package org.lsst.ccs.gconsole.base;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Dimension;
import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.Executable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Parameter;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import static org.lsst.ccs.gconsole.base.Const.*;
import org.lsst.ccs.gconsole.util.session.Savable;
import org.openide.util.Lookup.Template;

/**
 * Dialog that allows selecting a class name and instantiating an object.
 *
 * @author onoprien
 */
public class InstanceDialog extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private Object out;
    private Item item;
    private Object[] parameters;
    
    private Box rightPanel;
    private List<ParPanel> parPanels;
    private JButton okButton;
    
    private Descriptor descriptor;
    

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructor.
     * <p>
     * Items provided to this constructor may wrap:<ul>
     * <li>String. Interpreted as a class name. Must be a subclass of the
     * factory class. If the class declares static {@code getName()} method, it
     * is used in this dialog. If the class declares static {@code getInstance(...)} method,
     * it is used to create an instance. Otherwise, it's public constructor is used.
     * <li>An object of target class. This object is returned if selected.
     * <li>An object with {@code getInstance()} method that returns an instance of the target class.
     * </ul>
     *
     * @param parentComponent Owner.
     * @param title Dialog title.
     * @param items Collection of items to select from.
     */
    private InstanceDialog(Component parentComponent, String title, Collection<Item> items) {
        super(parentComponent == null ? Console.getConsole().getWindow() : SwingUtilities.getWindowAncestor(parentComponent), title, Dialog.ModalityType.APPLICATION_MODAL);
        
        ItemNode root = new ItemNode("Root");
        for (Item item : items) {
            ItemNode parent = root;
            String[] ss = item.getPath().split("/+");
            for (int i=0; i<ss.length-1; i++) {
                parent = parent.getChild(ss[i]);
            }
            parent.getChild(ss[ss.length-1]).item = item;
        }
        JTree tree = new JTree(root);
        tree.setRootVisible(false);
        tree.setShowsRootHandles(true);
        tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        tree.addTreeSelectionListener(e -> {
            ItemNode node = (ItemNode) tree.getLastSelectedPathComponent();
            showItem(node == null ? null : node.item);
        });
        Enumeration<TreeNode> en = root.children();
        while (en.hasMoreElements()) {
            tree.expandPath(new TreePath(((ItemNode)en.nextElement()).getPath()));
        }
        JScrollPane treeScrollPane = new JScrollPane(tree);

        rightPanel = Box.createVerticalBox();
        JScrollPane rightScrollPane = new JScrollPane(rightPanel);
 
        JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, treeScrollPane, rightScrollPane);
        splitPane.setDividerLocation(300);
        Dimension minimumSize = new Dimension(100, 50);
        treeScrollPane.setMinimumSize(minimumSize);
        rightScrollPane.setMinimumSize(minimumSize);
        splitPane.setPreferredSize(new Dimension(850, 500));
        add(splitPane, BorderLayout.CENTER);
        
        Box buttonBox = Box.createHorizontalBox();
        buttonBox.setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
        buttonBox.add(Box.createHorizontalGlue());
        JButton b = new JButton("Cancel");
        b.addActionListener(e -> {
            out = null;
            item = null;
            dispose();
        });
        buttonBox.add(b);
        buttonBox.add(Box.createRigidArea(HDIM));
        okButton = new JButton("OK");
        okButton.addActionListener(e -> {
            if (item != null) {
                int n = parPanels.size();
                Object[] pars = new Object[n];
                String[] parStrings = new String[n];
                boolean parsValid = true;
                for (int i=0; i<n; i++) {
                    pars[i] = parPanels.get(i).getValue();
                    if (pars[i] == null) {
                        parsValid = false;
                    } else {
                        parStrings[i] = pars[i].toString();
                    }
                }
                if (parsValid) {
                    out = item.getInstance(pars);
                    if (out != null) {
                        descriptor = new Descriptor();
                        descriptor.setPath(item.getPath());
                        descriptor.setParameters(parStrings);
                        if (out instanceof Product) {
                            ((Product) out).setCreator(descriptor);
                        }
                        dispose();
                    }
                }
            }
        });
        buttonBox.add(okButton);
        buttonBox.add(Box.createRigidArea(HDIM));
        add(buttonBox, BorderLayout.SOUTH);
        showItem(null);
    }
    
    
// -- Static methods for displaying dialog and returning created instance : ----
    
    static public InstanceDialog show(Class<?> targetClass, Component parentComponent, String title, Collection<?> factories) {
        InstanceDialog dialog = new InstanceDialog(parentComponent, title, wrapFactories(targetClass, factories));
        dialog.setSize(dialog.getPreferredSize());
        dialog.pack();
        dialog.setLocationRelativeTo(parentComponent);
        dialog.setVisible(true);
        return dialog;
    }
    
    static public InstanceDialog show(Class<?> targetClass, Component parentComponent, String title, String lookupId) {
        return show(targetClass, parentComponent, title, findItems(targetClass, lookupId));
    }

    
    static public <T> T getInstance(Class<T> targetClass, Component parentComponent, String title, Collection<?> factories) {
        return (T) show(targetClass, parentComponent, title, factories).getInstance();
    }
    
    static public <T> T getInstance(Class<T> targetClass, Component parentComponent, String title, String lookupId) {
        return (T) show(targetClass, parentComponent, title, lookupId).getInstance();
    }
    
    static public <T> T getInstance(Class<T> targetClass, Component parentComponent, String title) {
        return getInstance(targetClass, parentComponent, title, targetClass.getCanonicalName());
    }

    
    static public <T> T getInstance(Descriptor descriptor, Class<T> targetClass, Collection<?> factories) {
        for (Item item : wrapFactories(targetClass, factories)) {
            if (item.getPath().equals(descriptor.getPath())) {
                T out = (T) item.getInstance(descriptor);
                if (out instanceof Product) {
                    ((Product)out).setCreator(descriptor);
                }
                return out;
            }
        }
        return null;
    }
    
    static public <T> T getInstance(Descriptor descriptor, Class<T> targetClass, String lookupId) {
        return getInstance(descriptor, targetClass, findItems(targetClass, lookupId));
    }
    
    static public <T> T getInstance(Descriptor descriptor, Class<T> targetClass) {
        return getInstance(descriptor, targetClass, targetClass.getCanonicalName());
    }

    
    static public <T> T getInstance(Descriptor descriptor, Class<T> targetClass, Component parentComponent, String title, Collection<?> factories) {
        throw new UnsupportedOperationException();  // FIXME
    }
    
    static public <T> T getInstance(Descriptor descriptor, Class<T> targetClass, Component parentComponent, String title, String lookupId) {
        return getInstance(descriptor, targetClass, parentComponent, title, findItems(targetClass, lookupId));
    }
    
    static public <T> T getInstance(Descriptor descriptor, Class<T> targetClass, Component parentComponent, String title) {
        return getInstance(descriptor, targetClass, parentComponent, title, targetClass.getCanonicalName());
    }
    

// -- Getters : ----------------------------------------------------------------
    
    public Object getInstance() {
        return out;
    }
    
    public Descriptor getDescriptor() {
        return descriptor;
    }
            

// -- Helper classes : ---------------------------------------------------------
    
    static public class Item implements Comparable<Item> {
        
        private final Object factory;
//        private final Class<?> targetClass;
        
        private Executable exec;
        
        private String path;
        private String name;
        private String description;
        
        public Item(Class<?> targetClass, Object factory) {
            this(targetClass, factory, null, null, null);
        }
        
        /**
         * Constructs an {@code Item}.
         * @param targetClass Type of the target object the {@code InstanceDialog} is supposed to create.
         * @param factory Object that is used to create the target.
         * @param path Path to this item in the tree displayed by the dialog (slash-separated).
         * @param name Name of this item.
         * @param description Description of this item.
         * @throws IllegalArgumentException if there is no valid way to construct an
         *                                  instance of {@code targetClass} from {@code factory}.
         */
        public Item(Class<?> targetClass, Object factory, String path, String name, String description) {
            
            this.factory = factory;
            
            try {
                
                // If factory is String, convert to Class
 
                if (factory instanceof String) {  
                    factory = Class.forName((String) factory);
                }
                
                // Handle 3 different types of factories
            
                if (factory instanceof Class<?>) { // factory is Class - find executable
                    Class<?> c = (Class<?>) factory;
                    if (targetClass.isAssignableFrom(c) && Modifier.isPublic(c.getModifiers())) {
                        for (Method m : c.getMethods()) {
                            if (m.getName().equals("getInstance") && targetClass.isAssignableFrom(m.getReturnType()) && check(m)) {
                                exec = m;
                                break;
                            }
                        }
                        if (exec == null) {
                            for (Constructor<?> con : c.getConstructors()) {
                                if (check(con)) {
                                    exec = con;
                                    break;
                                }
                            }
                        }
                        if (exec == null) {
                            throw new IllegalArgumentException("Factory class does not provide a way to create an instance");
                        }
                        if (name == null || path == null || description == null) {
                            ConsoleLookup ann = c.getAnnotation(ConsoleLookup.class);
                            if (ann != null) {
                                if (path == null) {
                                    path = ann.path();
                                    if (path == null || path.trim().isEmpty()) {
                                        path = c.getName().replace(".", "/");
                                    }
                                }
                                if (name == null) name = ann.name();
                                if (description == null) description = ann.description();
                            }
                        }
                    } else {
                        throw new IllegalArgumentException("Factory class is incompatible with target class");
                    }
                    
                } else if (targetClass.isInstance(factory)) { // factory is instance of target class - will return it
                    
                    // do not need exec, factory already set
                    
                } else { // factory is instance of arbitrary class - look for getInstance(...) method
                    
                    for (Method m : factory.getClass().getMethods()) {
                        if (m.getName().equals("getInstance") && targetClass.isAssignableFrom(m.getReturnType()) && check(m)) {
                            exec = m;
                            break;
                        }
                    }
                    if (exec == null) throw new IllegalArgumentException("Factory object does not have suitable getInstance(...) method");
                    
                }
                
                // if path, name, or description has not been set, try to extract it from factory object
                
                if ((name == null || path == null || description == null) && !(factory instanceof Class<?>)) {
                    Class<?> factoryClass = factory.getClass();
                    
                    if (path == null) {
                        try {
                            Method m = factoryClass.getMethod("getPath");
                            m.setAccessible(true);
                            path = (String) m.invoke(factory);
                        } catch (IllegalAccessException | IllegalArgumentException | NoSuchMethodException | SecurityException | InvocationTargetException x) {
                        }
                    }
                    if (path == null) path = factory.toString();
                    
                    if (name == null) {
                        try {
                            Method m = targetClass.getMethod("getName");
                            m.setAccessible(true);
                            name = (String) m.invoke(factory);
                        } catch (IllegalAccessException | IllegalArgumentException | NoSuchMethodException | SecurityException | InvocationTargetException x) {
                        }
                    }
                    
                    if (description == null) {
                        try {
                            Method m = targetClass.getMethod("getDescription");
                            m.setAccessible(true);
                            description = (String) m.invoke(factory);
                        } catch (IllegalAccessException | IllegalArgumentException | NoSuchMethodException | SecurityException | InvocationTargetException x) {
                        }
                    }
                    
                }
                
                // If name is still unknown, use the last segment of path
                
                if (name == null || name.trim().isEmpty()) {
                    name = path.substring(path.lastIndexOf(path) + 1);
                }
                
                this.path = path;
                this.name = name;
                this.description = description;
                if (exec != null) exec.setAccessible(true);

            } catch (Throwable x) {
                throw new IllegalArgumentException("Factory "+ factory +"is not suitable for creating an instance of "+ targetClass);
            }
        }
        
        Object getInstance(Object[] parameters) {
            Object out = null;
            try {
                if (exec == null) {
                    out = factory;
                } else if (exec instanceof Method) {
                    out = ((Method)exec).invoke(factory, parameters);
                } else if (exec instanceof Constructor) {
                    out = ((Constructor)exec).newInstance(parameters);
                }
            } catch (IllegalAccessException | IllegalArgumentException | InstantiationException | InvocationTargetException t) {
                Console.getConsole().error("Unable to construct", t);
            }
            return out;
        }
        
        Object getInstance(Descriptor desc) {
            Object out = null;
            try {
                if (exec == null) {
                    out = factory;
                } else {
                    Parameter[] pp = exec.getParameters();
                    String[] ss = desc.getParameters();
                    Object[] pars = new Object[pp.length];
                    for (int i=0; i<pp.length; i++) {
                        Class<?> pc = pp[i].getType();
                        if (pc.equals(int.class)) {
                            pars[i] = Integer.valueOf(ss[i]);
                        } else if (pc.equals(long.class)) {
                            pars[i] = Long.valueOf(ss[i]);
                        } else if (pc.equals(double.class)) {
                            pars[i] = Double.valueOf(ss[i]);
                        } else if (pc.equals(float.class)) {
                            pars[i] = Float.valueOf(ss[i]);
                        } else if (pc.equals(boolean.class)) {
                            pars[i] = Boolean.valueOf(ss[i]);
                        } else if (pc.equals(String.class)) {
                            pars[i] = ss[i];
                        } else if (pc.isEnum()) {
                            for (Object e : pc.getEnumConstants()) {
                                if (e.toString().equals(ss[i])) {
                                    pars[i] = e;
                                    break;
                                }
                            }
                        }
                    }
                    if (exec instanceof Method) {
                        out = ((Method) exec).invoke(factory, pars);
                    } else if (exec instanceof Constructor) {
                        out = ((Constructor) exec).newInstance(pars);
                    }
                }
            } catch (IllegalAccessException | IllegalArgumentException | InstantiationException | InvocationTargetException x) {
            }
            return out;            
        }

        public String getName() {
            return name;
        }

        public String getPath() {
            return path;
        }

        public String getDescription() {
            return description == null ? "" : description;
        }

        public Object getFactory() {
            return factory;
        }

        @Override
        public boolean equals(Object other) {
            return (other instanceof Item) && (((Item)other).path.equals(path));
        }

        @Override
        public int hashCode() {
            return path.hashCode();
        }

        @Override
        public int compareTo(Item other) {
            return path.compareTo(other.path);
        }
        
        // Locals : ------------------------------------------------------------
        
        private static boolean check(Executable executable) {
            if (!Modifier.isPublic(executable.getModifiers())) return false;
            for (Class<?> pc : executable.getParameterTypes()) {
                if (!(pc.equals(int.class) || pc.equals(long.class) || pc.equals(double.class) || pc.equals(float.class)
                        || pc.equals(boolean.class) || pc.equals(String.class) || Enum.class.isAssignableFrom(pc))) {
                    return false;
                }
            }
            return true;
        }
        
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    static private class ItemNode extends DefaultMutableTreeNode {
        
        Item item;
        
        ItemNode(String name) {
            super(name);
        }
        
        ItemNode getChild(String name) {
            int n = getChildCount();
            for (int i = 0; i < n; i++) {
                TreeNode child = getChildAt(i);
                if (child.toString().equals(name)) {
                    return (ItemNode) child;
                }
            }
            ItemNode child = new ItemNode(name);
            add(child);
            return child;
        }
        
    }
    
    static private class ParPanel extends JPanel {
        
        Parameter par;
        JComponent selector;
        
        ParPanel(Parameter parameter) {
            par = parameter;
            setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
            setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
            add(new JLabel(par.getType().getSimpleName() +" "+ par.getName()));
            add(Box.createRigidArea(HDIM));
            Class<?> pc = par.getType();
            if (pc.equals(int.class) || pc.equals(long.class) || pc.equals(double.class) || pc.equals(float.class) || pc.equals(String.class)) {
                JTextField field = new JTextField();
                field.setColumns(30);
                field.setMaximumSize(field.getPreferredSize());
                selector = field;
            } else if (pc.equals(boolean.class)) {
                selector = new JCheckBox();
            } else if (Enum.class.isAssignableFrom(pc)) {
                try {
                    selector = new JComboBox((Object[])pc.getMethod("values").invoke(null));
                } catch (Throwable t) {
                    selector = new JLabel("Unable to set parameter");
                    selector.setBorder(BorderFactory.createLineBorder(Color.RED, 2));
                }
            }
            add(selector);
            add(Box.createHorizontalGlue());
        }
        
        Object getValue() {
            Object out = null;
            Class<?> pc = par.getType();
            try {
                if (pc.equals(int.class)) {
                    JTextField field = (JTextField) selector;
                    out = Integer.valueOf(field.getText());
                } else if (pc.equals(long.class)) {
                    JTextField field = (JTextField) selector;
                    out = Long.valueOf(field.getText());
                } else if (pc.equals(double.class)) {
                    JTextField field = (JTextField) selector;
                    out = Double.valueOf(field.getText());
                } else if (pc.equals(float.class)) {
                    JTextField field = (JTextField) selector;
                    out = Float.valueOf(field.getText());
                } else if (pc.equals(String.class)) {
                    JTextField field = (JTextField) selector;
                    out = field.getText();
                } else if (pc.equals(boolean.class)) {
                    JCheckBox cb = (JCheckBox) selector;
                    out = cb.isSelected();
                } else if (Enum.class.isAssignableFrom(pc)) {
                    JComboBox cb = (JComboBox) selector;
                    out = cb.getSelectedItem();
                }
//                selectorPanel.setBorder(null);
                selector.setBorder(null);
            } catch (Throwable t) {
//                selectorPanel.setBorder(BorderFactory.createLineBorder(Color.RED, 2));
                selector.setBorder(BorderFactory.createLineBorder(Color.RED, 2));
            }
            return out;
        }
        
    }
    
    private class DescriptionPane extends JEditorPane {
        
        DescriptionPane(String s) {
            super("text/html", s);
        }

        @Override
        public Dimension getPreferredSize() {
            Dimension d = new Dimension(super.getPreferredSize());
            d.width = rightPanel.getWidth() - 2*HSPACE;
            return d;
        }
        
    }
    

// -- Local methods : ----------------------------------------------------------
    
    private void showItem(Item item) {
        this.item = item;
        rightPanel.removeAll();
        parPanels = Collections.emptyList();
        if (item != null) {
            StringBuilder sb = new StringBuilder();
            sb.append("<html>").append(item.getPath()).append("<br>");
            sb.append("<h3>").append(item.getName()).append("</h3>");
            String descr = item.getDescription();
            if (descr != null && !descr.isEmpty()) {
                sb.append(descr).append("");
            }
            sb.append("</html>");
            JEditorPane ep = new DescriptionPane(sb.toString());
            ep.setEditable(false);
            rightPanel.add(ep);
            
            if (item.exec != null) {
                Parameter[] pars = item.exec.getParameters();
                parPanels = new ArrayList<>(pars.length);
                if (pars.length > 0) {
                    Box parBox = Box.createVerticalBox();
                    parBox.setBorder(BorderFactory.createTitledBorder("Parameters"));
                    parBox.add(Box.createRigidArea(VDIM));
                    for (Parameter p : pars) {
                        ParPanel panel = new ParPanel(p);
                        parBox.add(panel);
                        parPanels.add(panel);
                        parBox.add(Box.createRigidArea(VDIM));
                    }
                    rightPanel.add(Box.createRigidArea(VDIM));
                    rightPanel.add(parBox);
                    rightPanel.add(Box.createRigidArea(VDIM));
                }
            }
            
            rightPanel.add(Box.createVerticalGlue());
            rightPanel.revalidate();
            okButton.setEnabled(true);
        } else {
            okButton.setEnabled(false);
        }
    }
        
    static private Set<Item> wrapFactories(Class<?> targetClass, Collection<?> factories) {
        TreeSet<Item> items = new TreeSet<>();
        for (Object factory : factories) {
            if (factory instanceof Item) {
                items.add((Item)factory);
            } else {
                try {
                    items.add(new Item(targetClass, factory));
                } catch (IllegalArgumentException x) {
                }
            }
        }
        return items;
    }
    
    static private Set<Item> findItems(Class<?> targetClass, String lookupId) {
        if (lookupId == null) lookupId = targetClass.getName();
        Collection<Object> instances = Console.getConsole().getConsoleLookup().lookup(new Template(String.class, lookupId, null)).allInstances();
        return wrapFactories(targetClass, instances);
    }

    
// -- Instance descriptor class : ----------------------------------------------
    
    static public class Descriptor implements Serializable {
        
        static private final String DELIMETER = "&";

        private String path;
        private String[] parameters;
        
        /**
         * Default constructor.
         */
        public Descriptor() {
        }
        
        /**
         * Constructs a descriptor from an encoded string.
         * @param encoded Encoded string as returned by {@code toString()} method.
         */
        public Descriptor(String encoded) {
            if (encoded != null && !encoded.isEmpty()) {
                String[] ss = encoded.split(DELIMETER, -1);
                path = ss[0];
                if (ss.length > 1) {
                    parameters = Arrays.copyOfRange(ss, 1, ss.length);
                }
            }
        }
        
        public Descriptor(String path, String... parameters) {
            this.path = path;
            this.parameters = (parameters == null || parameters.length == 0) ? null : Arrays.copyOf(parameters, parameters.length);
        }
        
        // --

        public String getPath() {
            return path;
        }

        public void setPath(String path) {
            this.path = path;
        }

        public String[] getParameters() {
            return parameters;
        }

        public void setParameters(String[] parameters) {
            this.parameters = parameters;
        }

        public String getParameters(int index) {
            return this.parameters[index];
        }

        public void setParameters(int index, String parameters) {
            this.parameters[index] = parameters;
        }
        
        // --

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            if (path != null) {
                sb.append(path);
                if (parameters != null && parameters.length != 0) {
                    for (String par : parameters) {
                        sb.append(DELIMETER).append(par);
                    }
                }
            }
            return sb.toString();
        }
        
    }
    
// -- Instantiated object interface : ------------------------------------------
    
    public interface Product {
        
        default void setCreator(Descriptor dialogDescriptor) {
            if (this instanceof Savable) {
                ((Savable)this).restore(dialogDescriptor);
            }
        }
        
    }

}
