package org.lsst.ccs.gconsole.plugins.monitor;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;
import org.lsst.ccs.gconsole.agent.AgentStatusListener;

/**
 * Adapter class that simplifies implementing monitor views.
 * <p>
 * Concrete classes extending {@code AbstractMonitorView1} should not, in general, override 
 * implementations of the {@link AgentStatusListener} interface methods provided by this adapter.
 * Instead, they should implement: <ul>
 * <li>{@link connect(AgentInfo)}
 * <li>{@link disconnect(AgentInfo)}
 * <li>{@link addChannels(AgentInfo agent, Map)}
 * <li>{@link removeChannels(AgentInfo, List)}
 * <li>{@link updateChannels(AgentInfo, Map)}
 * </ul>
 * Additionally, they may implement or override any {@link MonitorView} methods 
 * that are not a part of {@link AgentStatusListener} interface.
 * All these methods are called on the EDT.
 *
 * @author onoprien
 */
abstract public class AbstractMonitorView1 implements MonitorView {

// -- Fields : -----------------------------------------------------------------
    
    protected AgentChannelsFilter filter;
    protected Descriptor descriptor;
    
    private boolean fixedDisplayList;
    private Set<String> displayChannels;

// -- Life cycle : -------------------------------------------------------------
    
// -- Getters and setters : ----------------------------------------------------

    @Override
    public AgentChannelsFilter getFilter() {
        return filter;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        this.filter = filter;
        fixedDisplayList = filter != null && filter.getDisplayChannels() != null;
    }
    
    protected boolean isChannelListFixed() {
        return fixedDisplayList;
    }
    
    
// -- Agent status listener implementation : -----------------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            connect(event.getSource());
        });
    }

    @Override
    public void configure(AgentStatusEvent event) {
        if (displayChannels == null) {
            List<String> fixed = filter.getDisplayChannels();
            if (fixed == null) {
                fixedDisplayList = false;
                displayChannels = Collections.emptySet();
            } else {
                fixedDisplayList = true;
                displayChannels = new HashSet<>(fixed);
                SwingUtilities.invokeLater(() -> {
                    addChannels(event.getSource(), fixed.stream().collect(Collectors.toMap(dp -> dp, dp -> null)));
                });
            }
        }
        update(event);
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        update(event);
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        update(event);
        SwingUtilities.invokeLater(() -> {
            disconnect(event.getSource());
        });
    }
    
    private void update(AgentStatusEvent event) {
        
        if (fixedDisplayList) {
            
            Map<String,Map.Entry<AgentChannel,List<String>>> changedChannels = new LinkedHashMap<>();
            
            for (AgentChannel channel : event.getAddedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (displayChannels.contains(displayPath)) {
                        changedChannels.put(displayPath, new AbstractMap.SimpleEntry<>(channel, null));
                    }
                }
            }
            
            for (AgentChannel channel : event.getRemovedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (displayChannels.contains(displayPath)) {
                        changedChannels.put(displayPath, null);
                    }
                }
            }
            
            for (Map.Entry<AgentChannel, List<String>> change : event.getStatusChanges().entrySet()) {
                for (String displayPath : filter.getDisplayPath(change.getKey())) {
                    if (displayChannels.contains(displayPath)) {
                        changedChannels.put(displayPath, change);
                    }
                }
            }
            
            SwingUtilities.invokeLater(() -> {
                updateChannels(event.getSource(), changedChannels);
            });
            
        } else {
            
            Map<String,AgentChannel> addedChannels = null;
            for (AgentChannel channel : event.getAddedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (addedChannels == null) addedChannels = new LinkedHashMap<>();
                    addedChannels.put(displayPath, channel);
                }
            }
            
            List<String> removedChannels = null;
            for (AgentChannel channel : event.getRemovedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (removedChannels == null) removedChannels = new ArrayList<>(event.getRemovedChannels().size());
                    removedChannels.add(displayPath);
                }
            }
            
            Map<String,Map.Entry<AgentChannel,List<String>>> changedChannels = null;
            for (Map.Entry<AgentChannel,List<String>> e : event.getStatusChanges().entrySet()) {
                for (String displayPath : filter.getDisplayPath(e.getKey())) {
                    if (changedChannels == null) changedChannels = new LinkedHashMap<>();
                    changedChannels.put(displayPath, e);
                }
            }
            
            Map<String, AgentChannel> added = addedChannels;
            List<String> removed = removedChannels;
            Map<String,Map.Entry<AgentChannel,List<String>>> changed = changedChannels;
            SwingUtilities.invokeLater(() -> {
                if (added != null) addChannels(event.getSource(), added);
                if (removed != null) removeChannels(event.getSource(), removed);
                if (changed != null) updateChannels(event.getSource(), changed);
            });
            
        }
        
    }
    
    
// -- EDT hooks for updating the graphical component : -------------------------
    
    protected void connect(AgentInfo agent) {}
    
    protected void disconnect(AgentInfo agent) {}

    /**
     * Adds channels to display.
     * @param agent Source agent.
     * @param channels Map of display paths to channels.
     */
    abstract protected void addChannels(AgentInfo agent, Map<String,AgentChannel> channels);

    /**
     * Removes a channels from this view.
     * @param agent Source agent.
     * @param paths Display paths to remove.
     */
    abstract protected void removeChannels(AgentInfo agent, List<String> paths);
    
    /**
     * Called to notify this view that the specified channels have been updated.
     * @param agent Source agent.
     * @param channels Map of display paths to key-value pairs, where the key is as {@code AgentChannel} object
     *                 and the value is a list of changed attributes.
     */
    abstract protected void updateChannels(AgentInfo agent, Map<String,Map.Entry<AgentChannel,List<String>>> channels);
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor implements MonitorView.Descriptor {
        
        private MonitorPageDialog.Descriptor creator;
        private String name;
        
        /**
         * Returns a descriptor of the creator of this view.
         * @return Creator descriptor.
         */
        @Override
//        public MonitorPageDialog.Descriptor getCreator() {
        public Serializable getCreator() {
            return creator;
        }
        
        /**
         * Sets a descriptor of the creator of this view.
         * @param creator Creator descriptor.
         */
        @Override
        public void setCreator(Serializable creator) {
            if (creator instanceof MonitorPageDialog.Descriptor) {
                this.creator = (MonitorPageDialog.Descriptor) creator;
            }
        }
        
        /**
         * Returns the name of the view.
         * @return View name.
         */
        @Override
        public String getName() {
            return name;
        }
        
        /**
         * Sets the name of the view.
         * @param name View name.
         */
        @Override
        public void setName(String name) {
            this.name = name;
        }
        
    }
    
    /**
     * Returns a descriptor that contains information required to re-create this view in its current state.
     * The implementation provided by this class returns the descriptor previously set with {@code restore(...)}.
     * 
     * @return View descriptor, or {@code null} is this view does not support saving.
     */
    @Override
    public Descriptor save() {
        return descriptor;
    }
    
    /**
     * Restores this view to the state described by the provided descriptor, to the extent possible.
     * The implementation provided by this class stores the provided descriptor, so that it can be
     * retrieved with {@code save()} later.
     * 
     * @param descriptor View descriptor.
     */
    @Override
    public void restore(MonitorView.Descriptor descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
        }
    }
    
    /**
     * Returns a reference to the descriptor maintained by this view.
     * This is a convenience method intended to be overridden by subclasses to return
     * an appropriate subtype of {@code AbstractMonitorView.Descriptor}.
     * 
     * @return Descriptor maintained by this view.
     */
    protected Descriptor getDescriptor() {
        return descriptor;
    }
    
}
