package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;

/**
 * Data model for a graphical component that displays a single monitored value.
 * There is a single {@link MonitorField} instance associated with this model, 
 * describing the type of the displayed value, and a list of contributing {@code ChannelHandle} objects.
 * The model also contains the current {@link FormattedValue}.
 * 
 * @author onoprien
 */
public class MonitorCell {

// -- Fields : -----------------------------------------------------------------

    static public MonitorCell EMPTY = new MonitorCell(Collections.emptyList(), MonitorField.NULL, FormattedValue.EMPTY);
    
    private final List<ChannelHandle> items;
    private final MonitorField field;
    
    private FormattedValue data;


// -- Life cycle : -------------------------------------------------------------

    /**
     * Constructs a cell with no associated channels, {@code NULL} field, and no formatted current value.
     */
    MonitorCell() {
        this(Collections.emptyList(), MonitorField.NULL);
    }

    /**
     * Constructs a cell affected by a single data channel, with no current formatted value.
     * @param item Handle of the channel to be displayed, or {@code null} if cell is unaffected by any channels.
     * @param field Field to be displayed.
     */
    MonitorCell(ChannelHandle item, MonitorField field) {
        this(item == null ? Collections.emptyList() : Collections.singletonList(item), field);
    }

    /**
     * Constructs a cell affected by zero or more data channels, with no current formatted value.
     * @param items Handles of the channels contributing to this cell. The list will be owned by this cell.
     * @param field Field to be displayed.
     */
    MonitorCell(List<? extends ChannelHandle> items, MonitorField field) {
        this(items, field, null);
    }

    /**
     * Constructs a cell affected by zero or more data channels, with no current formatted value.
     * @param items Handles of the channels contributing to this cell. The list will be owned by this cell.
     * @param field Field to be displayed.
     */
    MonitorCell(List<? extends ChannelHandle> items, MonitorField field, FormattedValue data) {
        this.items = items == null ? Collections.emptyList() : (List<ChannelHandle>) items;
        this.field = field == null ? MonitorField.NULL : field;
        this.data = data;
    }
    
    
// -- Getters and setters : ----------------------------------------------------

    /**
     * Returns the list of handles for data channels that affect this cell.
     * @return List of contributing channels.
     */
    public List<ChannelHandle> getChannels() {
        return items;
    }

    /**
     * Returns the descriptor of the monitored field this cell displays.
     * @return Field descriptor.
     */
    public MonitorField getField() {
        return field;
    }

    /**
     * Sets the current formatted value to be displayed by this cell.
     * @param data New formatted value.
     */
    public void setData(FormattedValue data) {
        this.data = data;
    }

    /**
     * Returns the current formatted value to be displayed by this cell.
     * @return Current formatted value.
     */
    public FormattedValue getData() {
        return data;
    }

}
