package org.lsst.ccs.gconsole.plugins.alert;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;

/**
 * Alert handling service definition.
 * Instances of classes implementing this interface can be registered with the graphical console lookup.
 * Note that this service does not notify listeners when agents connect 
 *
 * @author onoprien
 */
public interface AlertService {
    
    /**
     * Submits a locally generated alert for processing. 
     * The alert is not published on the buses but stored by this service and the
     * listeners are notified.
     * <p>
     * Note that {@code AlertService} implementation is responsible for processing alerts published on the buses.
     * 
     * @param source Source of alert (typically the name of the subsystem).
     * @param alert Alert instance.
     * @param level Level at which the alert has been raised.
     * @param cause Information string describing the alert being raised.
     */
    void submitAlert(String source, Alert alert, AlertState level, String cause);
    
    /**
     * Adds a listener that will be notified of any future raised or cleared alerts.
     * 
     * @param listener Listener to be registered.
     */
    void addListener(AlertListener listener);
    
    /**
     * Removes an alert listener.
     * 
     * @param listener Listener to be removed.
     */
    void removeListener(AlertListener listener);
    
    /**
     * Returns a summary of alerts from the specified source currently known to this {@code AlertService}.
     * 
     * @param source Alerts source (typically the name of the subsystem).
     * @return Alerts summary.
     */
    RaisedAlertSummary getSummary(String source);
    
    /**
     * Returns the history for the specified alert ID.
     * 
     * @param source Source of the alerts (typically the name of the subsystem).
     * @param alertID Alert ID.
     * @return History of alerts.
     */
    default RaisedAlertHistory getHistory(String source, String alertID) {
        RaisedAlertSummary summary = getSummary(source);
        return summary == null ? null : summary.getRaisedAlert(alertID);
    }
    
}
