package org.lsst.ccs.gconsole.agent;

import java.util.List;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderInfo.Attribute;

/**
 * Encapsulates static properties and current status of a data channel.
 * Each channel has a central value (an {@code Object}) and optional attributes identified by string keys.
 * A channel is identified by a unique path (slash-separated string, with the subsystem name as the first element). 
 * <p>
 * Implementations should ensure that the path is used to establish channel equality and hash code.
 *
 * @author onoprien
 */
public interface AgentChannel {
    
    /**
     * Enumeration of standard attributes.
     */
    public interface Key {
        
        String VALUE = "value";
        
        String DESCR = Attribute.DESCRIPTION.getName();
        String UNITS = Attribute.UNITS.getName();
        String FORMAT = Attribute.FORMAT.getName();
        String PAGE = Attribute.PAGE.getName();
        String SECTION = "section";
        String ALERT_HIGH = Attribute.ALARMHI.getName();
        String ALERT_LOW = Attribute.ALARMLO.getName();
        
        String LOW_ALARM = "limitLo";
        String LOW_WARN = "dbandLo";
        String HIGH_ALARM = "limitHi";
        String HIGH_WARN = "dbandHi";
        
        String STATE = "_channel_state_";
        String TRENDING = "_key_";
        String TYPE = "_type_";
        
    }
    
// -- Getters : ----------------------------------------------------------------

    /**
     * Return the unique path of this channel.
     * @return Channel path in (agent name)/(local path) form.
     */
    public String getPath();
    
    /**
     * Returns the name of the {@code Agent} this channel belongs to.
     * @return Agent name.
     */
    public String getAgentName();
    
    /**
     * Returns the descriptor of the {@code Agent} this channel belongs to. 
     * @return Agent descriptor.
     */
    public AgentInfo getAgent();
    
    /**
     * Returns the path of this channel with the agent name and the following slash stripped.
     * Local paths are unique inside an agent.
     * @return Local path.
     */
    public String getLocalPath();
    
    /**
     * Returns the value of the channel attribute identified by the {@code key}.
     * 
     * @param <T> Type of the attribute value.
     * @param key The attribute is identified by the string returned by {@code toString()} method of the key object.
     *            If {@code key} is {@code null} or evaluates to {@link Key.VALUE}, the central value is returned.
     * @return Attribute value, or {@code null} if this channel does not have the specified attribute.
     */
    public <T> T get(Object key);
    
    /**
     * Returns the current central value of this channel.
     * @param <T> Type of the central value.
     * @return Current value.
     */
    public <T> T get();
    
    /**
     * Returns the list of keys of attributes this channel has.
     * @return List of existing attribute keys.
     */
    public List<String> getAttributes();
}
