package org.lsst.ccs.gconsole.plugins.dictionary;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.net.URL;
import java.util.concurrent.ExecutionException;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableModel;
import javax.swing.tree.DefaultTreeModel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.structs.TreeBranch;

/**
 *
 * @author emarin
 */
public class DictionaryTreeMainPanel extends JPanel implements ListSelectionListener, AgentPresenceListener, ActionListener {

   // private ModelSubsystemBusMaster busMaster;
    private final CommandListener commandListener;

    private final Logger logger = Logger.getLogger("org.lsst.ccs.plugin.jas3.dictionary");
    private CommandListPanel cmdList;

    private final AgentTable agentTable;

    private final DictionaryTreePanel dictionaryTree;
    private final ArgInputPanel argInputPanel;
    private final HTMLTextPane resultPane;
    private final JButton sendCmdButton;
    private final JButton refreshAgentsButton;
    private final JPanel topPane;
    private final JCheckBox [] checkBoxArray = new JCheckBox[AgentInfo.AgentType.values().length];

    //Optionally play with line styles.  Possible values are
    //"Angled" (the default), "Horizontal", and "None".
    private static final boolean playWithLineStyle = false;
    private static final String lineStyle = "Horizontal";

    //Optionally set the look and feel.
    DictionaryTreeMainPanel(CommandListener cmdListener) {
        super(new BorderLayout());
        commandListener = cmdListener;

        // Agent Table creation
        agentTable = new AgentTable();
        agentTable.getSelectionModel().addListSelectionListener(this);
        
        // Dictionary Tree creation
        dictionaryTree = new DictionaryTreePanel(commandListener);

        // Create a scroll pane and adds the subsystem list to it
        JScrollPane agentsView = new JScrollPane(agentTable);
        agentTable.setFillsViewportHeight(true);

        if (playWithLineStyle) {
            System.out.println("line style = " + lineStyle);
            dictionaryTree.putClientProperty("JTree.lineStyle", lineStyle);
        }

        //Create the scroll pane and add the dictionaryJTree to it. 
        JScrollPane treeView = new JScrollPane(dictionaryTree);

        // Create the command list viewing pane.
        cmdList = new CommandListPanel();
        dictionaryTree.addTreeSelectionListener(cmdList);
        //agentTable.getSelectionModel().addListSelectionListener(cmdList);

        cmdList.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                if (e.getClickCount() == 2) {
                    DictionaryCommand cmd = cmdList.getSelectedValue();
                    // Double click is only enabled for command without arguments
                    if (cmd.getArguments().length == 0) {
                        sendCommand();
                    }
                }
            }
        });

        JScrollPane cmdListView = new JScrollPane(cmdList);

        //Add the views to the TOP panel
        topPane = new JPanel(new GridLayout(1, 3));

        // Add subsystemview to the main Panel (not the top panel)
        JPanel leftPane = new JPanel(new BorderLayout());
        leftPane.setPreferredSize(new Dimension(300,800));

        // Refresh agents button
        URL url = this.getClass().getResource("/org/lsst/ccs/gconsole/plugins/dictionary/refresh_24dp.png");
        refreshAgentsButton = new JButton();
        refreshAgentsButton.setIcon(new ImageIcon(url));
        refreshAgentsButton.setToolTipText("Refresh");
        refreshAgentsButton.addActionListener(this);

        // JCheckBox panel
        JPanel checkBoxPanel = new JPanel(new GridLayout(4,2));
        for (AgentInfo.AgentType type : AgentInfo.AgentType.values()){
            checkBoxArray[type.ordinal()]= new JCheckBox(type.displayName());
            checkBoxArray[type.ordinal()].addItemListener(agentTable);
            checkBoxArray[type.ordinal()].setName(type.toString());
            checkBoxPanel.add(checkBoxArray[type.ordinal()]);
        }
        
        JPanel leftPanelBottom = new JPanel(new BorderLayout()) ;
        leftPanelBottom.add(checkBoxPanel,BorderLayout.CENTER);
        leftPanelBottom.add(refreshAgentsButton,BorderLayout.SOUTH);

        leftPane.add(agentsView, BorderLayout.CENTER);
        leftPane.add(leftPanelBottom, BorderLayout.SOUTH);

        add(leftPane, BorderLayout.WEST);

        // Add TreeView to the top panel
        topPane.add(treeView);

        //Add command list view to the top panel
        topPane.add(cmdListView);

        // Result Panel
        // create and add the argInputPanel to the top panel
        argInputPanel = new ArgInputPanel();
        cmdList.addListSelectionListener(argInputPanel);

        JPanel buttonPane = new JPanel();
        buttonPane.setLayout(new FlowLayout(FlowLayout.RIGHT));
        sendCmdButton = new JButton(new ImageIcon(this.getClass().getResource("/org/lsst/ccs/gconsole/plugins/dictionary/ic_send_black_24dp.png")));
        sendCmdButton.setText("Send");
        sendCmdButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand();

            }
        });

        buttonPane.add(sendCmdButton);
        add(buttonPane, BorderLayout.SOUTH);

        JPanel topPaneRightPanel = new JPanel(new BorderLayout());
        topPaneRightPanel.add(argInputPanel, BorderLayout.CENTER);
        topPaneRightPanel.add(buttonPane, BorderLayout.SOUTH);

        topPane.add(topPaneRightPanel);

        // Console Pane
        resultPane = new HTMLTextPane();
        JScrollPane resultPaneView = new JScrollPane(resultPane);

        // Split Panel creation
        JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        splitPane.setTopComponent(topPane);
        splitPane.setBottomComponent(resultPaneView);
        splitPane.setContinuousLayout(true);

        add(splitPane, BorderLayout.CENTER);
        
        // set the worker checkbox to true
        checkBoxArray[AgentInfo.AgentType.WORKER.ordinal()].setSelected(true);
    }
    
    // Agent selected has changed
    @Override
    public void valueChanged(ListSelectionEvent e) {
        dictionaryTree.setModel(new DefaultTreeModel(null));
        argInputPanel.clearForm();
        if(agentTable.getSelectedRow() >=0){ // If an agent is selected
            String selectedSubsystem = ((AgentInfo)agentTable.getValueAt(agentTable.getSelectedRow(), 0)).getName();
            new IsEMWorker(selectedSubsystem).execute();
            dictionaryTree.updateData(selectedSubsystem);
        } 
    }

    @Override
    public void connecting(AgentInfo agent) {
        SwingUtilities.invokeLater(new Runnable() {
            
            @Override
            public void run() {
                ((DefaultTableModel)agentTable.getModel()).addRow(new Object[]{agent});
            }
        });
    }
    
    @Override
    public void disconnecting(AgentInfo agent) {
        ((AgentDefaultTableModel)agentTable.getModel()).removeRow(agent);
    }

    public void sendCommand(){
        TreeBranch<DictionaryContext> destinationNode = (TreeBranch<DictionaryContext>) dictionaryTree.getLastSelectedPathComponent();
                String destination = destinationNode.toString();
                String rootNode = dictionaryTree.getModel().getRoot().toString();

                if (!destination.equals(rootNode)) {
                    destination = rootNode + "/" + destination;
                }

                String s = " ";
                DictionaryCommand cmd = cmdList.getSelectedValue();
                if (cmd != null) {
                    int n = cmd.getArguments().length;
                    for (int i = 0; i < n; i++) {
                        s += argInputPanel.getArgInput()[i].getText() + " ";
                    }
                    new SendCommandWorker(cmd.getCommandName() + s, destination).execute();
                }
    }

    /**
     * Refresh agents performed
     * @param e 
     */
    @Override
    public void actionPerformed(ActionEvent e) {
        new SwingWorker<Object[][], String>() {
            
            @Override
            protected Object[][] doInBackground() throws Exception {
                return commandListener.getConnectedAgents();
            }
            
            @Override
            public void done() {
                try {
                    (((AgentDefaultTableModel)agentTable.getModel())).update(get());
                } catch (InterruptedException ex) {
                    logger.error(ex);
                } catch (ExecutionException ex) {
                    logger.error(ex);
                }
                // Update Engineering Mode state
                if(agentTable.getSelectedRow() >=0){ // If an agent is selected
                    String selectedSubsystem = ((AgentInfo)agentTable.getValueAt(agentTable.getSelectedRow(), 0)).getName();
                    new IsEMWorker(selectedSubsystem).execute();
                } 
            }
        }.execute();
    }
    
    public class SendCommandWorker extends SwingWorker<String, String> {
        
        private String cmd;
        private String destination;

        public SendCommandWorker(String cmdName, String destination) {
            this.cmd = cmdName;
            this.destination = destination;
        }

        @Override
        protected String doInBackground() throws Exception {
            resultPane.insertHTML("invoke " + destination + " " + cmd +" ...");
            return commandListener.sendCommand(cmd, destination);
        }

        @Override
        public void done() {
            try {
                String result = get();
                if ((cmd.equals("switchToEngineeringMode")&& result.equals("true")) ||
                        (cmd.equals("switchToNormalMode") && result.equals("ok : VOID"))){
                        //resultPane.insertHTML(" #### cmd " + cmdName + " true ");
                        boolean isEM = cmd.equals("switchToEngineeringMode")?true:false;
                        //Update Tree Panel
                        dictionaryTree.updateRenderer(isEM);
                        //Update List Command Panel
                        cmdList.updateRenderer(isEM);
                }
                resultPane.insertHTML(result);
            } catch (InterruptedException ex) {
                logger.error(ex);
            } catch (ExecutionException ex) {
                logger.error(ex);
            }
        }
    }
    
    private class IsEMWorker extends SwingWorker<Boolean, String>{

        private final String destination;
        
        IsEMWorker(String destination){
            this.destination = destination;
        }
        @Override
        protected Boolean doInBackground() throws Exception {
            return commandListener.isInEngineeringMode(destination);
        }
        
        @Override
        public void done(){
            try {
                boolean result = get().booleanValue();
                dictionaryTree.updateRenderer(result);
                cmdList.updateRenderer(result);
            } catch (InterruptedException ex) {
                logger.error(ex);
            } catch (ExecutionException ex) {
                logger.error(ex);
            }
        }
    }
}