package org.lsst.ccs.gconsole.agent;

import java.time.Duration;
import java.time.Instant;
import java.util.*;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;
import org.lsst.ccs.messaging.CommandRejectedException;
import org.lsst.ccs.subsystem.monitor.data.MonitorChan;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonitorState;

/**
 * Handles status aggregator data for a specific {@code Agent}.
 * 
 * Implementation notes:<ul>
 * <li>
 * A handle exists for any channel present on the buses.
 * <li>
 * Unwatched agents will have an empty list of listeners. For such agents, all non-final fields are null.
 * <li>
 * When the agent transfers to the watched status, the following fields are initialized: {@code state}, {@code stateTime}.
 * The state is maintained from this point on, unless the list of listeners becomes empty again at some point.
 * The {@code channels} map is set to either {@code null} if no listener is interested in channels from
 * this agent, or to an empty list. In the letter case, the dictionary is requested.
 * The listeners receive "connect" event and the dictionary is requested at this time.
 * <li>
 * Once the dictionary is received, it is used to populate {@code channels}.
 * {@code rejectedPaths} remains {@code null} if there are no templates in listeners, otherwise it is initialized to an empty set.
 * The listeners receive "configuration" event at this time.
 * </ul>
 */
class AgentHandle {
    
// -- Fields : -----------------------------------------------------------------
    
    private final AgentInfo agentInfo;
    private final AgentStatusAggregator aggregator;
    private final ArrayList<ListenerHandle> listeners = new ArrayList<>(0);
    
    HashMap<String, String> state; // last recordered agent state
    Instant stateTime; // time stamp of the last status message from the agent (remote computer time)
    
    /* Map of inner paths to channels; null if there are no listeners interested in channels. */
    LinkedHashMap<String, MutableAgentChannel> channels;
    
    /* Cached rejections; null if there are no expected channels or templates in any of the listener handles */
    HashSet<String> rejectedPaths;

    volatile Instant dictTime; // time when the dictionary was last requested; null if never or already installed
    AgentChannelsDictionary dictionary;
    
    // Dirty stuff required to interpret legacy data:
    
    MonitorState monitorState; // last monitor state processed
    Instant monitorStateTime; // time stamp of the last monitor state (remote time)


// -- Life cycle : -------------------------------------------------------------
    
    /** Constructs an instance */
    AgentHandle(AgentInfo agentInfo, AgentStatusAggregator aggregator) {
        this.aggregator = aggregator;
        this.agentInfo = agentInfo;
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    boolean hasListeners() {
        return !listeners.isEmpty();
    }
    
    AgentInfo getAgent() {
        return agentInfo;
    }
    
    MutableAgentChannel getChannel(String innerPath) {
        return channels == null ? null : channels.get(innerPath);
    }
    
    /**
     * Returns a collection of watched channels. 
     * An empty collection is returned if there are no channels, regardless of whether there
     * are any templates that can result in channels added once they appear in status messages.
     */
    Collection<MutableAgentChannel> getChannels() {
        return channels == null ? Collections.emptyList() : channels.values();
    }
    
    
// -- Processing messages : ----------------------------------------------------
    
    void onConnect(StatusMessage message) {
        
        // Save state:
        
        state = new HashMap<>(message.getState().getAllStatesAsStrings());
        stateTime = message.getState().getLastModified();
        
        // Create handles for relevant listeners (channel dictionary is requested here if necessary):
        
        aggregator.listeners.forEach(globalHandle -> createLocalListenerHangle(globalHandle));
        listeners.trimToSize();
        
        // Notify listeners
        
        AgentStatusEvent event = new AgentStatusEvent(aggregator, agentInfo);
        listeners.forEach(lh -> lh.global.listener.connect(event));
        
    }
    
    void onConfiguration(Object data) {
        
        ArrayList<MutableAgentChannel> allChannels = new ArrayList<>();
        
        // Add monitoring channels: TEMPORARY

        if (data instanceof MonitorFullState) {
            MonitorFullState mfs = (MonitorFullState) data;
            List<MonitorChan> mChannels = mfs.getChannels();
            MutableAgentChannel[] chs = new MutableAgentChannel[mChannels.size()];
            String section = null;
            for (int i=0; i<chs.length; i++) {
                MonitorChan mc = mChannels.get(i);
                chs[i] = new MutableAgentChannel(agentInfo.getName() +"/"+ mc.getName(), agentInfo);
                chs[i].set(mc.getValue());
                chs[i].set(MonitorField.UNITS.name(), mc.getUnits());
                String descr = mc.getDescription();
                int ii = descr.indexOf("\\");
                if (ii != -1) {
                    section = descr.substring(0, ii);
                    descr = descr.substring(ii+1);
                }
                chs[i].set(MonitorField.DESCR.name(), descr);
                if (section != null) {
                    chs[i].set(MonitorField.SECTION_KEY, section);
                }
                chs[i].set(MonitorField.ALERT_HIGH.name(), mc.getHighAlarm());
                chs[i].set(MonitorField.ALERT_LOW.name(), mc.getLowAlarm());
                chs[i].set(MonitorField.HIGH_ALARM.name(), mc.getHighLimit());
                chs[i].set(MonitorField.HIGH_WARN.name(), mc.getHighWarning());
                chs[i].set(MonitorField.LOW_ALARM.name(), mc.getLowLimit());
                chs[i].set(MonitorField.LOW_WARN.name(), mc.getLowWarning());
                chs[i].set(MonitorField.ID_KEY, i);
                String page = mfs.getPages().get(mc.getPage());
                if (page != null) {
                    chs[i].set(MonitorField.PAGE_KEY, page);
                }
                chs[i].set(MonitorField.FORMAT_KEY, mc.getFormat());
                chs[i].set(MonitorField.STATE_KEY, AgentChannelState.packMonitorState(mfs.getMonitorState(), i));
            }
            monitorState = mfs.getMonitorState();
            monitorStateTime = Instant.MIN;
            allChannels.addAll(Arrays.asList(chs));
        } else {
            monitorStateTime = Instant.MAX;
        }
        
        // Add states: TEMPORARY
        
        for (Map.Entry<String,String> e : state.entrySet()) {
            MutableAgentChannel channel = new MutableAgentChannel(agentInfo.getName() +"/state/"+ e.getKey(), agentInfo);
            channel.set("state", true);
            channel.set(e.getValue());
            allChannels.add(channel);
        }
        
        // Process dictionary for each listener (includes listener notification)
        
        allChannels.trimToSize();
        dictionary = new AgentChannelsDictionary(allChannels);
        dictTime = null;
        listeners.forEach(lh -> lh.processDictionary());
        resetRejectedPaths();
    }
    
    void onMessage(StatusMessage mess) {
        
        // Message from an localAgent that is not being watched - ignore:
        
        if (listeners.isEmpty()) return;
        
        // Dictionary not yet available - request it:
        
        if (dictionary == null && channels != null) {
            requestDictionary();
        }
        
        // Detect status changes and accumulate into event:
        
        MutableAgentStatusEvent event = new MutableAgentStatusEvent(aggregator, mess);

        synchronized (this) { // keep lock while modifying data
        
            Instant messTimeStamp = Instant.ofEpochMilli(mess.getTimeStamp());
            Instant stateTimeStamp = mess.getState().getLastModified();
            
            // Check if this handle is interested in channel updates:
            
            if (channels == null || dictionary == null) {
                if (!stateTime.isAfter(stateTimeStamp)) {
                    state = new HashMap<>(mess.getState().getAllStatesAsStrings());
                    stateTime = mess.getState().getLastModified();
                }
                return;
            }

            // Update agent state:
            
            if (/*mess instanceof StatusStateChangeNotification && */!stateTime.isAfter(stateTimeStamp)) {
                Map<String, String> current = mess.getState().getAllStatesAsStrings();
                for (Map.Entry<String, String> e : current.entrySet()) {
                    String key = e.getKey();
                    String oldValue = state.get(key);
                    String newValue = e.getValue();
                    if (!Objects.equals(oldValue, newValue)) {
                        state.put(key, newValue);
                        String innerPath = "state/" + e.getKey();
                        MutableAgentChannel channel = channels.get(innerPath);
                        if (channel == null) {
                            channel = createChannel(innerPath);
                            if (channel != null) {
                                channel.set("state", true);
                                channel.set(newValue);
                                event.addAddedChannel(channel);
                            }
                        } else {
                            boolean changed = channel.set(newValue);
                            if (changed) {
                                event.addChange(channel, AgentChannel.VALUE_KEY);
                            }
                        }
                    }
                }
                stateTime = stateTimeStamp;
            }
            
            //  Update trending data :
        
            Object o = mess.getEncodedData();
            if (o instanceof KeyValueDataList) {
                KeyValueDataList encodedData = (KeyValueDataList) o;
                for (KeyValueData d : encodedData) {
                    KeyValueData.KeyValueDataType type = d.getType();
                    String attrKey = null;
                    String innerPath = null;
                    switch (type) {
                        case KeyValueTrendingData:
                            innerPath = d.getKey();
                            attrKey = AgentChannel.VALUE_KEY;
                            break;
                        case KeyValueMetaData:
                            innerPath = d.getKey();
                            int lastIndex = innerPath.lastIndexOf('/');
                            attrKey = innerPath.substring(lastIndex + 1);
                            innerPath = innerPath.substring(0, lastIndex);
                            break;
                    }
                    if (attrKey != null) {
                        Object value = d.getValue();
                        MutableAgentChannel channel = channels.get(innerPath);
                        if (channel == null) {
                            channel = createChannel(innerPath);
                            if (channel != null) {
                                channel.set(attrKey, value);
                                event.addAddedChannel(channel);
                            }
                        } else {
                            boolean changed = channel.set(attrKey, value);
                            if (changed) {
                                event.addChange(channel, attrKey);
                            }
                        }
                    }
                }
            }
        
        // Update monitored channels flags: // TEMPORARY
            
            if (mess instanceof StatusSubsystemData && !monitorStateTime.isAfter(messTimeStamp)) {
                StatusSubsystemData sd = (StatusSubsystemData) mess;
                String key = sd.getDataKey();
                if (key.equals(MonitorState.KEY)) {
                    try {
                        MonitorState current = (MonitorState) sd.getSubsystemData().getValue();
                        MonitorState prev = monitorState;
                        prev.getGoodChans().xor(current.getGoodChans());
                        BitSet change = (BitSet) prev.getGoodChans().clone();
                        prev.getOnlineChans().xor(current.getOnlineChans());
                        change.or(prev.getOnlineChans());
                        prev.getLowLimitChange().xor(current.getLowLimitChange());
                        change.or(prev.getLowLimitChange());
//                        int size = current.getHighLimitChange().cardinality();
                        prev.getHighLimitChange().xor(current.getHighLimitChange());
//                        size = prev.getHighLimitChange().cardinality();
                        change.or(prev.getHighLimitChange());
//                        size = change.cardinality();
//                        System.out.println(size);
                        for (int i = change.nextSetBit(0); i != -1; i = change.nextSetBit(i+1)) {
                            String path = dictionary.getAll().get(i).getLocalPath();
                            MutableAgentChannel channel = channels.get(path);
                            if (channel != null) {
                                channel.set(MonitorField.STATE_KEY, AgentChannelState.packMonitorState(current, i));
                                event.addChange(channel, MonitorField.STATE_KEY);
                            }
                        }
                        monitorState = current;
                    } catch (RuntimeException x) {
                        System.out.println("Failed to get monitor state: " + x); // FIXME
                    }
                }
            }

        } // finished mofifying data, release lock on localAgent handle
        
        // Recompile the set of rejected paths if necessary
        
        if (!(event.getAddedChannels().isEmpty() && event.getRemovedChannels().isEmpty())) {
            resetRejectedPaths();
        }
        
        // Notify listeners:
        
        if (!(event.getStatusChanges().isEmpty() && event.getAddedChannels().isEmpty() && event.getRemovedChannels().isEmpty())) {
            listeners.forEach(lh -> {
                AgentStatusEvent e = event.filter(lh.paths);
                if (!e.isEmpty()) lh.global.listener.statusChanged(e);
            });
        }
        
    }
    
    void onDisconnect(StatusMessage mess) {
        List<AgentChannel> removed = channels == null ? Collections.emptyList() : new ArrayList<>(channels.values());
        AgentStatusEvent event = new AgentStatusEvent(aggregator, agentInfo, Collections.emptyMap(), Collections.emptyList(), removed);
        listeners.forEach(lh -> {
            AgentStatusEvent e = event.filter(lh.paths);
            lh.global.listener.disconnect(e);
        });
    }
    
    
// -- Adding/removing listeners : ----------------------------------------------
    
    void addListener(AgentStatusAggregator.GlobalListenerHandle global) {
        ListenerHandle localHandle = createLocalListenerHangle(global);
        if (localHandle != null) {
            global.listener.connect(new AgentStatusEvent(aggregator, agentInfo));
            if (localHandle.paths != null) {
                if (dictionary == null) {
                    requestDictionary();
                } else {
                    localHandle.processDictionary();
                    resetRejectedPaths();
                }
            }
        }
    }
    
    void removeListener(AgentStatusAggregator.GlobalListenerHandle global) {
        
        // remove local listener handles
        
        Iterator<ListenerHandle> it = listeners.iterator();
        boolean changed = false;
        while (it.hasNext()) {
            ListenerHandle local = it.next();
            if (local.global == global) {
                it.remove();
                changed = true;
            }
        }
        
        // if any handles have been removed, update {@code channels} and {@code rejectedPaths}.
        
        if (changed) {
            HashSet<String> paths = null;
            for (ListenerHandle lh : listeners) {
                if (lh.paths != null) {
                    if (paths == null) paths = new HashSet<>();
                    paths.addAll(lh.paths);
                }
            }
            synchronized (this) {
                if (paths == null) {
                    channels = null;
                } else {
                    Iterator<Map.Entry<String, MutableAgentChannel>> iter = channels.entrySet().iterator();
                    while (iter.hasNext()) {
                        Map.Entry<String, MutableAgentChannel> e = iter.next();
                        if (!paths.contains(e.getKey())) {
                            iter.remove();
                        }
                    }
                }
            }
            resetRejectedPaths();
        }
    }
    
    
// -- Requesting and processing dictionary: ------------------------------------

    /*
     * Dirty monitoring-specific code goes here - to be removed once we have generic
     * dictionary retrieval API.
     */
    void requestDictionary() {
        Duration timeout = Duration.ofSeconds(60);
        if (dictionary == null && (dictTime == null || Instant.now().minus(timeout).isAfter(dictTime))) {
            dictTime = Instant.now();
            CommandRequest cmd = new CommandRequest(agentInfo.getName(), "getMonitorFullState");
            Runnable task = () -> {
                try {
                    Object result = null;
                    try {
                        result = aggregator.getMessenger().sendSynchronousCommand(cmd, timeout);
                    } catch (CommandRejectedException x) {
                        if (!x.getCommandNack().getReason().contains("Could not find command")) {
                            throw x;
                        }
                    }
                    Object dictData = result;
                    dictTime = Instant.MAX;
                    aggregator.exec.execute(() -> {
                        if (dictionary == null && !listeners.isEmpty()) {
                            onConfiguration(dictData);
                        }
                    });
                } catch (Exception x) {
                }

            };
            Thread thread = new Thread(task, "Fetching channels dictionary from " + agentInfo.getName());
            thread.setDaemon(true);
            thread.run();
        }
    }

    
// -- Local methods : ----------------------------------------------------------

    /**
     * Called when data is encountered in status message that is not associated with any existing channel in AgentHandle.channels.
     * Returns {@code null} if there are no listeners interested in that channel, or a newly created channel.
     */
    MutableAgentChannel createChannel(String innerPath) {
        if (rejectedPaths == null || rejectedPaths.contains(innerPath)) return null;
        boolean accept = false;
        for (ListenerHandle lh : listeners) {
            if (lh.paths != null) {
                if (lh.paths.contains(innerPath)) {
                    accept = true;
                } else {
                    for (String template : lh.templates) {
                        if (innerPath.startsWith(template)) {
                            accept = true;
                            lh.paths.add(innerPath);
                            break;
                        }
                    }
                }
            }
        }
        if (accept) {
            MutableAgentChannel channel = new MutableAgentChannel(agentInfo.getName() + "/" + innerPath, agentInfo);
            channels.put(innerPath, channel);
            return channel;
        } else {
            rejectedPaths.add(innerPath);
            return null;
        }
    }

    /**
     * Creates local listener handle and requests a dictionary if necessary.
     * Adds the new listener to the list maintained by the agent handle.
     * {@code paths} is set to null if this listener is only interested in connects and disconnects;
     * otherwise, it is set to an empty list (to be filled when the dictionary is processed).
     */
    private ListenerHandle createLocalListenerHangle(AgentStatusAggregator.GlobalListenerHandle globalHandle) {
        
        ListenerHandle localHandle = null;        
        if (globalHandle.agents == null || globalHandle.agents.contains(agentInfo.getName())) {
            localHandle = new ListenerHandle(globalHandle);
            if (globalHandle.channels == null) {
                localHandle.paths = new HashSet<>();
            } else {
                String prefix = agentInfo.getName() + "/";
                for (String s : globalHandle.channels) {
                    if (isSelector(s)) {
                        String a = parseSelector(s).get("agent.name");
                        if (a == null || agentInfo.getName().equals(a)) {
                            localHandle.paths = new HashSet<>();
                            break;
                        }
                    } else if (s.startsWith(prefix) || s.startsWith("/")) {
                            localHandle.paths = new HashSet<>();
                            break;
                    }
                }
            }
        } else {
            if (globalHandle.channels != null) {
                String prefix = agentInfo.getName() + "/";
                for (String s : globalHandle.channels) {
                    if (isSelector(s)) {
                        String a = parseSelector(s).get("agent.name");
                        if (agentInfo.getName().equals(a)) {
                            localHandle = new ListenerHandle(globalHandle);
                            localHandle.paths = new HashSet<>();
                            break;
                        }
                    } else if (s.startsWith(prefix)) {
                        localHandle = new ListenerHandle(globalHandle);
                        localHandle.paths = new HashSet<>();
                        break;
                    }
                }
            }
        }
        
        if (localHandle != null) {
            listeners.add(localHandle);
            if (localHandle.paths != null) {
                if (channels == null) channels = new LinkedHashMap<>();
                if (dictionary == null) requestDictionary();
            }
        }

        return localHandle;
    }
    
    private void resetRejectedPaths() {
        rejectedPaths = null;
        if (channels == null) return;
        for (ListenerHandle lh : listeners) {
            if (lh.paths != null) {
                if (lh.templates.isEmpty()) {
                    for (String localPath : lh.paths) {
                        if (!channels.containsKey(localPath)) {
                            rejectedPaths = new HashSet<>();
                            return;
                        }
                    }
                } else {
                    rejectedPaths = new HashSet<>();
                    return;
                }
            }
        }
    }
    
    
// -- Local classes : ----------------------------------------------------------

    /**
     * Agent-specific handle for a listener.
     * Created by {@code AgentHandle.createLocalListenerHangle(AgentStatusAggregator.GlobalListenerHandle)}.
     */
    private class ListenerHandle {

        /** Global listener handle. */
        AgentStatusAggregator.GlobalListenerHandle global;
        
        /**
         * Explicit inner paths of acceptable channels;
         * {@code null} if this listener is only interested in agent connects and disconnects.
         */
        Set<String> paths;
        
        /** Path templates; may be empty, never {@code null}. */
        List<String> templates; // inner path templates
        
        ListenerHandle(AgentStatusAggregator.GlobalListenerHandle global) {
            this.global = global;
        }
        
        void processDictionary() {
            
            if (paths == null) return;
            
            // Set paths and templates
            
            templates = new ArrayList<>();
            boolean inAgentList = global.agents == null || global.agents.contains(agentInfo.getName());
            String prefix = agentInfo.getName() + "/";
            for (String s : global.channels) {
                if (isSelector(s)) {
                    for (MutableAgentChannel channel : dictionary.getAll()) {
                        if (acceptSelector(channel, s)) {
                            paths.add(channel.getLocalPath());
                        }
                    }
                } else {
                    if (s.startsWith(prefix)) {
                        s = s.substring(prefix.length());
                    } else if (inAgentList && s.startsWith("/")) {
                        s = s.substring(1);
                    } else {
                        continue;
                    }
                    if (s.endsWith("/") || s.isEmpty()) {
                        templates.add(s);
                    } else {
                        paths.add(s);
                    }
                }
            }
            
            // Update channel map in agent handle
            
            ArrayList<AgentChannel> added = new ArrayList<>();
            synchronized (AgentHandle.this) {
                for (MutableAgentChannel channel : dictionary.getAll()) {
                    String localPath = channel.getLocalPath();
                    if (paths.contains(localPath)) {
                        channels.put(localPath, channel);
                        added.add(channel);
                    } else {
                        for (String template : templates) {
                            if (localPath.startsWith(template)) {
                                paths.add(localPath);
                                channels.put(localPath, channel);
                                added.add(channel);
                                break;
                            }
                        }
                    }
                }
            }
            
            // Clean up
            
            if (templates.isEmpty()) {
                templates = Collections.emptyList();
            } else {
                ((ArrayList)templates).trimToSize();
            }
            paths = new HashSet<>(paths);
            
            // Notify the listener
            
            AgentStatusEvent event = new AgentStatusEvent(aggregator, agentInfo, null, added, null);
            global.listener.configure(event);
        }
        
    }
    
// -- Utility methods : --------------------------------------------------------
    
    /** Returns {@code true} if the specified string is in a channel selector format (as opposed to explicit path or template). */
    static boolean isSelector(String s) {
        if (s.contains("=")) return true;
        return ! s.contains("/");
    }
    
    /**
     * Applies selector string to a channel.
     * Selector is in
     * {@code [localAgent.name=value&][localAgent.type=value&][localAgent.key[=value]&…&localAgent.key[=value]&][key[=value]&…&key[=value]]}
     * format.
     * 
     * @param channel Channel to test.
     * @param selector Selector.
     * @return {@code true} is the channel satisfies the selector.
     */
    static boolean acceptSelector(AgentChannel channel, String selector) {
        for (String condition : selector.split("&")) {
            String[] ss = condition.split("=");
            String key = ss[0];
            Object value;
            if (key.startsWith("agent.")) {
                key = key.substring("agent.".length());
                switch (key) {
                    case "name":
                        value = channel.getAgentName(); break;
                    case "type":
                        value = channel.getAgent().getType(); break;
                    default:
                        value = channel.get(key);
                }
            } else {
                value = channel.get(key);
            }
            if (ss.length == 1) {
                if (value == null) return false;
            } else {
                if (value == null) {
                    if (!"null".equals(ss[1])) return false;
                } else {
                    if (!value.toString().equals(ss[1])) return false;
                }
            }
        
        }
        return true;
    }
    
    static Map<String,String> parseSelector(String selector) {
        String[] ss = selector.split("&");
        HashMap<String,String> out = new LinkedHashMap<>(ss.length*2);
        for (String s : ss) {
            String[] tt = s.split("=");
            out.put(tt[0], tt.length == 1 ? null : tt[1]);
        }
        return out;
    }
    
    static boolean matchTemplate(String path, String template) {
        if (template.startsWith("/")) {
            path = path.substring(path.indexOf("/"));
        }
        return template.endsWith("/") ? path.startsWith(template) : path.equals(template);
    }
    
    static boolean matchLocalTemplate(String localPath, String localTemplate) {
        if (localTemplate.isEmpty()) return true;
        return localTemplate.endsWith("/") ? localPath.startsWith(localTemplate) : localPath.equals(localTemplate);
    }
    
    

}
