package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.beans.Transient;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.AbstractAction;
import javax.swing.AbstractCellEditor;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeCellEditor;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.subsystem.monitor.ui.tree.MonitorDisplay;

/**
 * {@link MonitorView} that displays monitoring data as a tree of tables.
 * All access to instances of this class, including construction, should happen on the EDT.
 *
 * @author onoprien
 */
@ConsoleLookup(id="org.lsst.ccs.gconsole.plugins.monitor.MonitorView",
               name="Tree View",
               path="Built-In/Tree",
               description="Monitoring data view that displays its data in a tree of tables.")
public class TreeView extends AbstractMonitorView {

// -- Fields : -----------------------------------------------------------------
    
    private final InterNode root;
    DefaultTreeModel treeModel;
    private final Tree tree;
    private final JScrollPane scrollPane;

    private List<MonitorField> fields = Arrays.asList(MonitorField.VALUE, MonitorField.UNITS, MonitorField.LOW_ALARM, MonitorField.ALERT_LOW, MonitorField.HIGH_ALARM, MonitorField.ALERT_HIGH, MonitorField.DESCR);
    private List<MonitorField> compactFields = Arrays.asList(MonitorField.VALUE);

// -- Life cycle : -------------------------------------------------------------
    
    public TreeView() {
        root = new InterNode("");
        treeModel = new DefaultTreeModel(root);
        tree = new Tree(treeModel);
        scrollPane = new JScrollPane(tree);
        descriptor = new Descriptor();
        root.init(Collections.emptyList());        
    }

    @Override
    protected void resetChannels() {
        updateDescriptor(root);
        root.clear();
        root.init(path2data.entrySet());
        treeModel.reload();
        root.restoreExpansion();
    }

// -- Getters and setters : ----------------------------------------------------
    
    /**
     * Returns the graphical component maintained by this view.
     * @return Graphical component to be displayed by the GUI.
     */
    @Override
    public JComponent getPanel() {
        return scrollPane;
    }

    /**
     * Called to set a filter on this view.
     * Currently, the following is done in addition to saving e reference to the filter:<ul>
     * <li>Set monitor fields displayed by this view.
     * </ul>
     * @param filter Filter to use.
     */
    @Override
    public void setFilter(AgentChannelsFilter filter) {
        super.setFilter(filter);
        List<String> ff = filter.getFields(true);
        if (ff != null) {
            compactFields = new ArrayList<>(ff.size());
            for (String field : ff) {
                compactFields.add(MonitorField.getInstance(field));
            }
        }
        ff = filter.getFields(false);
        if (ff != null) {
            fields = new ArrayList<>(ff.size());
            for (String field : ff) {
                fields.add(MonitorField.getInstance(field));
            }
        }
    }
    
    
// -- Customized JTree : -------------------------------------------------------

    class Tree extends JTree implements MonitorDisplay {
        
        Tree(TreeModel model) {
            super(model);
            setRootVisible(true);
            setShowsRootHandles(true);
            setRowHeight(0);
            setCellRenderer(new TreeRenderer(false));
            setCellEditor(new TreeEditor());
            addMouseMotionListener(new MouseMotionListener() {
                @Override
                public void mouseMoved(MouseEvent e) {
                    if (tree.getRowForLocation(e.getX(), e.getY()) != -1) {
                        tree.startEditingAtPath(tree.getPathForLocation(e.getX(), e.getY()));
                    }
                }
                @Override
                public void mouseDragged(MouseEvent e) {
                }
            });
            setEditable(true);
            getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        } 
        
        @Override
        public void saveData(OutputStream out, String mimeType) {
            List<AgentChannel> channels = path2data.values().stream().map(handle -> handle.getChannel()).filter(channel -> channel != null).collect(Collectors.toList());
            LsstMonitorPlugin.saveData(out, mimeType, channels, fields);
        }
        
    }

    class TreeRenderer extends DefaultTreeCellRenderer {

        private final boolean editable;

        TreeRenderer(boolean editable) {
            this.editable = editable;
        }

        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            if (value instanceof InterNode) {
                Component c = super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
                return c;
            } else if (value instanceof DisplayNode) {
                DisplayNode node = (DisplayNode) value;
                return editable ? node.getEditorComponent() : node.getRendererComponent();
            } else {
                return super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            }
        }

    }

    class TreeEditor extends AbstractCellEditor implements TreeCellEditor {

        private final TreeRenderer renderer;
        private Object editorValue;
        
        private final JPopupMenu popup;
        private final Action compactAct, sectionAct;
        private final EnumMap<InterNodeDescriptor.DisplayMode, DisplayModeAction> modeAct;
        private final EnumMap<InterNodeDescriptor.Sort, AbstractAction> sortAct;

        public TreeEditor() {
            
            renderer = new TreeRenderer(true);
            
            popup = new JPopupMenu();
            JMenu menu = new JMenu("Display mode...");
            JMenuItem menuItem;
            
            modeAct = new EnumMap<>(InterNodeDescriptor.DisplayMode.class);
            for (InterNodeDescriptor.DisplayMode mode : InterNodeDescriptor.DisplayMode.values()) {
                DisplayModeAction act = new DisplayModeAction(mode) {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        if (editorValue instanceof InterNode) {
                            InterNode node = (InterNode) editorValue;
                            modeAct.values().forEach( act -> act.putValue(AbstractAction.SELECTED_KEY, act.equals(this)));
                            node.getDescriptor().setDisplayMode(mode);
                            node.rebuild();
                        }
                    }
                };
                modeAct.put(mode, act);
                menuItem = new JCheckBoxMenuItem(act);
                menu.add(menuItem);
            }
            popup.add(menu);
            
            menu = new JMenu("Order...");
            sortAct = new EnumMap<>(InterNodeDescriptor.Sort.class);
            for (InterNodeDescriptor.Sort mode : InterNodeDescriptor.Sort.values()) {
                AbstractAction act = new AbstractAction(mode.toString()) {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        if (editorValue instanceof InterNode) {
                            InterNode node = (InterNode) editorValue;
                            InterNodeDescriptor.Sort selectedMode = null;
                            for (Map.Entry<InterNodeDescriptor.Sort, AbstractAction> ee : sortAct.entrySet()) {
                                AbstractAction action = ee.getValue();
                                boolean isChosen = (action == this);
                                action.putValue(AbstractAction.SELECTED_KEY, isChosen);
                                if (isChosen) selectedMode = ee.getKey();
                            }
                            node.getDescriptor().setSort(selectedMode);
                            node.rebuild();
                        }
                    }
                };
                sortAct.put(mode, act);
                menuItem = new JCheckBoxMenuItem(act);
                menu.add(menuItem);
            }
            popup.add(menu);
            
            compactAct = new AbstractAction("Compact") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    if (editorValue instanceof InterNode) {
                        InterNode node = (InterNode) editorValue;
                        Boolean v = (Boolean) getValue(AbstractAction.SELECTED_KEY);
                        node.getDescriptor().setCompact(Boolean.TRUE.equals(v));
                        node.rebuild();
                    }
                }
            };
            menuItem = new JCheckBoxMenuItem(compactAct);
            popup.add(menuItem);
            sectionAct = new AbstractAction("Sectioned") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    if (editorValue instanceof InterNode) {
                        InterNode node = (InterNode) editorValue;
                        Boolean v = (Boolean) getValue(AbstractAction.SELECTED_KEY);
                        node.getDescriptor().setSection(Boolean.TRUE.equals(v));
                        node.rebuild();
                    }
                }
            };
            menuItem = new JCheckBoxMenuItem(sectionAct);
            popup.add(menuItem);
            renderer.addMouseListener(new MouseAdapter() {
                public void mousePressed(MouseEvent e) {
                    maybeShowPopup(e);
                }
                public void mouseReleased(MouseEvent e) {
                    maybeShowPopup(e);
                }
                private void maybeShowPopup(MouseEvent e) {
                    if (e.isPopupTrigger()) {
                        popup.show(e.getComponent(), e.getX(), e.getY());
                    }
                }
            });

        }

        @Override
        public Component getTreeCellEditorComponent(JTree tree, Object value, boolean isSelected, boolean expanded, boolean leaf, int row) {
            editorValue = value;
            if (editorValue instanceof InterNode) {
                InterNode node = (InterNode) editorValue;
                compactAct.putValue(AbstractAction.SELECTED_KEY, node.getDescriptor().isCompact());
                sectionAct.putValue(AbstractAction.SELECTED_KEY, node.getDescriptor().isSection());
                for (Map.Entry<InterNodeDescriptor.DisplayMode, DisplayModeAction> e : modeAct.entrySet()) {
                    e.getValue().putValue(AbstractAction.SELECTED_KEY, e.getKey().equals(node.getDescriptor().getDisplayMode()));
                }
                for (Map.Entry<InterNodeDescriptor.Sort, AbstractAction> e : sortAct.entrySet()) {
                    e.getValue().putValue(AbstractAction.SELECTED_KEY, e.getKey().equals(node.getDescriptor().getSort()));
                }
            }
            return renderer.getTreeCellRendererComponent(tree, value, isSelected, expanded, leaf, row, true);
        }

        /** At this point the component is added to the tree (undocumented!) but tree's internal cleanup might not yet be done. */
        @Override
        public boolean shouldSelectCell(EventObject anEvent) {
            return false;
        }

        @Override
        public Object getCellEditorValue() {
            return editorValue;
        }
    }
    
    abstract static private class DisplayModeAction extends AbstractAction {
        private InterNodeDescriptor.DisplayMode mode;
        DisplayModeAction(InterNodeDescriptor.DisplayMode mode) {
            super(mode.toString());
            this.mode = mode;
        }
    }


// -- Tree node classes : ------------------------------------------------------
    
    private class Node extends DefaultMutableTreeNode {
        
        Node() {
        }
        
        Node(String name) {
            super(name);
        }
        
        void clear() {
            Enumeration<Node> en = children();
            while (en.hasMoreElements()) {
                en.nextElement().clear();
            }
            removeAllChildren();
        }
        
        Serializable save() {
            return null;
        }
        
        void restore(Serializable storageBean) {
        }
        
        protected String getPathString() {
            StringJoiner sj = new StringJoiner("/");
            for (TreeNode node : getPath()) {
                if (node != root) {
                    sj.add(node.toString());
                }
            }
            return sj.toString();
        }
        
    }
    
    private class InterNode extends Node {
        
        // Fields:
        
        private InterNodeDescriptor desc;        
        private Collection<Map.Entry<String,ChannelHandle>> data;
       
        // Life cycle:
        
        InterNode(String name) {
            super(name);
        }
        
        void init(Collection<Map.Entry<String,ChannelHandle>> data) {
            
            this.data = data;
            if (desc == null) {
                String pathKey = getPathString();
                TreeMap<String, Serializable> nodes = TreeView.this.getDescriptor().getNodes();
                Serializable sz = nodes == null ? null : nodes.get(pathKey);
                if (sz == null || !(sz instanceof InterNodeDescriptor)) {
                    desc = new InterNodeDescriptor();
                } else {
                    desc = (InterNodeDescriptor) sz;
                }
            }
            
            // preprocess data: make sections, sort, etc.
            
            ArrayList<Map.Entry<String,ChannelHandle>> processedData;
            if (desc.isSection()) {
                processedData = new ArrayList<>(data.size());
                for (Map.Entry<String,ChannelHandle> e : data) {
                    String processedKey = e.getKey().replaceFirst("/+", "//");
                    processedData.add(new AbstractMap.SimpleImmutableEntry<>(processedKey, e.getValue()));
                }
            } else {
                processedData = new ArrayList<>(data);
            }
            
            InterNodeDescriptor.Sort sortAlgorithm = desc.getSort();
            if (!InterNodeDescriptor.Sort.NONE.equals(sortAlgorithm)) {
                processedData.sort(sortAlgorithm);
            }
            
            // create child nodes
            
            if (desc.isFlat()) {  // everything goes into a single SectionedTable
                
                TreeMap<String, Serializable> nodes = TreeView.this.getDescriptor().getNodes();
                String key = getPathString();
                if (!key.isEmpty()) key += "/";
                key += SectionedTable.class.getName();
                Serializable disDesc = nodes == null ? null : nodes.get(key);
                SectionedTable.Descriptor descriptor = (!(disDesc instanceof SectionedTable.Descriptor)) ? null : (SectionedTable.Descriptor)disDesc;
                SectionedTable tableModel = SectionedTable.getInstance(processedData, desc.isCompact() ? compactFields : fields, descriptor);
                DisplayNode dNode = new MonitorTableNode(tableModel);
                tableModel.setListener(dNode);
                add(dNode);
                
            } else {
                
                ArrayList<Map.Entry<String,ChannelHandle>> leaves = new ArrayList<>(processedData.size());
                ArrayList<Map.Entry<String,ChannelHandle>> mesh = new ArrayList<>(processedData.size());
                LinkedHashMap<String, ArrayList<Map.Entry<String,ChannelHandle>>> branches = new LinkedHashMap<>();
                
                for (Map.Entry<String,ChannelHandle> e : processedData) {
                    String[] pathSegments = e.getKey().split("/+");
                    if (pathSegments.length < 2) {
                        leaves.add(e);
                    } else if (desc.isMesh() && (pathSegments.length == 2 || pathSegments.length == 3)) {
                        if (desc.isFlip()) {
                            String dp = pathSegments.length == 2 ? pathSegments[1]+"/"+pathSegments[0] : pathSegments[0]+"/"+pathSegments[2]+"/"+pathSegments[1];
                            e = new AbstractMap.SimpleImmutableEntry<>(dp, e.getValue());
                        }
                        mesh.add(e);
                    } else {
                        ArrayList<Map.Entry<String,ChannelHandle>> branch = branches.get(pathSegments[0]);
                        if (branch == null) {
                            branch = new ArrayList<>();
                            branches.put(pathSegments[0], branch);
                        }
                        StringJoiner sj = new StringJoiner("/");
                        for (int i=1; i<pathSegments.length; i++) {
                            sj.add(pathSegments[i]);
                        }
                        branch.add(new AbstractMap.SimpleImmutableEntry<>(sj.toString(), e.getValue()));
                    }
                }
                
                if (!leaves.isEmpty()) {
                    SectionedTable tableModel = SectionedTable.getInstance(leaves, desc.isCompact() ? compactFields : fields, null);
                    DisplayNode dNode = new MonitorTableNode(tableModel);
                    tableModel.setListener(dNode);
                    add(dNode);
                }
                
                if (!mesh.isEmpty()) {
                    MeshTable tableModel = MeshTable.getInstance(mesh, desc.isCompact() ? compactFields : fields);
                    DisplayNode dNode = new MonitorTableNode(tableModel);
                    tableModel.setListener(dNode);
                    add(dNode);
                }
                
                if (!branches.isEmpty()) {
                    for (Map.Entry<String, ArrayList<Map.Entry<String,ChannelHandle>>> e : branches.entrySet()) {
                        InterNode child = new InterNode(e.getKey());
                        add(child);
                        e.getValue().trimToSize();
                        child.init(e.getValue());
                    }
                }
               
            }
            
        }
        
        protected void rebuild() {
            if (data != null && !data.isEmpty()) {
                updateDescriptor(this);
                Collection<Map.Entry<String,ChannelHandle>> data = this.data;
                clear();
                init(data);
                treeModel.nodeStructureChanged(this);
                restoreExpansion();
                tree.expandPath(new TreePath(getPath()));
            }
        }
        
        @Override
        void clear() {
            super.clear();
            data = null;
        }
        
        @Override
        InterNodeDescriptor save() {
            InterNodeDescriptor descriptor = new InterNodeDescriptor(desc);
            descriptor.setExpanded(tree.isExpanded(new TreePath(getPath())));
            return descriptor;
        }
        
        void restoreExpansion() {
            Enumeration<Node> en = depthFirstEnumeration();
            while (en.hasMoreElements()) {
                Node node = en.nextElement();
                if (node instanceof InterNode) {
                    InterNode n = (InterNode) node;
                    if (n.desc.isExpanded()) {
                        tree.expandPath(new TreePath(n.getPath()));
                    }
                }
            }
        }
        
        // Setters/getters :
        
        public InterNodeDescriptor getDescriptor() {
            return desc;
        }

        // Local methods :
        
        Collection<Map.Entry<String,ChannelHandle>> getData() {
            return data;
        }
        
    }
    
    public static class InterNodeDescriptor implements Serializable {
        
        public static enum DisplayMode {TREE, FLAT, MESH, FLIPPED_MESH}
        
        /**
         * Enumeration of sorting algorithms supported by {@code InterNode}.
         */
        public static enum Sort implements Comparator<Map.Entry<String,ChannelHandle>> {
            
            /** No sorting - channels are displayed or passed to children in the order in which they were supplied by the parent. */
            NONE {
                @Override
                public int compare(Map.Entry<String, ChannelHandle> o1, Map.Entry<String, ChannelHandle> o2) {
                    throw new UnsupportedOperationException("This value indicates no sorting");
                }
            },
            
            /** Alphabetic sorting by display path. */
            ALPHABETIC {
                @Override
                public int compare(Map.Entry<String, ChannelHandle> e1, Map.Entry<String, ChannelHandle> e2) {
                    return e1.getKey().compareTo(e2.getKey());
                }
            };

            @Override
            abstract public int compare(Map.Entry<String, ChannelHandle> o1, Map.Entry<String, ChannelHandle> o2);
        }
        
        static final int FLAT = 0b1;
        static final int MESH = 0b10;
        static final int FLIP = 0b100;
        static final int COMPACT = 0b1000;
        static final int SECTION = 0b10000;
        static final int EXPANDED = 0b100000;
        static final int SORT = 0b1000000;

        private int flags;
        
        public InterNodeDescriptor() {
        }
        
        public InterNodeDescriptor(InterNodeDescriptor other) {
            flags = other.flags;
        }
        
        @Transient
        public boolean isDefault() {
            return flags == 0;
        }

        public int getFlags() {
            return flags;
        }
        
        @Transient
        public DisplayMode getDisplayMode() {
            if (isFlat()) {
                return DisplayMode.FLAT;
            } else if (isMesh()) {
                return isFlip() ? DisplayMode.FLIPPED_MESH : DisplayMode.MESH;
            } else {
                return DisplayMode.TREE;
            }
        }

        @Transient
        public boolean isFlat() {
            return (flags & FLAT) > 0;
        }

        @Transient
        public boolean isMesh() {
            return (flags & MESH) > 0;
        }

        @Transient
        public boolean isFlip() {
            return (flags & FLIP) > 0;
        }

        @Transient
        public boolean isCompact() {
            return (flags & COMPACT) > 0;
        }

        @Transient
        public boolean isSection() {
            return (flags & SECTION) > 0;
        }
        
        @Transient
        public boolean isExpanded() {
            return (flags & EXPANDED) > 0;
        }
        
        @Transient
        public Sort getSort() {
            return (flags & SORT) > 0 ? Sort.ALPHABETIC : Sort.NONE;
        }

        public void setFlags(int flags) {
            this.flags = flags;
        }
        
        @Transient
        public void setDisplayMode(DisplayMode value) {
            switch (value) {
                case TREE:
                    setFlat(false);
                    setMesh(false);
                    setFlip(false);
                    break;
                case FLAT:
                    setFlat(true);
                    setMesh(false);
                    setFlip(false);
                    break;
                case MESH:
                    setFlat(false);
                    setMesh(true);
                    setFlip(false);
                    break;
                 case FLIPPED_MESH:
                    setFlat(false);
                    setMesh(true);
                    setFlip(true);
                    break;
           }
        }

        @Transient
        public void setFlat(boolean value) {
            setBits(value, FLAT);
        }

        @Transient
        public void setMesh(boolean value) {
            setBits(value, MESH);
        }

        @Transient
        public void setFlip(boolean value) {
            setBits(value, FLIP);
        }

        @Transient
        public void setCompact(boolean value) {
            setBits(value, COMPACT);
        }

        @Transient
        public void setSection(boolean value) {
            setBits(value, SECTION);
        }
        
        @Transient
        public void setExpanded(boolean value) {
            setBits(value, EXPANDED);
        }
        
        @Transient
        public void setSort(Sort value) {
            setBits(Sort.ALPHABETIC.equals(value), SORT);
        }
        
        private void setBits(boolean value, int mask) {
            if (value) {
                flags |= mask;
            } else {
                flags &= ~mask;
            }            
        }
        
    }
    
    protected abstract class DisplayNode extends Node implements MonitorTable.Listener {
        
        DisplayNode() {
            super("display");
        }
        
        DisplayNode(String name) {
            super(name);
        }
        
        abstract JComponent getRendererComponent();
        
        abstract JComponent getEditorComponent();

        @Override
        public void stateChanged(MonitorTable.Event e) {
            if (parent == null) return;
            switch (e.getReason()) {
                case CELLS:
                    treeModel.nodeChanged(this);
                    break;
                case TABLE:
                    InterNode p = (InterNode)parent;
                    p.rebuild();
                    tree.startEditingAtPath(new TreePath(p.getPath()));
                    break;
            }
            
        }
        
    }
    
    protected class MonitorTableNode extends DisplayNode {
        
        private final MonitorTable model;
        private final JPanel renderPanel, editPanel;
        
        MonitorTableNode(MonitorTable tableModel) {
            this(tableModel, tableModel.getClass().getName());
        }
        
        MonitorTableNode(MonitorTable tableModel, String name) {
            
            super(name);
            model = tableModel;

            JComponent table = model.getTable();
            renderPanel = new JPanel(new BorderLayout());
            renderPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0), BorderFactory.createLineBorder(Color.BLACK)));
            if (model.showHeader()) {
                renderPanel.add(((JTable)table).getTableHeader(), BorderLayout.NORTH);
            }
            renderPanel.add(table, BorderLayout.CENTER);
            
            table = model.getTable();
            editPanel = new JPanel(new BorderLayout());
            editPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0), BorderFactory.createLineBorder(Color.BLACK)));
            if (model.showHeader()) {
                editPanel.add(((JTable)table).getTableHeader(), BorderLayout.NORTH);
            }
            editPanel.add(table, BorderLayout.CENTER);
        }
        
        @Override
        JComponent getRendererComponent() {
            return renderPanel;
        }

        @Override
        JComponent getEditorComponent() {
            return editPanel;
        }
        
        @Override
        void clear() {
            model.destroy();
            super.clear();
        }

        @Override
        Serializable save() {
            return model.save();
        }
        
    }

    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor extends AbstractMonitorView.Descriptor {
        
        private TreeMap<String, Serializable> nodes;

        public TreeMap<String, Serializable> getNodes() {
            return nodes;
        }

        public void setNodes(TreeMap<String, Serializable> nodes) {
            this.nodes = nodes;
        }
        
    }
    
    /**
     * Returns a descriptor that contains information required to re-create this view in its current state.
     * @return View descriptor, or {@code null} is this view does not support saving.
     */
    @Override
    public Descriptor save() {
        getDescriptor().setNodes(null);
        updateDescriptor(root);
        return getDescriptor();
    }
    
    /**
     * Restores this view to the state described by the provided descriptor, to the extent possible.
     * @param descriptor View descriptor.
     */
    @Override
    public void restore(MonitorView.Descriptor descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
        }
    }

    /**
     * Returns a reference to the descriptor maintained by this view.
     * @return Descriptor maintained by this view.
     */
    @Override
    protected Descriptor getDescriptor() {
        return (Descriptor) descriptor;
    }
    
    private void updateDescriptor(Node node) {
        TreeMap<String, Serializable> nodes = getDescriptor().getNodes();
        if (nodes == null) {
            nodes = new TreeMap<>();
            getDescriptor().setNodes(nodes);
        }
        String path = node == root ? "" : (node.getPathString() +"/");
        Enumeration<Node> en = node.children();
        while (en.hasMoreElements()) {
            updateDescriptor(en.nextElement(), path, nodes);
        }
    }
    
    private void updateDescriptor(Node node, String path, TreeMap<String, Serializable> map) {
            Serializable desc = node.save();
            path += node.toString();
            if ( desc == null || (desc instanceof InterNodeDescriptor && ((InterNodeDescriptor)desc).isDefault()) ) {
                map.remove(path);
            } else {
                map.put(path, desc);
            }
            path += "/";
            Enumeration<Node> en = node.children();
            while (en.hasMoreElements()) {
                updateDescriptor(en.nextElement(), path, map);
            }
    }
    
}
