package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Point;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.event.TableModelEvent;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView.ChannelHandle;

/**
 *
 *
 * @author onoprien
 */
public class SectionedTable extends MonitorTable {

// -- Fields : -----------------------------------------------------------------
    
    private List<MonitorField> columns;
    
    private LinkedHashMap<String, List<Map.Entry<String, ChannelHandle>>> sections;
    private Set<String> expandedSections;

// -- Life cycle : -------------------------------------------------------------
    
    static public SectionedTable getInstance(Collection<Map.Entry<String,ChannelHandle>> channels, List<MonitorField> fields, Descriptor descriptor) {
        SectionedTable instance = new SectionedTable();
        instance.init(channels, fields, descriptor);
        return instance;
    }
    
    private SectionedTable() {
    }
    
    public void init(Collection<Map.Entry<String,ChannelHandle>> channels, List<MonitorField> fields, Descriptor descriptor) {
        
        // Columns and collapsed sections:

        List<ChannelHandle> handles = channels.stream().map(e -> e.getValue()).collect(Collectors.toList());
        columns = trimAbsentFields(fields, handles);
        nColumns = columns.size()+1;
        expandedSections = (descriptor == null || descriptor.getExpandedSections() == null) ? new HashSet<>() : new HashSet<>(Arrays.asList(descriptor.getExpandedSections()));

        // Sectioned map of channels:
        
        sections = new LinkedHashMap<>();
        List<Map.Entry<String, ChannelHandle>> defSection = new ArrayList<>();
        sections.put("", defSection);

        for (Map.Entry<String, ChannelHandle> e : channels) {
            String dpath = e.getKey();
            int i = dpath.indexOf("//");
            String section;
            String name;
            if (i == -1) {
                defSection.add(e);
            } else {
                section = dpath.substring(0, i);
                name = dpath.substring(i + 2);
                List<Map.Entry<String, ChannelHandle>> cc = sections.get(section);
                if (cc == null) {
                    cc = new ArrayList<>();
                    sections.put(section, cc);
                }
                cc.add(new AbstractMap.SimpleImmutableEntry<>(name, e.getValue()));
            }
        }
        
        // Build list of rows:
        
        int rowOffset = 0;
        cells = new ArrayList<>();
        for (String section : sections.keySet()) {
            List<ArrayList<Cell>> rows = makeSection(section, rowOffset);
            rowOffset += rows.size();
            cells.addAll(rows);
        }
        cells.trimToSize();
        nRows = cells.size();
    }
    
    @Override
    public void destroy() {
        sections.values().forEach(items -> items.forEach(e -> e.getValue().setView(null)));
    }
    
    
// -- Table model : ------------------------------------------------------------
    
    @Override    
    public String getColumnName(int column) {
        return column == 0 ? "Channel" : columns.get(column-1).getTitle();
    }
    
    
// -- Customizing table : ------------------------------------------------------
    
//    @Override
//    protected JTable makeTable() {
//        return new Table(this);
//    }
//    
//    static private class Table extends JTable {
//        
//        Table(SectionedTable model) {
//            super(model);
//        }
//
//        @Override
//        public void tableChanged(TableModelEvent e) {
//            super.tableChanged(e);
//            int type = e.getType();
//            if (type == TableModelEvent.INSERT || type == TableModelEvent.DELETE) {
//                resizeAndRepaint();
//            }
//        }
//        
//    }

// -- Handling mouse events : --------------------------------------------------
    @Override
    protected void tableMouseClicked(MouseEvent evt) {
        if (evt.getClickCount() == 2) {
            JTable table = (JTable) evt.getSource();
            Point point = evt.getPoint();
            int row = table.rowAtPoint(point);
            Cell cell = cells.get(row).get(0);
            if (cell instanceof SectionTitleCell) {
                SectionTitleCell c = (SectionTitleCell) cell;
                String section = c.getTitle();
                if (c.isCollapsed()) {
                    expandSection(row, section);
                } else {
                    collapseSection(row, section);
                }
            }
        }
        super.tableMouseClicked(evt);
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private List<ArrayList<Cell>> makeSection(String section, int rowOffset) {
        List<Map.Entry<String, ChannelHandle>> channels = sections.get(section);
        if (channels == null || channels.isEmpty()) return Collections.emptyList();
        boolean collapsed = !section.isEmpty() && !expandedSections.contains(section);
        if (collapsed) { // make collapsed section row and return a singleton list
            ArrayList<Cell> row = new ArrayList<>(nColumns);
            row.add(new SectionTitleCell(section, true));
            RowUpdater updater = new RowUpdater(rowOffset);
            ArrayList<ChannelHandle> handles = new ArrayList<>(channels.size());
            for (Map.Entry<String, ChannelHandle> e : channels) {
                ChannelHandle ch = e.getValue();
                handles.add(ch);
                ch.setView(updater);
            }
            for (MonitorField field : columns) {
                Cell cell = new Cell(handles, field);
                format(cell);
                row.add(cell);
            }
            return Collections.singletonList(row);
        } else { // section is not collapsed
            List<ArrayList<Cell>> rows = new ArrayList<>(channels.size()+1);
            if (!section.isEmpty()) { // make expanded section row
                ArrayList<Cell> row = new ArrayList<>(nColumns);
                row.add(new SectionTitleCell(section, false));
                for (MonitorField field : columns) {
                    row.add(Cell.EMPTY);
                }
                rows.add(row);
                rowOffset++;
            }
            for (Map.Entry<String, ChannelHandle> channel : channels) { // make regular row
                ArrayList<Cell> row = new ArrayList<>(nColumns);
                row.add(new Cell(null, null, new Data(channel.getKey(), SwingConstants.LEFT)));
                List<ChannelHandle> handles = Collections.singletonList(channel.getValue());
                for (MonitorField field : columns) {
                    Cell cell = new Cell(handles, field);
                    format(cell);
                    row.add(cell);
                }
                rows.add(row);
                RowUpdater updater = new RowUpdater(rowOffset++);
                channel.getValue().setView(updater);
            }
            return rows;
        }
    }
    
    private void expandSection(int row, String section) {
        expandedSections.add(section);
        List<ArrayList<Cell>> sectionRows = makeSection(section, row);
        cells.remove(row);
        cells.addAll(row, sectionRows);
        nRows = cells.size();
        updateUpdaters(row + sectionRows.size() - 1);
        fireTableRowsUpdated(row, row);
        fireTableRowsInserted(row+1, row+sectionRows.size()-1);
        fireChangeEvent(TABLE_EVENT);
    }
    
    private void collapseSection(int row, String section) {
        expandedSections.remove(section);
        List<ArrayList<Cell>> sectionRows = makeSection(section, row);
        cells.set(row, sectionRows.get(0));
        int sectionSize = sections.get(section).size();
        for (int i=0; i<sectionSize; i++) {
            cells.remove(row+1);
        }
        nRows = cells.size();
        updateUpdaters(row);
        fireTableRowsUpdated(row, row);
        fireTableRowsDeleted(row + 1, row + sectionSize);
        fireChangeEvent(TABLE_EVENT);
    }
    
    private void updateUpdaters(int row) {
        while (++row < nRows) {
            Cell cell = cells.get(row).get(1);
            List<ChannelHandle> handles = cell.getItems();
            for (ChannelHandle h : handles) {
                RowUpdater updater = (RowUpdater) h.getView();
                updater.setRow(row);
            }
        }
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class RowUpdater implements AbstractMonitorView.ChannelDisplay {
        
        private int row;
        
        RowUpdater(int row) {
            this.row = row;
        }

        public void setRow(int row) {
            this.row = row;
        }

        @Override
        public void update(ChannelHandle channelHandle, List<String> attributes) {
            HashSet<MonitorField> fields = translateAttributesToFields(attributes);
            for (MonitorField f : fields) {
                int column = columns.indexOf(f);
                if (column != -1) {
                    format(cells.get(row).get(column+1), channelHandle);
                }
            }
            fireTableRowsUpdated(row, row);
            fireChangeEvent(CELLS_EVENT);
        }
    }
    
    private class SectionTitleCell extends Cell {
        
        private final String title;
        private final boolean collapsed;
        
        SectionTitleCell(String title, boolean collapsed) {
            super(null, null, new Data("<html> "+ (collapsed ? "&#8853;" : "&#8854;") +"<b> "+ title, SwingConstants.LEFT, (collapsed ? "Double-click to expand section" : "Double-click to collaps section")));
            this.title = title;
            this.collapsed = collapsed;
        }

        public boolean isCollapsed() {
            return collapsed;
        }
        
        String getTitle() {
            return title;
        }        
    }
    
    
// -- Saving/Restoring : -------------------------------------------------------
    
    static public class Descriptor implements Serializable {

        private String[] expandedSections;

        public String[] getExpandedSections() {
            return expandedSections;
        }

        public void setExpandedSections(String[] expandedSections) {
            this.expandedSections = expandedSections;
        }
        
    }
    
    @Override
    public Descriptor save() {
        if (expandedSections != null && !expandedSections.isEmpty()) {
//            HashSet<String> expanded = new HashSet<>(expandedSections);
//            expanded.retainAll(sections.keySet());
            Descriptor desc = new Descriptor();
            desc.setExpandedSections(expandedSections.toArray(new String[expandedSections.size()]));
            return desc;
        } else {
            return null;
        }
    }
    
    public void restore(Descriptor desc) {
        // At the moment, there is no use for this method: descriptor is supplied to constructor.
        // If it ever becomes desirable to modify table settings without re-creating the table,
        // we might consider using this method from getInstance() instead of init().
    }

}
