package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import java.util.stream.Collectors;
import javax.swing.SwingConstants;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView.ChannelHandle;

/**
 * {@link MonitorTable} that displays channel values in a 2D grid.
 *
 * @author onoprien
 */
public class MeshTable extends MonitorTable {

// -- Fields : -----------------------------------------------------------------
    
    private List<MonitorField> extraColumns;
    private int firstExtraColumn;
    
    private LinkedHashMap<String, List<Map.Entry<String[], ChannelHandle>>> sections;

// -- Life cycle : -------------------------------------------------------------
    
    static public MeshTable getInstance(Collection<Map.Entry<String,ChannelHandle>> channels, List<MonitorField> fields) {
        MeshTable instance = new MeshTable();
        instance.init(channels, fields);
        return instance;
    }
    
    private MeshTable() {
    }
    
    public void init(Collection<Map.Entry<String,ChannelHandle>> channels, List<MonitorField> fields) {
        
        // Columns and collapsed sections:

        List<ChannelHandle> handles = channels.stream().map(e -> e.getValue()).collect(Collectors.toList());
        extraColumns = trimAbsentFields(fields, handles, Arrays.asList(MonitorField.NAME, MonitorField.VALUE));

        // Sectioned map of channels:
        
        sections = new LinkedHashMap<>();
        List<Map.Entry<String[], ChannelHandle>> defSection = new ArrayList<>();
        sections.put("", defSection);

        for (Map.Entry<String, ChannelHandle> e : channels) {
            String[] ss = e.getKey().split("/+");
            if (ss.length == 2) {
                defSection.add(new AbstractMap.SimpleImmutableEntry<>(ss, e.getValue()));
            } else if (ss.length == 3) {
                String section = ss[0];
                List<Map.Entry<String[], ChannelHandle>> cc = sections.get(section);
                if (cc == null) {
                    cc = new ArrayList<>();
                    sections.put(section, cc);
                }
                cc.add(new AbstractMap.SimpleImmutableEntry<>(Arrays.copyOfRange(ss, 1, 3), e.getValue()));
            }
        }
        
        // Build list of rows:
        
        ArrayList<String> columns = new ArrayList<>();
        cells = new ArrayList<>();
        ArrayList<Cell> columnNamesRow = new ArrayList<>();
        columnNamesRow.add(Cell.EMPTY);
        cells.add(columnNamesRow);
        int rowOffset = 1;
        
        for (Map.Entry<String, List<Map.Entry<String[], ChannelHandle>>> es : sections.entrySet()) {
            String section = es.getKey();
            if (!section.isEmpty()) { // section title row
                ArrayList<Cell> sectionTitleRow = new ArrayList<>();
                sectionTitleRow.add(new SectionTitleCell(section));
                cells.add(sectionTitleRow);
                rowOffset++;
            }
            ArrayList<String> rows = new ArrayList<>();
            for (Map.Entry<String[], ChannelHandle> e : es.getValue()) {
                String rowName = e.getKey()[0];
                String columnName = e.getKey()[1];
                int rowIndex = rows.indexOf(rowName);
                ArrayList<Cell> row;
                if (rowIndex == -1) {
                    rows.add(rowName);
                    rowIndex = rows.size() - 1 + rowOffset;
                    row = new ArrayList<>();
                    row.add(new Cell(null, null, new Data(rowName, SwingConstants.LEFT)));
                    cells.add(row);
                } else {
                    rowIndex += rowOffset;
                    row = cells.get(rowIndex);
                }
                int columnIndex = columns.indexOf(columnName);
                if (columnIndex == -1) {
                    columns.add(columnName);
                    columnNamesRow.add(new Cell(null, null, new Data(columnName, SwingConstants.CENTER)));
                    columnIndex = columns.size();
                } else {
                    columnIndex++;
                }
                while (row.size() <= columnIndex) {
                    row.add(Cell.EMPTY);
                }
                ValueCell cell = new ValueCell(e.getValue(), rowIndex);
                e.getValue().setView(cell);
                format(cell);
                row.set(columnIndex, cell);
            }
            rowOffset += rows.size();
        }
        
        cells.trimToSize();
        nRows = cells.size();
        nColumns = 1 + columns.size() + extraColumns.size();
        firstExtraColumn = columns.size();
        
        ArrayList<Cell> titleRow = cells.get(0);
        for (MonitorField f : extraColumns) {
            titleRow.add(new Cell(null, null, new Data(f.getTitle(), SwingConstants.CENTER)));
        }
        
        for (int i=1; i<nRows; i++) {
            ArrayList<Cell> row = cells.get(i);
            ArrayList<ChannelHandle> items = new ArrayList<>(row.size());
            for (Cell cell : row) {
                items.addAll(cell.getItems());
            }
            items.trimToSize();
            while (row.size() < columns.size()+1) {
                row.add(Cell.EMPTY);
            }
            for (MonitorField f : extraColumns) {
                if (items.isEmpty()) {
                    row.add(Cell.EMPTY);
                } else {
                    Cell cell = new Cell(items, f);
                    format(cell);
                    row.add(cell);
                }
            }
            row.trimToSize();
        }
    }
    
    @Override
    public void destroy() {
        sections.values().forEach(items -> items.forEach(e -> e.getValue().setView(null)));
    }

// -- Overriding MonitorTable : ------------------------------------------------
    
    @Override
    public boolean showHeader() {
        return false;
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class SectionTitleCell extends Cell {
        
        private final String title;
        
        SectionTitleCell(String title) {
            super(null, null, new Data("<html><b> "+ title, SwingConstants.LEFT));
            this.title = title;
        }
        
        String getTitle() {
            return title;
        }        
    }
    
    private class ValueCell extends Cell implements AbstractMonitorView.ChannelDisplay {
        
        private final int row;
        
        ValueCell(ChannelHandle channelHandle, int row) {
            super(channelHandle, MonitorField.VALUE);
            this.row = row;
        }

        @Override
        public void update(ChannelHandle channelHandle, List<String> attributes) {
            HashSet<MonitorField> fields = translateAttributesToFields(attributes);
            for (MonitorField f : fields) {
                if (MonitorField.VALUE.equals(f)) {
                    format(this);
                } else {
                    int column = extraColumns.indexOf(f);
                    if (column != -1) {
                        format(cells.get(row).get(column + firstExtraColumn), channelHandle);
                    }
                }
            }
            fireTableRowsUpdated(row, row);
            fireChangeEvent(CELLS_EVENT);
        }
        
    }
    
}
