package org.lsst.ccs.gconsole.plugins.monitor;

import java.io.Serializable;
import java.util.*;
import javax.swing.JComponent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusAggregator;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;

/**
 * A monitoring data view that manages several child views.
 * 
 * Depending view modes:<ul>
 * <li>Listens to the status aggregator independently, with its own filter.
 * <li>Receives events filtered by the parent view filter.
 * <li>Has its AbstractMonitorView methods called by the parent view.
 * </ul>
 * Calls to setFilter(...) are forwarded to children as well. That means children added
 * before and after this call are treated differently.
 *
 * @author onoprien
 */
public class CompoundView extends AbstractMonitorView {
    
    /**
     * Enumeration of management modes for child views.
     */
    public enum Mode {
        
        /**
         * Listens to the status aggregator independently, possibly with its own filter.
         * When the compound view is installed or un-installed,  {@code install(...)} 
         * or {@code uninstall()} methods of its children added in this mode
         * are also called. The {@code setFilter(...)} method is forwarded as well.
         * Otherwise, child views of this type are independent, updates received by
         * the parent are not forwarded to them.
         */
        INDEPENDENT,
        
        /**
         * Receives events filtered by the compound view filter.
         * Child views of this type are not independently installed, update events
         * received by the compound view are forwarded to them.
         */
        FORWARD, 
        
        /**
         * {@code AbstractMonitorView} hooks are called by the compound view.
         * Only compatible with views that extend {@link AbstractMonitorView}.
         * Child views of this type are not independently installed, their update
         * hooks declared by {@code AbstractMonitorView} class are called whenever
         * the parent compound view receives an update event.
         */
        HOOK
    }

// -- Fields : -----------------------------------------------------------------
    
    private final JComponent panel;
    private final ArrayList<MonitorView> children1 = new ArrayList<>(0); // INDEPENDENT
    private final ArrayList<MonitorView> children2 = new ArrayList<>(0); // FORWARD
    private final ArrayList<AbstractMonitorView> children3 = new ArrayList<>(0); // HOOK

// -- Life cycle : -------------------------------------------------------------
    
    public CompoundView(JComponent panel) {
        this.panel = panel;
    }
    

// -- Getters and setters : ----------------------------------------------------

    /**
     * Returns the graphical component maintained by this view.
     * @return Graphical component to be displayed by the GUI.
     */
    @Override
    public JComponent getPanel() {
        return panel;
    }

    /**
     * Sets a channels filter.
     * The call is forwarded to children added in {@code INDEPENDENT} and {@code FORWARD} modes.
     * 
     * @param filter Channels filter to be used by this view.
     */
    @Override
    public void setFilter(AgentChannelsFilter filter) {
        super.setFilter(filter);
        children1.forEach(child -> child.setFilter(filter));
        children2.forEach(child -> child.setFilter(filter));
    }
    
    
// -- Adding views : -----------------------------------------------------------
    
    public void addView(MonitorView view, Mode mode) {
        switch (mode) {
            case INDEPENDENT:
                children1.add(view);
                break;
            case FORWARD:
                children2.add(view);
                break;
            case HOOK:
                try {
                    children3.add((AbstractMonitorView)view);
                } catch (ClassCastException x) {
                    throw new IllegalArgumentException("HOOK mode can only be used to manage views of types that extend AbstractMonitorView", x);
                }
                break;
        }
    }
    
    
// -- Listening to the aggregator : --------------------------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        super.connect(event);
        children2.forEach(view -> view.connect(event.filter(view.getFilter())));
    }

    @Override
    public void configure(AgentStatusEvent event) {
        super.configure(event);
        children2.forEach(view -> view.configure(event.filter(view.getFilter())));
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        super.statusChanged(event);
        children2.forEach(view -> view.statusChanged(event.filter(view.getFilter())));
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        super.disconnect(event);
        children2.forEach(view -> view.disconnect(event.filter(view.getFilter())));
    }


// -- Calling AbstractMonitorView hooks : --------------------------------------    

    @Override
    protected void connect(AgentInfo agent) {
        children3.forEach(view -> view.connect(agent));
    }

    @Override
    protected void disconnect(AgentInfo agent) {
        children3.forEach(view -> view.disconnect(agent));
    }

    @Override
    protected void addChannels(AgentInfo agent, Map<String, AgentChannel> channels) {
        children3.forEach(view -> view.addChannels(agent, channels));
    }

    @Override
    protected void removeChannels(AgentInfo agent, List<String> paths) {
        children3.forEach(view -> view.removeChannels(agent, paths));
    }

    @Override
    protected void updateChannels(AgentInfo agent, Map<String, Map.Entry<AgentChannel, List<String>>> channels) {
        children3.forEach(view -> view.updateChannels(agent, channels));
    }
    
    
// -- Installation : -----------------------------------------------------------

    @Override
    public void install() {
        super.install();
        children1.forEach(view -> view.install());
    }

    @Override
    public void uninstall() {
        super.uninstall();
        children1.forEach(view -> view.uninstall());
    }

    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor extends AbstractMonitorView.Descriptor {

        private MonitorView.Descriptor[] children;

        public MonitorView.Descriptor[] getChildren() {
            return children;
        }

        public void setChildren(MonitorView.Descriptor[] children) {
            this.children = children;
        }
        
    }
    
    /**
     * Returns a descriptor that contains information required to re-create this view in its current state.
     * @return View descriptor.
     */
    @Override
    public Descriptor save() {
        if (descriptor == null) descriptor = new Descriptor();
        List<MonitorView> views = getChildren();
        MonitorView.Descriptor[] descriptors = new MonitorView.Descriptor[views.size()];
        for (int i=0; i<descriptors.length; i++) {
            descriptors[i] = views.get(i).save();
        }
        ((Descriptor)descriptor).setChildren(descriptors);
        return (Descriptor)descriptor;
    }
    
    /**
     * Restores this view to the state described by the provided descriptor, to the extent possible.
     * @param descriptor View descriptor.
     */
    @Override
    public void restore(MonitorView.Descriptor descriptor) {
        if (descriptor instanceof Descriptor) {
            Descriptor desc = (Descriptor) descriptor;
            this.descriptor = desc;
            MonitorView.Descriptor[] dd = desc.getChildren();
            if (dd != null) {
                List<MonitorView> vv = getChildren();
                if (dd.length == vv.size()) {
                    for (int i=0; i<dd.length; i++) {
                        vv.get(i).restore(dd[i]);
                    }
                }
            }
        }
    }
    
    private List<MonitorView> getChildren() {
        ArrayList<MonitorView> out = new ArrayList<>(children1.size() + children2.size() + children3.size());
        out.addAll(children1);
        out.addAll(children2);
        out.addAll(children3);
        return out;
    }

}
