package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.awt.Dialog;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.io.Serializable;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.InstanceDialog;

/**
 * Dialog to select monitoring data view type and filter.
 * <p>
 * Currently implemented to look at these sources:<ul>
 * <li>Objects of type {@code AgentChannelsFilter} in console lookup.
 * <li>Strings with ID "AgentChannelsFilter" and "MonitorView" in console lookup.
 </ul>
 * @author onoprien
 */
class MonitorPageDialog extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private final String defaultName = "Monitor";
    private boolean nameNotEdited = true;
    private MonitorView view;
    private AgentChannelsFilter filter;
    
    private InstanceDialog viewDialog, filterDialog;
    private Descriptor descriptor;
    
    private JTextField nameField, viewField, filterField;

// -- Life cycle : -------------------------------------------------------------
    
    private MonitorPageDialog(Component parentComponent) {
        
        super(parentComponent == null ? null : SwingUtilities.getWindowAncestor(parentComponent), "Open new monitoring page", Dialog.ModalityType.APPLICATION_MODAL);
        GridBagLayout gridbag = new GridBagLayout();
        setLayout(gridbag);
        
        GridBagConstraints c = new GridBagConstraints();
        c.insets = new Insets(3, 8, 3, 8);
        
        c.anchor = GridBagConstraints.EAST;
        
        c.gridx = 0;
        c.gridy = 0;
        JLabel label = new JLabel("Page name:");
        gridbag.setConstraints(label, c);
        add(label);
        c.gridy = 1;
        label = new JLabel("View type:");
        gridbag.setConstraints(label, c);
        add(label);
        c.gridy = 2;
        label = new JLabel("Filter:");
        gridbag.setConstraints(label, c);
        add(label);
        
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        
        c.gridx = 1;
        c.gridy = 0;
        nameField = new JTextField(defaultName, 50);
        nameField.addCaretListener(e -> {
            if (nameNotEdited) nameNotEdited = false;
        });
        gridbag.setConstraints(nameField, c);
        add(nameField);
        
        c.gridy = 1;
        viewField = new JTextField("Default view", 50);
        viewField.setEditable(false);
        gridbag.setConstraints(viewField, c);
        add(viewField);
        
        c.gridy = 2;
        filterField = new JTextField("Unfiltered", 50);
        filterField.setEditable(false);
        gridbag.setConstraints(filterField, c);
        add(filterField);
        
        c.fill = GridBagConstraints.NONE;
        c.weightx = 0.;
        
        c.gridx = 2;
        c.gridy = 1;
        JButton button = new JButton("...");
        button.setToolTipText("Select view type");
        button.addActionListener(e -> {
            viewDialog = InstanceDialog.show(MonitorView.class, MonitorPageDialog.this, "Select view type", MonitorView.class.getName());
            MonitorView v = (MonitorView) viewDialog.getInstance();
            if (v != null) {
                view = v;
                String name = view.getName();
                if (name == null || name.trim().isEmpty()) {
                    Class<?> viewClass = view.getClass();
                    ConsoleLookup cl = viewClass.getAnnotation(ConsoleLookup.class);
                    if (cl != null) name = cl.name();
                    if (name == null || name.isEmpty()) name = viewClass.getSimpleName();
                } else if (nameNotEdited && nameField.getText().equals(defaultName)) {
                    nameField.setText(name);
                }
                viewField.setText(name);
            }
        });
        gridbag.setConstraints(button, c);
        add(button);
        
        c.gridy = 2;
        button = new JButton("...");
        button.setToolTipText("Select filter");
        button.addActionListener(e -> {
            filterDialog = InstanceDialog.show(AgentChannelsFilter.class, MonitorPageDialog.this, "Select filter", AgentChannelsFilter.class.getName());
            AgentChannelsFilter f = (AgentChannelsFilter) filterDialog.getInstance();
            if (f != null) {
                filter = f;
                String name = filter.getName();
                if (name == null || name.isEmpty()) {
                    Class<?> filterClass = filter.getClass();
                    ConsoleLookup cl = filterClass.getAnnotation(ConsoleLookup.class);
                    if (cl != null) name = cl.name();
                    if (name == null || name.isEmpty()) name = filterClass.getSimpleName();
                } else {
                    if (nameNotEdited) {
                        nameField.setText(name);
                    }
                }
                filterField.setText(name);
            }
        });
        gridbag.setConstraints(button, c);
        add(button);
        
        c.gridx = 0;
        c.gridy = 3;
        c.anchor = GridBagConstraints.SOUTHEAST;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.weighty = 1.;
        c.insets = new Insets(16, 8, 4, 8);
        
        JPanel box = new JPanel();
        box.setLayout(new GridLayout(1,2,16,5));
        button = new JButton("Cancel");
        button.addActionListener(e -> {
            view = null;
            filter = null;
            viewDialog = null;
            filterDialog = null;
            dispose();
        });
        box.add(button);
        button = new JButton("OK");
        button.addActionListener(e -> {
            done();
            dispose();
        });
        box.add(button);
        gridbag.setConstraints(box, c);
        add(box);
    }
    
    static MonitorPageDialog show(Component parentComponent) {
        MonitorPageDialog dialog = new MonitorPageDialog(parentComponent);
        dialog.setSize(dialog.getPreferredSize());
        dialog.pack();
        dialog.setLocationRelativeTo(parentComponent);
        dialog.setVisible(true);
        return dialog;
    }
    
    static MonitorView getView(Descriptor descriptor) {
        MonitorView view = descriptor.viewDescriptor == null ? new DefaultView() : InstanceDialog.getInstance(descriptor.viewDescriptor, MonitorView.class, (String)null);
        if (view == null) return null;
        if (descriptor.name != null) view.setName(descriptor.name);
        if (descriptor.filterDescriptor != null) {
            AgentChannelsFilter filter = InstanceDialog.getInstance(descriptor.filterDescriptor, AgentChannelsFilter.class, (String)null);
            if (filter == null) return null;
            view.setFilter(filter);
        } else {
            view.setFilter(AgentChannelsFilter.ALL);
        }
        return view;
    }
    
    static MonitorView getView(MonitorView.Descriptor descriptor) {
        Serializable desc = descriptor.getCreator();
        if (desc instanceof Descriptor) {
            MonitorView view = getView((Descriptor)(desc));
            if (view == null) return null;
            view.restore(descriptor);
            return view;
        } else {
            return null;
        }
    }
    
// -- Getters : ----------------------------------------------------------------
    
    MonitorView getView() {
        return view;
    }
    
    Descriptor getDescriptor() {
        return descriptor;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void done() {
        
        descriptor = new Descriptor();
        if (view == null) {
            view = new DefaultView();
        } else {
            descriptor.setViewDescriptor(viewDialog.getDescriptor());
        }
        String viewName = nameField.getText();
        view.setName(viewName);
        descriptor.setName(viewName);
        if (filter == null) {
            filter = AgentChannelsFilter.ALL;
        } else {
            descriptor.setFilterDescriptor(filterDialog.getDescriptor());
        }
        view.setFilter(filter);
        
        MonitorView.Descriptor viewDesc = view.save();
        if (viewDesc == null) {
            viewDesc = new AbstractMonitorView.Descriptor();
        }
        viewDesc.setCreator(descriptor);
        viewDesc.setName(viewName);
        view.restore(viewDesc);
    }

    
// -- Descripror class : -------------------------------------------------------
    
    static public class Descriptor implements Serializable {

        private String name;
        private InstanceDialog.Descriptor viewDescriptor;
        private InstanceDialog.Descriptor filterDescriptor;

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }
        
        public InstanceDialog.Descriptor getViewDescriptor() {
            return viewDescriptor;
        }

        public void setViewDescriptor(InstanceDialog.Descriptor viewDescriptor) {
            this.viewDescriptor = viewDescriptor;
        }
        
        public InstanceDialog.Descriptor getFilterDescriptor() {
            return filterDescriptor;
        }

        public void setFilterDescriptor(InstanceDialog.Descriptor filterDescriptor) {
            this.filterDescriptor = filterDescriptor;
        }
       
    }

}
