package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.HashMap;

/**
 * Identifier for a data, metadata, or attribute that may be associated with a monitored channel.
 * Also identifies a column in a monitoring table, and may contain hints on how that column
 * should be rendered.
 * <p>
 * Equality and hash code are based on the {@code name} parameter provided to the constructor
 * and returned by {@code name()} method.
 *
 * @author onoprien
 */
public class MonitorField {
    
    static private final HashMap<String,MonitorField> fields = new HashMap<>();
    
    static public final MonitorField NULL = getInstance("_NULL_", "", false, false);
    static public final MonitorField NAME = getInstance("name", "Name", false, false);
    static public final MonitorField VALUE = getInstance("value", "Value", true, false);
    static public final MonitorField UNITS = getInstance("unit", "Units", false, false);
    static public final MonitorField LOW = getInstance("alarmLow", "Low Limit", true, true);
    static public final MonitorField ALERT_LOW = getInstance("alertLow", "Al.", false, false);
    static public final MonitorField HIGH = getInstance("alarmHigh", "High Limit", true, true);
    static public final MonitorField ALERT_HIGH = getInstance("alertHigh", "Al.", false, false); 
    static public final MonitorField DESCR = getInstance("description", "Description", false, false);
    static public final MonitorField LOW_WARN = getInstance("warningLow", "Low Warn", true, true);
    static public final MonitorField HIGH_WARN = getInstance("warningHigh", "High Warn", true, true);
    
    static public final String ID_KEY = "MONITOR_ID";
    static public final String PAGE_KEY = "MONITOR_PAGE";
    static public final String STATE_KEY = "MONITOR_STATE";
    static public final String FORMAT_KEY = "MONITOR_FORMAT";
    static public final String SECTION_KEY = "MONITOR_SECTION";

// -- Fields : -----------------------------------------------------------------
    
    private final String id;
    private final boolean updatable;
    private final boolean editable;
    private final String title;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an instance.
     * 
     * @param name Unique string ID. Used as the last key segment in status messages.
     * @param title Human readable title.
     * @param updatable {@code true} if this field is not guaranteed to remain constant.
     * @param editable {@code true} if this field is editable through monitoring views.
     */
    public MonitorField(String name, String title, boolean updatable, boolean editable) {
        this.id = name;
        this.updatable = updatable;
        this.editable = editable;
        this.title = title;
    }
    
    /**
     * Returns a newly created instance with the specified parameters.
     * Instances created with this method can be later looked up by calling {@link #valueOf(String)}.
     * 
     * @param name Unique string ID. Used as the last key segment in status messages.
     * @param title Human readable title.
     * @param updatable {@code true} if this field is not guaranteed to remain constant.
     * @param editable {@code true} if this field is editable through monitoring views.
     * @return New instance.
     * @throws IllegalArgumentException if an instance with this {@code id} already exists.
     */
    static public synchronized MonitorField getInstance(String name, String title, boolean updatable, boolean editable) {
        if (fields.containsKey(name)) throw new IllegalArgumentException("Field with ID "+ name +" already exists.");
        MonitorField field = new MonitorField(name, title, updatable, editable);
        fields.put(name, field);
        return field;
    }
    
    /**
     * Returns an instance with the specified name.
     * If an instance with the specified name already exists, it is returned; otherwise, a new
     * instance is created with default parameters (title equal to name, updatable, not editable).
     * 
     * @param name Unique string ID. Used as the last key segment in status messages.
     * @return An instance with the specified name.
     */
    static public synchronized MonitorField getInstance(String name) {
        MonitorField field = valueOf(name);
        if (field == null) {
            field = getInstance(name, name, true, false);
        }
        return field;
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns an instance with the specified name, or {@code null} if no such instance exists.
     * @param name Field name.
     * @return Instance with the specified name, or {@code null} if no such instance exists.
     */
    static public synchronized MonitorField valueOf(String name) {
        return fields.get(name);
    }
    
    /**
     * Returns a unique name of this field.
     * @return Name of this field.
     */
    public String name() {
        return id;
    }
    
    /**
     * Returns human readable title associated with this field.
     * @return Title of this field.
     */
    public String getTitle() {
        return title;
    }
    
    public boolean isUpdatable() {return updatable;}
    
    public boolean isEditable() {return editable;}
    
    
// -- Overriding Object : ------------------------------------------------------

    /**
     * Returns the name of this field.
     * @return Name of this field.
     */
    @Override
    public String toString() {
        return title;
    }

    @Override
    public boolean equals(Object obj) {
        return (obj instanceof MonitorField && id.equals(((MonitorField)obj).id)) || id.equals(obj);
    }

    @Override
    public int hashCode() {
        return id.hashCode();
    }

}
