package org.lsst.ccs.gconsole.plugins.monitor;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;

/**
 * 
 *
 * @author onoprien
 */
abstract public class AbstractMonitorView implements MonitorView {

// -- Fields : -----------------------------------------------------------------
    
    protected AgentChannelsFilter filter;
    protected Descriptor descriptor;
    
    private boolean fixedDisplayList;
    private Set<String> displayChannels;
    
    protected final LinkedHashMap<String,ChannelHandle> path2data = new LinkedHashMap<>(); // display path to channel handler, in addition order

// -- Life cycle : -------------------------------------------------------------
    
// -- Getters and setters : ----------------------------------------------------

    @Override
    public AgentChannelsFilter getFilter() {
        return filter;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        this.filter = filter;
        fixedDisplayList = filter != null && filter.getDisplayChannels() != null;
    }
    
    protected boolean isChannelListFixed() {
        return fixedDisplayList;
    }
    
    @Override
    public boolean isEmpty() {
        return path2data.isEmpty();
    }
    
    
// -- Default agent status listener implementation : ---------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            connect(event.getSource());
        });
    }

    @Override
    public void configure(AgentStatusEvent event) {
        if (displayChannels == null) {
            List<String> fixed = filter.getDisplayChannels();
            if (fixed == null) {
                fixedDisplayList = false;
                displayChannels = Collections.emptySet();
            } else {
                fixedDisplayList = true;
                displayChannels = new HashSet<>(fixed);
                SwingUtilities.invokeLater(() -> {
                    addChannels(event.getSource(), fixed.stream().collect(Collectors.toMap(dp -> dp, dp -> null)));
                });
            }
        }
        update(event);
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        update(event);
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        update(event);
        SwingUtilities.invokeLater(() -> {
            disconnect(event.getSource());
        });
    }
    
    private void update(AgentStatusEvent event) {
        
        if (fixedDisplayList) {
            
            Map<String,Map.Entry<AgentChannel,List<String>>> changedChannels = new LinkedHashMap<>();
            
            for (AgentChannel channel : event.getAddedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (displayChannels.contains(displayPath)) {
                        changedChannels.put(displayPath, new AbstractMap.SimpleEntry<>(channel, null));
                    }
                }
            }
            
            for (AgentChannel channel : event.getRemovedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (displayChannels.contains(displayPath)) {
                        changedChannels.put(displayPath, null);
                    }
                }
            }
            
            for (Map.Entry<AgentChannel, List<String>> change : event.getStatusChanges().entrySet()) {
                for (String displayPath : filter.getDisplayPath(change.getKey())) {
                    if (displayChannels.contains(displayPath)) {
                        changedChannels.put(displayPath, change);
                    }
                }
            }
            
            SwingUtilities.invokeLater(() -> {
                updateChannels(event.getSource(), changedChannels);
            });
            
        } else {
            
            Map<String,AgentChannel> addedChannels = null;
            for (AgentChannel channel : event.getAddedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (addedChannels == null) addedChannels = new LinkedHashMap<>();
                    addedChannels.put(displayPath, channel);
                }
            }
            
            List<String> removedChannels = null;
            for (AgentChannel channel : event.getRemovedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (removedChannels == null) removedChannels = new ArrayList<>(event.getRemovedChannels().size());
                    removedChannels.add(displayPath);
                }
            }
            
            Map<String,Map.Entry<AgentChannel,List<String>>> changedChannels = null;
            for (Map.Entry<AgentChannel,List<String>> e : event.getStatusChanges().entrySet()) {
                for (String displayPath : filter.getDisplayPath(e.getKey())) {
                    if (changedChannels == null) changedChannels = new LinkedHashMap<>();
                    changedChannels.put(displayPath, e);
                }
            }
            
            Map<String, AgentChannel> added = addedChannels;
            List<String> removed = removedChannels;
            Map<String,Map.Entry<AgentChannel,List<String>>> changed = changedChannels;
            SwingUtilities.invokeLater(() -> {
                if (added != null) addChannels(event.getSource(), added);
                if (removed != null) removeChannels(event.getSource(), removed);
                if (changed != null) updateChannels(event.getSource(), changed);
            });
            
        }
        
    }
    
    
// -- First level EDT hooks for updating the graphical component : -------------
    
    protected void connect(AgentInfo agent) {}
    
    protected void disconnect(AgentInfo agent) {}

    /**
     * Adds channels to display.
     * @param agent Source agent.
     * @param channels Map of paths to channels.
     */
    protected void addChannels(AgentInfo agent, Map<String,AgentChannel> channels) {
        boolean modified = false;
        for (Map.Entry<String, AgentChannel> e : channels.entrySet()) {
            String path = e.getKey();
            ChannelHandle handle = createChannelHandle(path, e.getValue(), null);
            modified = (path2data.put(path, handle) == null) || modified;
        }
        if (modified) resetChannels();
    }

    /**
     * Removes a channels from this view.
     * @param agent Source agent.
     * @param paths Paths to remove.
     */
    protected void removeChannels(AgentInfo agent, List<String> paths) {
        boolean modified = false;
        for (String path : paths) {
            modified = (path2data.remove(path) != null) || modified;
        }
        if (modified) resetChannels();
    }
    
    /**
     * Called to notify this view that the specified channels have been updated.
     * @param agent Source agent.
     * @param channels Map of paths to key-value pairs, where the key is as {@code AgentChannel} object
     *                 and the value is a list of changed attributes.
     */
    protected void updateChannels(AgentInfo agent, Map<String,Map.Entry<AgentChannel,List<String>>> channels) {
        channels.forEach((path, e) -> {
            ChannelHandle handle = path2data.get(path);
            if (handle != null && handle.getView() != null) {
                if (!Objects.equals(handle.getChannel(), e.getKey())) {
                    handle.setChannel(e.getKey());
                    handle.getView().update(handle, null);
                } else {
                    handle.getView().update(handle, e.getValue());
                }
            }
        });
    }
    
    
// -- Second level EDT hooks for updating the graphical component : ------------
    
    /**
     * Rebuilds the view based on {@code path2data}.
     */
    protected void resetChannels() {
    }
    
    /**
     * Creates a new instance of {@code ChannelHandle}.
     * The implementation provided by this class creates an instance of {@code DefaultChannelHandle}.
     * Subclasses may override this method to return a different implementation of {@code ChannelHandle}.
     * 
     * @param path Display path.
     * @param channel Agent channel.
     * @param view View displaying the channel.
     * @return New instance of {@code ChannelHandle}.
     */
    protected ChannelHandle createChannelHandle(String path, AgentChannel channel, ChannelDisplay view) {
        return new DefaultChannelHandle(path, channel, view);
    }
    
    
// -- Interfaces : -------------------------------------------------------------
    
    public interface ChannelHandle {

        AgentChannel getChannel();
        
        String getPath();
        
        ChannelDisplay getView();
        
        void setPath(String path);

        void setChannel(AgentChannel channel);

        void setView(ChannelDisplay view);
        
    }
    
    public interface ChannelDisplay {
        
        void update(ChannelHandle channelHandle, List<String> attributes);
        
    }
    
    public static class DefaultChannelHandle implements ChannelHandle {
        
        private String path;
        private AgentChannel channel;
        private ChannelDisplay view;
        
        public DefaultChannelHandle(String path, AgentChannel channel, ChannelDisplay view) {
            this.path = path;
            this.channel = channel;
            this.view = view;
        }

        @Override
        public String getPath() {
            return path;
        }

        @Override
        public AgentChannel getChannel() {
            return channel;
        }

        @Override
        public ChannelDisplay getView() {
            return view;
        }

        @Override
        public void setPath(String path) {
            this.path = path;
        }

        @Override
        public void setChannel(AgentChannel channel) {
            this.channel = channel;
        }

        @Override
        public void setView(ChannelDisplay view) {
            this.view = view;
        }
        
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor implements MonitorView.Descriptor {
        
        private MonitorPageDialog.Descriptor creator;
        private String name;
        
        /**
         * Returns a descriptor of the creator of this view.
         * @return Creator descriptor.
         */
        @Override
//        public MonitorPageDialog.Descriptor getCreator() {
        public Serializable getCreator() {
            return creator;
        }
        
        /**
         * Sets a descriptor of the creator of this view.
         * @param creator Creator descriptor.
         */
        @Override
        public void setCreator(Serializable creator) {
            if (creator instanceof MonitorPageDialog.Descriptor) {
                this.creator = (MonitorPageDialog.Descriptor) creator;
            }
        }
        
        /**
         * Returns the name of the view.
         * @return View name.
         */
        @Override
        public String getName() {
            return name;
        }
        
        /**
         * Sets the name of the view.
         * @param name View name.
         */
        @Override
        public void setName(String name) {
            this.name = name;
        }
        
    }
    
    /**
     * Returns a descriptor that contains information required to re-create this view in its current state.
     * The implementation provided by this class returns the descriptor previously set with {@code restore(...)}.
     * 
     * @return View descriptor, or {@code null} is this view does not support saving.
     */
    @Override
    public Descriptor save() {
        return descriptor;
    }
    
    /**
     * Restores this view to the state described by the provided descriptor, to the extent possible.
     * The implementation provided by this class stores the provided descriptor, so that it can be
     * retrieved with {@code save()} later.
     * 
     * @param descriptor View descriptor.
     */
    @Override
    public void restore(MonitorView.Descriptor descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
        }
    }
    
    /**
     * Returns a reference to the descriptor maintained by this view.
     * This is a convenience method intended to be overridden by subclasses to return
     * an appropriate subtype of {@code AbstractMonitorView.Descriptor}.
     * 
     * @return Descriptor maintained by this view.
     */
    protected Descriptor getDescriptor() {
        return descriptor;
    }
    
}
