package org.lsst.ccs.gconsole.agent.filter;

import java.util.*;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;

/**
 * 
 *
 * @author onoprien
 */
@ConsoleLookup(id="org.lsst.ccs.gconsole.agent.AgentChannelsFilter",
               name="Explicit Lists",
               path="Built-In/Generic",
               description="Channels filter that allows the user to specify lists of accepted agents and channels.<p>"
                + "<b>Optional parameters:</b>"
                + "<dl>"
                + "<dt>name</dt><dd>Filter name.</dd>"
                + "<dt>agents</dt><dd>Comma-separated list of subsystems accepted by this filter.</dd>"
                + "<dt>channels</dt><dd>Comma-separated list of channels accepted by this filter (may contain templates and selectors).</dd>"
                + "<dt>delimeter</dt><dd>Characters in the original channel path that should be replaced by a slash (and therefore used as a path delimeter). Enter literal string, or regular expression in double quotes.</dd>"
                + "<dt>separatePagesForAgents</dt><dd>If true, the subsystem name is used as a page name.</dd>"
                + "</dl>")
public class GenericFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final String name;
    private final List<String> agents;
    private final List<String> channels;
    private final String delimeter;
    private final boolean agentTabs;
    
    private HashMap<String,String> display2origin;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs a filter.
     * 
     * @param name Name of the filter.
     * @param agents Comma-separated list of subsystems accepted by this filter.
     * @param channels Comma-separated list of channels accepted by this filter (may contain templates and selectors).
     * @param delimeter Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter).
     * @param separatePagesForAgents If {@code true}, the subsystem name is used as a page name.
     */
    public GenericFilter(String name, String agents, String channels, String delimeter, boolean separatePagesForAgents) {
        
        this.name = name;
        
        if (agents.trim().isEmpty()) {
            this.agents = null;
        } else if (agents.trim().equals("\"\"")) {
            this.agents = Collections.emptyList();
        } else {
            String[] ss = agents.split(",");
            this.agents = new ArrayList<>(ss.length);
            for (String s : ss) {
                this.agents.add(s.trim());
            }
        }
        
        if (channels.trim().isEmpty()) {
            this.channels = null;
        } else if (channels.trim().equals("\"\"")) {
            this.channels = Collections.emptyList();
        } else {
            String[] ss = channels.split(",");
            this.channels = new ArrayList<>(ss.length);
            for (String s : ss) {
                this.channels.add(s.trim());
            }
        }
        
        if (delimeter.isEmpty()) {
            this.delimeter = null;
        } else {
            String s = delimeter.trim();
            if (s.length() > 2 && s.startsWith("\"") && s.endsWith("\"")) {
                this.delimeter = s.substring(1, s.length()-1);
            } else {
                this.delimeter = Pattern.quote(delimeter);
            }
            display2origin = new HashMap<>();
        }
        
        this.agentTabs = separatePagesForAgents;
    }
    
    
// -- Implementing filter : ----------------------------------------------------

    @Override
    public List<String> getDisplayPath(String originPath) {
        String displayPath;
        if (delimeter == null) {
            if (agentTabs) {
                displayPath = originPath.replaceFirst("/", "//");
            } else {
                displayPath = originPath;
            }
        } else {
            int i = originPath.indexOf("/");
            String agent = originPath.substring(0, i);
            String localPath = originPath.substring(i+1);
            localPath = localPath.replaceAll(delimeter, "/");
            if (agentTabs) {
                displayPath = agent +"//"+ localPath;
            } else {
                displayPath = agent +"/"+ localPath;
            }
            display2origin.putIfAbsent(displayPath, originPath);
        }
        return Collections.singletonList(displayPath);
    }

    @Override
    public String getOriginPath(String displayPath) {
        if (delimeter == null) {
            if (agentTabs) {
                return displayPath.replaceFirst("//", "/");
            } else {
                return displayPath;
            }
        } else {
            return display2origin.get(displayPath);
        }
    }

    @Override
    public List<String> getOriginChannels() {
        return channels;
    }

    @Override
    public List<String> getAgents() {
        return agents;
    }

    @Override
    public String getName() {
        return name;
    }
    
    
// -- Specific filters : -------------------------------------------------------
    
    
}
