package org.lsst.ccs.gconsole.plugins.rest;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import java.time.Duration;
import java.util.Date;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Provides access to a connected REST server.
 *
 * @author onoprien
 */
public class RestServer {

// -- Fields : -----------------------------------------------------------------
    
    private final String url;
    
    private volatile String name;
    private volatile WebResource service;
    private final ReentrantLock lock = new ReentrantLock();


// -- Life cycle : -------------------------------------------------------------
    
    public RestServer(String url) {
        this.url = url;
    }


// -- Getters and setters : ----------------------------------------------------
    
    public String getName() {
        return name;
    }
    
    public void setName(String name) {
        this.name = name;
    }
    
    public WebResource getResource() {
        return service;
    }
    
    public WebResource getResource(Duration timeout) {
        long tout = timeout.toMillis();
        if (tout == 0L) {
            return service;
        } else if (tout < 0L) {
            tout = 10*365*24*3600*1000; // infinite timeout - 10 years
        }
        Date deadline = new Date(System.currentTimeMillis() + tout);
        try {
            if (lock.tryLock(tout, TimeUnit.MILLISECONDS)) {
                try {
                    Condition c = lock.newCondition();
                    while (service == null) {
                        if (!c.awaitUntil(deadline)) {
                            return null;
                        }
                    }
                    return service;
                } finally {
                    lock.unlock();
                }
            }
        } catch (InterruptedException x) {
        }
        return null;
    }

    public String getUrl() {
        return url;
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    public void refresh() {
        lock.lock();
        try {
            service = null;
        } finally {
            lock.unlock();
        }
        Thread t = new Thread(this::connect, "REST Connector");
        t.setDaemon(true);
        t.start();
    }

    private void connect() {
        
        Client client = Client.create(new DefaultClientConfig());
        lock.lock();
        try {
            service = client.resource(url).path("dataserver");
        } finally {
            lock.unlock();
        }
    }
    
}
