package org.lsst.ccs.gconsole.plugins.demo.monitor;

import java.util.*;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.gconsole.agent.AbstractChannelsFilter;

/**
 * Demo filter that displays a fixed set of channels from two different subsystems.
 * The channels are shown on three pages, with some channels appearing on more than
 * one page under different names.
 * Any page hints from groovy description are ignored.
 *
 * @author onoprien
 */
@ConsoleLookup(id="org.lsst.ccs.gconsole.agent.AgentChannelsFilter",
               name="Fixed channel set",
               path="Demo/Fixed",
               description="Channels filter that displays a fixed set of channels.")
public class FilterFixed extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final String name;
    private final List<String> agents;
    private final LinkedHashMap<String, String> d2o;
    private final HashMap<String, List<String>> o2d;

// -- Life cycle : -------------------------------------------------------------
    
    public FilterFixed(String name, Map<String, String> displayToOriginPath) {
        this.name = name;
        d2o = new LinkedHashMap(displayToOriginPath);
        o2d = new HashMap<>();
        d2o.forEach((display, origin) -> {
            List<String> paths = o2d.get(origin);
            if (paths == null) {
                paths = new ArrayList<>();
                o2d.put(origin, paths);
            }
            paths.add(display);
        });
        o2d.entrySet().forEach(e -> {
            ArrayList<String> paths = (ArrayList<String>) e.getValue();
            if (paths.size() == 1) {
                e.setValue(Collections.singletonList(paths.get(0)));
            } else {
                paths.trimToSize();
            }
        });
        HashSet<String> agentNames = new HashSet<>();
        o2d.keySet().forEach(origin -> {
            String[] ss = origin.split("/");
            agentNames.add(ss[0]);
        });
        agents = new ArrayList<>(agentNames);
    }
    
    /**
     * Demo for automatic filter instantiation.
     * 
     * @param intParameter Integer parameter.
     * @param name String parameter.
     * @param boolParameter Boolean parameter.
     * @param enumParameter Enum parameter.
     * @return Created filter.
     */
    static public FilterFixed getInstance(int intParameter, String name, boolean boolParameter, AlertState enumParameter) {
        LinkedHashMap<String, String> in = new LinkedHashMap<>();
        
        in.put("REB0//Global/Power/VP5_UNREG/Voltage", "monitor-test/REB0/Global/Power/VP5_UNREG/Voltage");
        in.put("REB0//Global/Power/VP5_UNREG/Current", "monitor-test/REB0/Global/Power/VP5_UNREG/Current");
        in.put("REB0//Global/Power/VP5_UNREG/Power", "monitor-test/REB0/Global/Power/VP5_UNREG/Power");
        in.put("REB0//Global/Power/VP7_UNREG/Voltage", "monitor-test/REB0/Global/Power/VP7_UNREG/Voltage");
        
        in.put("Voltages//Global/Power/VP5_UNREG", "monitor-test/REB0/Global/Power/VP5_UNREG/Voltage");
        in.put("Voltages//Global/Power/VP7_UNREG", "monitor-test/REB0/Global/Power/VP7_UNREG/Voltage");
        
        in.put("2 subs//From test", "monitor-test/REB0/Global/Power/VP5_UNREG/Voltage");
        in.put("2 subs//From demo", "demo-subsystem-with-monitoring/DemoChannel1");
        
        
        return new FilterFixed("Demo: Fixed Set", in);
    }
    
    static public String getPath() {
        return "static/getPath";
    }
    
    
// -- Filtering : --------------------------------------------------------------
    
    /**
     * Returns the name of this filter.
     * 
     * @return Name of this filter.
     */
    @Override
    public String getName() {
        return name;
    }
    
    /**
     * Returns a list of agent names accepted by this filter.
     * 
     * @return Agent names accepted by this filter, or {@code null} if any agent might be accepted.
     */
    @Override
    public List<String> getAgents() {
        return agents;
    }
    
    /**
     * Returns a list of display paths produced by this filter.
     * Typically, if this method returns a non-null set, components downstream from
     * this filter will display this fixed set of channels, whether or not the data
     * is available, regardless of what paths are returned by {@link #getDisplayPath}
     * applied to accepted original channels. Otherwise, any display channels for
     * which the data is available will be displayed.
     * 
     * @return Display paths produced by this filter, or {@code null} if the set is not fixed.
     */
    @Override
    public List<String> getDisplayChannels() {
        return new ArrayList<>(d2o.keySet());
    }
    
    /**
     * Returns the original path corresponding to the specified display channel.
     * 
     * @param displayPath Display channel path.
     * @return Original path, or {@code null} if the specified path does not correspond to any original channel.
     */
    @Override
    public String getOriginPath(String displayPath) {
        return d2o.get(displayPath);
    }
    
    /**
     * Returns a list of display channels for the specified original path.
     * 
     * @param originPath Original path.
     * @return List of display channels.
     */
    @Override
    public List<String> getDisplayPath(String originPath) {
        List<String> out = o2d.get(originPath);
        return out == null ? Collections.emptyList() : out;
    }

}
