package org.lsst.ccs.gconsole.agent;

import java.util.*;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * An event fired by {@link AgentStatusAggregator} to notify {@link AgentStatusListener}s of changes in the agent's status.
 *
 * @author onoprien
 */
public class AgentStatusEvent extends EventObject {

// -- Fields : -----------------------------------------------------------------
    
    private final AgentStatusAggregator aggregator;
    private final StatusMessage message;
    protected final Map<AgentChannel,List<String>> statusChanges;
    protected final List<AgentChannel> addedChannels;
    protected final List<AgentChannel> removedChannels;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructor for status change events.
     * 
     * @param aggregator Status aggregator instance.
     * @param message Status message that notified the status aggregator of the changes.
     * @param statusChanges Channels with changes.
     * @param addedChannels Added channels.
     * @param removedChannels Removed channels.
     */
    AgentStatusEvent(AgentStatusAggregator aggregator, StatusMessage message, Map<AgentChannel,List<String>> statusChanges, List<AgentChannel> addedChannels, List<AgentChannel> removedChannels) {
        super(message.getOriginAgentInfo());
        this.aggregator = aggregator;
        this.message = message;
        this.statusChanges = statusChanges;
        this.addedChannels = addedChannels;
        this.removedChannels = removedChannels;
    }
    
    /**
     * Constructor for configuration events.
     * 
     * @param aggregator Status aggregator instance.
     * @param agent Source agent descriptor.
     * @param statusChanges Channels with changes.
     * @param addedChannels Added channels.
     * @param removedChannels Removed channels.
     */
    AgentStatusEvent(AgentStatusAggregator aggregator, AgentInfo agent, Map<AgentChannel,List<String>> statusChanges, List<AgentChannel> addedChannels, List<AgentChannel> removedChannels) {
        super(agent);
        this.aggregator = aggregator;
        this.message = null;
        this.statusChanges = statusChanges == null ? Collections.emptyMap() : statusChanges;
        this.addedChannels = addedChannels == null ? Collections.emptyList() : addedChannels;
        this.removedChannels = removedChannels == null ? Collections.emptyList() : removedChannels;
    }
    
    /**
     * Constructor for connect and disconnect events.
     * 
     * @param aggregator Status aggregator instance.
     * @param agent Source agent descriptor.
     */
    AgentStatusEvent(AgentStatusAggregator aggregator, AgentInfo agent) {
        super(agent);
        this.aggregator = aggregator;
        this.message = null;
        this.statusChanges = Collections.emptyMap();
        this.addedChannels = Collections.emptyList();
        this.removedChannels = Collections.emptyList();
    }
    
    
// -- Getters : ----------------------------------------------------------------

    /**
     * Returns the descriptor of the source agent.
     * @return Agent descriptor.
     */
    @Override
    public AgentInfo getSource() {
        return (AgentInfo) source;
    }
    
    /**
     * Returns the bus message that informed the status aggregator of the change.
     * This method will only return non-null value for events passed to {@code statusChanged(...)} method of the listener.
     * @return Status message received over the bus.
     */
    public StatusMessage getMessage() {
        return message;
    }
    
    /**
     * Returns a reference to the status aggregator that fired this event.
     * @return Status aggregator instance.
     */
    public AgentStatusAggregator getAggregator() {
        return aggregator;
    }

    /**
     * Returns a map of modified channels to lists of keys of changed attributes.
     * Removed or added channels do not appear in this map.
     * This method may return non-empty map only for events passed to {@code statusChanged(...)} method of the listener.
     * @return Map of channels to modified attributes lists.
     */
    public Map<AgentChannel, List<String>> getStatusChanges() {
        return statusChanges;
    }
    
    /**
     * Returns a list of new channels.
     * This method may return non-empty list only for events passed to
     * {@code configure(...)} and {@code statusChanged(...)} methods of the listener.
     * @return List of new channels.
     */
    public List<AgentChannel> getAddedChannels() {
        return addedChannels;
    }
    
    /**
     * Returns a list of channels that are no longer published by the agent.
     * The strings in the returned list are original channel paths.
     * This method returns (possibly) non-empty map only for events passed to {@code statusChanged(...)} method of the listener.
     * <p>
     * Note: with the current implementation, this method never gets called.
     * It is declared in anticipation of implementing filter and dictionary updates.
     * @return List of removed channels.
     */
    public List<AgentChannel> getRemovedChannels() {
        return removedChannels;
    }
    
    /**
     * Returns {@code true} if this event has no added, modified, or removed channels.
     * @return {@code true} if this event has no added, modified, or removed channels.
     */
    public boolean isEmpty() {
        return statusChanges.isEmpty() && addedChannels.isEmpty() && removedChannels.isEmpty();
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    /**
     * Creates a new event, keeping only changes in channels whose local (no agent name)
     * paths are contained in the supplied set.
     * @param innerPaths Paths to keep.
     * @return Filtered Sub-event.
     */
    public AgentStatusEvent filter(Set<String> innerPaths) {
        Map<AgentChannel,List<String>> changes = statusChanges.entrySet().stream()
                .filter(e -> innerPaths.contains(e.getKey().getLocalPath()))
                .collect(Collectors.toMap(e -> e.getKey(), e -> e.getValue()));
        List<AgentChannel> added = addedChannels.stream()
                .filter(c -> innerPaths.contains(c.getLocalPath()))
                .collect(Collectors.toList());
        List<AgentChannel> removed = removedChannels.stream()
                .filter(c -> innerPaths.contains(c.getLocalPath()))
                .collect(Collectors.toList());
        if (message == null) {
            return new AgentStatusEvent(aggregator, getSource(), changes, added, removed);
        } else {
            return new AgentStatusEvent(aggregator, message, changes, added, removed);
        }
    }
    
    /**
     * Creates a new event, keeping only changes that pass the specified filter.
     * @param filter
     * @return 
     */
    public AgentStatusEvent filter(AgentChannelsFilter filter) {
        return this; // FIXME
    }
    
    
// -- Overriding Object : ------------------------------------------------------

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("Event from ");
        sb.append(getSource().getName()).append(":");
        sb.append(" added ").append(addedChannels.size()).append(" channels, removed ");
        sb.append(removedChannels.size()).append(" channels, modified ");
        sb.append(statusChanges.size()).append(" channels.");
        return sb.toString();
    }

}
