package org.lsst.ccs.gconsole.agent;

import java.util.HashMap;
import org.lsst.ccs.bus.data.AgentInfo;

/**
 * Encapsulates static properties and current status of a data channel.
 * Each channel has a central value (an {@code Object}) and optional attributes identified by string keys.
 * A channel is identified by a unique path (slash-separated string, with the subsystem
 * name as the first element). The path is used to establish channel equality and hash code.
 *
 * @author onoprien
 */
public class AgentChannel {

// -- Fields : -----------------------------------------------------------------
    
    /** Immutable empty channel. */
    public static AgentChannel NONE = new AgentChannel("", null);
    
    /**
     * Key that identifies the central value of a channel.
     * Allows treating the central value as an attribute whenever string keys are used to identify channel attributes.
     */
    public static final String VALUE_KEY = "value";
    
    private final String path;
    private final AgentInfo agent;
    
    protected Object value;
    protected HashMap<String,Object> data;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an instance.
     * @param path Unique path identifying this channel.
     * @param agent Agent this channel belongs to.
     */
    protected AgentChannel(String path, AgentInfo agent) {
        this.path = path;
        this.agent = agent;
    }
    
    /**
     * Copy constructor.
     * @param other Channel to copy.
     */
    public AgentChannel(AgentChannel other) {
        synchronized (other) {
            this.path = other.path;
            this.agent = other.agent;
            this.value = other.value;
            this.data = new HashMap<>(other.data);
        }
    }
    
// -- Getters : ----------------------------------------------------------------

    /**
     * Return the unique path of this channel.
     * @return Channel path in (agent name)/(local path) form.
     */
    public String getPath() {
        return path;
    }
    
    /**
     * Returns the name of the {@code Agent} this channel belongs to.
     * @return Agent name.
     */
    public String getAgentName() {
        return agent.getName();
    }
    
    /**
     * Returns the descriptor of the {@code Agent} this channel belongs to. 
     * @return Agent descriptor.
     */
    public AgentInfo getAgent() {
        return agent;
    }
    
    /**
     * Returns the path of this channel with the agent name and the following slash stripped.
     * Local paths are unique inside an agent.
     * @return Local path.
     */
    public String getLocalPath() {
        return path.substring(path.indexOf("/")+1);
    }
    
    /**
     * Returns the value of the channel attribute identified by the {@code key}.
     * 
     * @param key Attribute key. If {@code VALUE_KEY}, the central value is returned.
     * @return Attribute value, or {@code null} if this channel does not have the specified attribute.
     */
    public synchronized Object get(String key) {
        if (key.equals(VALUE_KEY)) {
            return get();
        } else {
            return data == null ? null : data.get(key);
        }
    }
    
    /**
     * Returns the current central value of this channel.
     * @return Current value.
     */
    public synchronized Object get() {
        return value;
    }
    
    
// -- Overriding Object : ------------------------------------------------------

    @Override
    public String toString() {
        return path;
    }

    @Override
    public boolean equals(Object obj) {
        return path.equals(obj) || (obj instanceof AgentChannel && path.equals(((AgentChannel)obj).path));
    }

    @Override
    public int hashCode() {
        return path.hashCode();
    }

}
