package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Color;
import java.awt.Component;
import java.util.*;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.event.TreeModelListener;
import javax.swing.table.JTableHeader;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.treetable.TreeTableModel;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView.ChannelDisplay;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView.ChannelHandle;

/**
 * 
 *
 * @author onoprien
 */
public class TreeTable extends MonitorTable implements TreeTableModel {

// -- Fields : -----------------------------------------------------------------
    
    private final List<MonitorField> columns;
    private final DefaultTreeModel treeModel;
    private final Node root = new Node("");
    
    private boolean isInitialized = false;

// -- Life cycle : -------------------------------------------------------------
    
    public TreeTable(Map<String,ChannelHandle> channels, List<MonitorField> fields) {
        
        columns = trimAbsentFields(fields, channels.values());
        
        // Build tree
        
        for (Map.Entry<String,ChannelHandle> e : channels.entrySet()) {
            String[] path = e.getKey().split("/+");
            Node node = root;
            for (String nodeName : e.getKey().split("/+")) {
                node = addChild(node, nodeName);
            }
            node.setChannelHandle(e.getValue());
        }
        treeModel = new DefaultTreeModel(root);
        
        // Assign row numbers to nodes
        
        nRows = setRowNumbers(-1, root);
        
        // Fill table cells
        
        nColumns = columns.size()+1;
        cells = new ArrayList<>(nRows);
        for (int i=0; i<nRows; i++) {
            cells.add(new ArrayList<>(nColumns));
        }
        
        
        Enumeration<Node> nodes = root.depthFirstEnumeration();
        while (nodes.hasMoreElements()) {
            Node node = nodes.nextElement();
            if (node == root) break;
            int row = node.getRow();
            List<ChannelHandle> handles;
            if (node.isLeaf()) {
                handles = Collections.singletonList(node.getChannelHandle());
            } else {
                handles = new ArrayList<>();
                Enumeration<Node> nn = node.depthFirstEnumeration();
                while (nn.hasMoreElements()) {
                    Node n = nn.nextElement();
                    if (n.isLeaf()) {
                        handles.add(n.getChannelHandle());
                    }
                }
                ((ArrayList)handles).trimToSize();
            }
            cells.get(row).add(new Cell(Collections.emptyList(), MonitorField.NULL, new Data(node.toString(), MonitorTable.COLOR_FG, MonitorTable.COLOR_BG, SwingConstants.LEFT)));
            for (MonitorField field : columns) {
                Cell cell = new Cell(handles, field);
                format(cell);
                cells.get(row).add(cell);
            }
        }
        
    }
    
    private Node addChild(Node parent, String name) {
        Enumeration<Node> e = parent.children();
        Node child = null;
        while (e.hasMoreElements()) {
            Node n = e.nextElement();
            if (name.equals(n.toString())) {
                child = n;
                break;
            }
        }
        if (child == null) {
            child = new Node(name);
            parent.add(child);
        }
        return child;
    }
    
    private int setRowNumbers(int start, Node node) {
        node.row = start++;
        Enumeration<Node> e = node.children();
        while (e.hasMoreElements()) {
            start = setRowNumbers(start, e.nextElement());
        }
        return start;
    }
    
    public void init() {
        if (isInitialized) return;
        Enumeration<Node> nodes = root.depthFirstEnumeration();
        while (nodes.hasMoreElements()) {
            Node node = nodes.nextElement();
            ChannelHandle ch = node.getChannelHandle();
            if (ch != null) ch.setView(node);
        }
        isInitialized = true;
    }
    
    public void destroy() {
        if (!isInitialized) return;
        Enumeration<Node> nodes = root.depthFirstEnumeration();
        while (nodes.hasMoreElements()) {
            Node node = nodes.nextElement();
            ChannelHandle ch = node.getChannelHandle();
            if (ch != null) ch.setView(null);
        }
        isInitialized = false;
    }

    @Override
    public JTable getTable() {
        JTable table = super.getTable();
        JTableHeader header = table.getTableHeader();
        header.setResizingAllowed(true);
        return table;
    }

    @Override
    public JTable makeTable() {
        init();
        return new Table(this);
    }

    
// -- Setters : ----------------------------------------------------------------

// -- Implements TreeTableModel : ----------------------------------------------
    
    @Override
    public String getColumnName(int column) {
        return column == 0 ? "Channel" : columns.get(column-1).getTitle();
    }

    @Override
    public int getHierarchicalColumn() {
        return 0;
    }

    @Override
    public Object getValueAt(Object node, int column) {
        return column == 0 ? node : getValueAt(((Node)node).row, column);
    }

    @Override
    public boolean isCellEditable(Object node, int column) {
        return column == 0 ? false : isCellEditable(((Node)node).getRow(), column);
    }

    @Override
    public void setValueAt(Object value, Object node, int column) {
        if (column != 0) {
            setValueAt(value, ((Node)node).row, column);
        }
    }


// -- Delegate TreeModel to treeModel : ----------------------------------------

    @Override
    public Object getRoot() {
        return treeModel.getRoot();
    }

    @Override
    public Object getChild(Object parent, int index) {
        return treeModel.getChild(parent, index);
    }

    @Override
    public int getChildCount(Object parent) {
        return treeModel.getChildCount(parent);
    }

    @Override
    public boolean isLeaf(Object node) {
        return treeModel.isLeaf(node);
    }

    @Override
    public void valueForPathChanged(TreePath path, Object newValue) {
        treeModel.valueForPathChanged(path, newValue);
    }

    @Override
    public int getIndexOfChild(Object parent, Object child) {
        return treeModel.getIndexOfChild(parent, child);
    }

    @Override
    public void addTreeModelListener(TreeModelListener l) {
        treeModel.addTreeModelListener(l);
    }

    @Override
    public void removeTreeModelListener(TreeModelListener l) {
        treeModel.removeTreeModelListener(l);
    }
    
    
// -- Tree of channels : -------------------------------------------------------
    
    private class Node extends DefaultMutableTreeNode implements ChannelDisplay {
        
        private int row;
        private ChannelHandle channelHandle;
        
        Node(String name) {
            super(name);
        }

        public int getRow() {
            return row;
        }

        public void setRow(int row) {
            this.row = row;
        }

        public ChannelHandle getChannelHandle() {
            return channelHandle;
        }

        public void setChannelHandle(ChannelHandle channelHandle) {
            this.channelHandle = channelHandle;
        }

        @Override
        public void update(ChannelHandle channelHandle, List<String> attributes) {
            Set<MonitorField> fields = MonitorTable.translateAttributesToFields(attributes);
            int topChangedRow = Integer.MAX_VALUE;
            for (MonitorField field : fields) {
                int col = columns.indexOf(field);
                if (col == -1) break;
                Node node = this;
                while (node != root && format(cells.get(node.row).get(col+1))) {
                    if (node.row < topChangedRow) topChangedRow = row;
                    node = (Node) node.getParent();
                }
            }
            if (topChangedRow < Integer.MAX_VALUE) {
                TreeTable.this.fireTableRowsUpdated(topChangedRow, row);
                fireChangeEvent();
            }
        }
        
    }
    
// -- Formatting cells : --------------------------------------------------------

    @Override
    protected boolean format(Cell cell) {
        MonitorField field = cell.getField();
        List<AbstractMonitorView.ChannelHandle> handles = cell.getItems();
        Data newData;
        switch (handles.size()) {
            case 0:
                newData = new Data(field.getTitle(), SwingConstants.CENTER);
                break;
            case 1:
                newData = format(handles.get(0), field);
                break;
            default:
                ArrayList<Data> all = new ArrayList<>(handles.size());
                for (AbstractMonitorView.ChannelHandle h : handles) {
                    all.add(format(h, field));
                }
                newData = Data.merge(all);
        }
        if (newData.equals(cell.getData())) {
            return false;
        } else {
            cell.setData(newData);
            return true;
        }
    }
    
    
// -- Customized JXTreeTable (hack around hack around DefaultTableCellRenderer "color memory") : ---
    
    private static class Table extends JXTreeTable {
        Table(TreeTable model) {
            super(model);
        }
        @Override
        protected void resetDefaultTableCellRendererColors(Component renderer, int row, int column) {
        }
    }


// -- Saving/restoring : -------------------------------------------------------
    
//    public static class Descriptor implements Serializable {
//
//        private String expansion;
//
//        public String getExpansion() {
//            return expansion;
//        }
//
//        public void setExpansion(String expansion) {
//            this.expansion = expansion;
//        }
//        
//    }
//    
//    public Descriptor save() {
//        
//    }
//    
//    public void restore(Descriptor desc) {
//        
//    }
    

}
