package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.util.*;
import java.util.function.Function;
import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.subsystem.monitor.ui.tree.MonitorDisplay;

/**
 * Monitoring view that displays several other views in tabs.
 *
 * @author onoprien
 */
@ConsoleLookup(id="org.lsst.ccs.gconsole.plugins.monitor.MonitorView",
               name="Tabbed View",
               path="Built-In/Tabbed",
               description="Monitoring data view that displays its data in a set of tabs.")
public class TabbedView extends AbstractMonitorView {

// -- Fields : -----------------------------------------------------------------
    
    private final JComponent panel = new JPanel(new BorderLayout());
    private JTabbedPane tabbedPane;
    
    private final LinkedHashMap<String,AbstractMonitorView> views = new LinkedHashMap<>();
    
    private Function<String,AbstractMonitorView> viewFactory;
    
    

// -- Life cycle : -------------------------------------------------------------
    
    public TabbedView() {
        viewFactory = name -> new TreeView();
        panel.add(new MonitorDisplay.EMPTY());
    }
    
    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the view to be displayed in the tab {@code name}.
     * @param name
     * @param view 
     */
    public void setTab(String name, AbstractMonitorView view) {
        // FIXME
    }
    
    /**
     * Sets the factory used to create views for tabs for which the view has not been explicitly set.
     * @param factory Factory that returns view instance given the tab name.
     */
    public void setViewFactory(Function<String,AbstractMonitorView> factory) {
        viewFactory = factory;
    }
    
// -- Implementing MonitorView : -----------------------------------------------

    @Override
    public JComponent getPanel() {
        return panel;
    }
   
// -- Calling AbstractMonitorView hooks : --------------------------------------    

    @Override
    protected void connect(AgentInfo agent) {
        views.values().forEach(view -> view.connect(agent));
    }

    @Override
    protected void disconnect(AgentInfo agent) {
        views.values().forEach(view -> view.disconnect(agent));
    }

    @Override
    protected void addChannels(AgentInfo agent, Map<String, AgentChannel> channels) {
        TreeMap<String,Map<String, AgentChannel>> allTabs = new TreeMap<>();
        channels.forEach((path, ch) ->  {
            String[] ss = split(path);
            Map<String, AgentChannel> oneTab = allTabs.get(ss[0]);
            if (oneTab == null) {
                oneTab = new HashMap<>();
                allTabs.put(ss[0], oneTab);
            }
            oneTab.put(ss[1], ch);
        });
        allTabs.forEach((tab, cc) -> {
            AbstractMonitorView view = views.get(tab);
            if (view == null) {
                view = viewFactory.apply(tab);
                views.put(tab, view);
                switch (views.size()) {
                    case 0:
                        tabbedPane = null;
                        panel.removeAll();
                        panel.add(new MonitorDisplay.EMPTY());
                        break;
                    case 1:
                        tabbedPane = null;
                        panel.removeAll();
                        panel.add(view.getPanel());
                        break;
                    case 2:
                        panel.removeAll();
                        tabbedPane = new JTabbedPane();
                        views.forEach((name,v) -> tabbedPane.add(name, v.getPanel()));
                        panel.add(tabbedPane);
                        break;
                    default:
                        tabbedPane.add(tab, view.getPanel());
                }
            }
            view.addChannels(agent, cc);
        });
    }

    @Override
    protected void removeChannels(AgentInfo agent, List<String> paths) {
        TreeMap<String,List<String>> allTabs = new TreeMap<>();
        paths.forEach(path ->  {
            String[] ss = split(path);
            List<String> oneTab = allTabs.get(ss[0]);
            if (oneTab == null) {
                oneTab = new ArrayList<>();
                allTabs.put(ss[0], oneTab);
            }
            oneTab.add(ss[1]);
        });
        allTabs.forEach((tab, cc) -> {
            AbstractMonitorView view = views.get(tab);
            if (view != null) {
                view.removeChannels(agent, cc);
                if (view.isEmpty()) {
                    views.remove(tab);
                    switch (views.size()) {
                        case 0:
                            tabbedPane = null;
                            panel.removeAll();
                            panel.add(new MonitorDisplay.EMPTY());
                            break;
                        case 1:
                            tabbedPane = null;
                            panel.removeAll();
                            panel.add(views.values().iterator().next().getPanel());
                            break;
                        default:
                            tabbedPane.remove(view.getPanel());
                    }
                }
            }
        });
    }

    @Override
    protected void updateChannels(AgentInfo agent, Map<String, Map.Entry<AgentChannel, List<String>>> channels) {
        TreeMap<String,Map<String, Map.Entry<AgentChannel, List<String>>>> allTabs = new TreeMap<>();
        channels.forEach((path, e) ->  {
            String[] ss = split(path);
            Map<String, Map.Entry<AgentChannel, List<String>>> oneTab = allTabs.get(ss[0]);
            if (oneTab == null) {
                oneTab = new HashMap<>();
                allTabs.put(ss[0], oneTab);
            }
            oneTab.put(ss[1], e);
        });
        allTabs.forEach((tab, cc) -> {
            AbstractMonitorView view = views.get(tab);
            if (view != null) {
                view.updateChannels(agent, cc);
            }
        });
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    String[] split(String displayPath) {
        String[] out = new String[2];
        int i = displayPath.indexOf("//");
        if (i == -1) {
            out[0] = "";
            out[1] = displayPath;
        } else {
            out[0] = displayPath.substring(0, i);
            out[1] = displayPath.substring(i+2);
        }
        return out;
    }

}
