package org.lsst.ccs.gconsole.plugins.monitor;

import java.io.Serializable;
import javax.swing.JComponent;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusAggregator;
import org.lsst.ccs.gconsole.agent.AgentStatusListener;

/**
 * View of monitoring data.
 * {@code AgentStatusListener} methods are called on the {@code AgentStatusAggregator} thread.
 * All other methods declared by this interface should be called on AWT Event Dispatch Thread.
 *
 * @author onoprien
 */
public interface MonitorView extends AgentStatusListener {
    
// -- Getters and setters : ----------------------------------------------------
    
    /**
     * Returns the graphical component maintained by this view.
     * @return Graphical component to be displayed by the GUI.
     */
    JComponent getPanel();
    
    /**
     * Returns the name of this view.
     * @return  View name.
     */
    default String getName() {
        JComponent panel = getPanel();
        return panel == null ? "" : panel.getName();
    }
    
    /**
     * Sets the name of this view.
     * @param name View name.
     */
    default void setName(String name) {
        JComponent panel = getPanel();
        if (panel != null) panel.setName(name);
    }
    
    /**
     * Returns the filter associated with this view, or {@code null} if there is none.
     * @return Channels filter.
     */
    AgentChannelsFilter getFilter();
    
    /**
     * Sets the channels filter to be used by this view.
     * The filter determines what channels are displayed by this view, and how they are displayed.
     * @param filter Filter to be associated with this view.
     */
    void setFilter(AgentChannelsFilter filter);
    
    /**
     * Returns {@code true} if this view has no content to display at the moment.
     * This can be used report that the view can be closed, for example.
     * The default implementation returns {@code false}.
     * @return 
     */
    default boolean isEmpty() {
        return false;
    }
    
    
// -- Installation : -----------------------------------------------------------
    
    /**
     * Installs this view, connecting it to the data source.
     * // FIXME: given the changes in toolkit/ui, do I need to provide reference to aggregator? Review.
     * @param aggregator 
     */
    default void install(AgentStatusAggregator aggregator) {
        AgentChannelsFilter filter = getFilter();
        if (filter == null) {
            filter = AgentChannelsFilter.ALL;
            setFilter(filter);
        }
        aggregator.addListener(this, filter);
    }
    
    /**
     * Un-installs this view, disconnecting it from the data source.
     * // FIXME: given the changes in toolkit/ui, do I need to provide reference to aggregator? Review.
     * @param aggregator 
     */
    default void uninstall(AgentStatusAggregator aggregator) {
        aggregator.removeListener(this);
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    public interface Descriptor extends Serializable {
        
        /**
         * Returns a descriptor of the creator of the view.
         * @return Creator descriptor.
         */
        default Serializable getCreator() {return null;}
        
        /**
         * Sets a descriptor of the creator of the view.
         * @param creator Creator descriptor.
         */
        default void setCreator(Serializable creator) {}
        
        /**
         * Returns the name of the view.
         * @return View name.
         */
        default String getName() {
            return null;
        }
        
        /**
         * Sets the name of the view.
         * @param name View name.
         */
        default void setName(String name) {
        }
        
    }
    
    /**
     * Returns a descriptor that contains information required to re-create this view in its current state.
     * The default implementation returns {@code null}.
     * 
     * @return View descriptor, or {@code null} is this view does not support saving.
     */
    default Descriptor save() {
        return null;
    }
    
    /**
     * Restores this view to the state described by the provided descriptor, to the extent possible.
     * The default implementation does nothing.
     * 
     * @param descriptor View descriptor.
     */
    default void restore(Descriptor descriptor) {
    }
        
}
