package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Color;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelState;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView.ChannelHandle;
import org.lsst.ccs.gconsole.plugins.trending.TrendingService;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView.ChannelDisplay;

/**
 * 
 *
 * @author onoprien
 */
abstract public class MonitorTable extends AbstractTableModel implements ChannelDisplay {

// -- Fields : -----------------------------------------------------------------

    public static final List<MonitorField> defaultFields = Collections.unmodifiableList(Arrays.asList(MonitorField.VALUE, MonitorField.UNITS, MonitorField.LOW, MonitorField.ALERT_LOW, MonitorField.HIGH, MonitorField.ALERT_HIGH, MonitorField.DESCR));
    public static final List<MonitorField> defaultCompactFields = Collections.unmodifiableList(Arrays.asList(MonitorField.VALUE));
    
    public static final Color COLOR_FG = Color.BLACK; // default foreground
    public static final Color COLOR_BG = Color.WHITE; // default background
    
    private static final Color COLOR_GOOD = new Color(160, 255, 160);
    private static final Color COLOR_WARN = new Color(255, 255, 100);
    private static final Color COLOR_ERR = new Color(255, 160, 160);
    private static final Color COLOR_OFF = new Color(160, 200, 255);
    private static final Color COLOR_POPUP = new Color(255, 255, 160);
    private static final Color COLOR_MULTI = Color.LIGHT_GRAY;
    private static final Color COLOR_NA = COLOR_BG;
    
    private static final String TEXT_MULTI = "---";
    private static final int HA_MULTI = SwingConstants.CENTER;

    protected int nRows, nColumns;
    protected ArrayList<ArrayList<Cell>> cells; // [row][column], covers actual table
    
    protected ChangeListener listener;
    
    static private final List<MonitorField> AFFECTED_BY_STATE = Arrays.asList((new MonitorField[] {MonitorField.VALUE, MonitorField.HIGH, MonitorField.LOW}));
    protected final ChangeEvent CHANGE_EVENT = new ChangeEvent(this);

// -- Life cycle : -------------------------------------------------------------
    
    MonitorTable() {
    }
    
    MonitorTable(Cell[][] cells) {
        nRows = cells.length;
        nColumns = cells[0].length;
        this.cells = new ArrayList<>(nRows);
        for (Cell[] row : cells) {
            this.cells.add(new ArrayList<>(Arrays.asList(row)));
        }
    }
    
    public void destroy() {
    }
    
// -- Implementing TableModel : ------------------------------------------------

    @Override
    public int getRowCount() {
        return nRows;
    }

    @Override
    public int getColumnCount() {
        return nColumns;
    }

    @Override
    public Data getValueAt(int row, int column) {
        return cells.get(row).get(column).getData();
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        Cell cell = cells.get(row).get(column);
        if (cell.getItems().size() != 1) return false;
        MonitorField field = cell.getField();
        return field != null && field.isEditable();
    }

    @Override
    public void setValueAt(Object aValue, int row, int column) {
        Cell cell = cells.get(row).get(column);
        MonitorField field = cell.getField();
        boolean isHigh = MonitorField.HIGH.equals(field);
        if ((isHigh || MonitorField.LOW.equals(field)) && cell.getItems().size() == 1) {
            try {
                double newValue = Double.parseDouble(aValue.toString());
                AgentChannel channel = cell.getItems().get(0).getChannel();
                if (channel != null) {
//                    double oldValue = (Double) (isHigh ? channel.get(MonitorField.HIGH.name()) : channel.get(MonitorField.LOW.name()));
//                    if (Double.isNaN(oldValue) || (Math.abs(oldValue - newValue) > 0.00001)) { // FIXME
                        String target = isHigh ? "limitHi" : "limitLo";
                        Console.getConsole().sendCommand(channel.getAgentName() +"/change", channel.getLocalPath(), target, newValue);
//                    }
                }
            } catch (NumberFormatException x) {
                return;
            }
            super.setValueAt(getValueAt(row, column), row, column);
        } else {
            super.setValueAt(aValue, row, column);
        }
    }
    
    
// -- Updating table model : ---------------------------------------------------

    @Override
    public void update(ChannelHandle item, List<String> attributes) {
        Set<MonitorField> fields = translateAttributesToFields(attributes);
        fields.forEach(field -> {
            List<int[]> affectedCells = getCells(item, field);
            for (int[] index : affectedCells) {
                Cell cell = cells.get(index[0]).get(index[1]);
                field = cell.getField();
                List<ChannelHandle> items = cell.getItems();
                int nItems = items.size();
                if (cell.getData() == Data.MULTI) {
                    Data data = format(items.get(0), field);
                    for (int i = 1; i < nItems; i++) {
                        Data other = format(items.get(i), field);
                        if (!data.equals(other)) {
                            data = null;
                            break;
                        }
                    }
                    if (data != null) {
                        cell.setData(data);
                        fireTableCellUpdated(index[0], index[1]);
                    }
                } else {
                    Data data = format(item, field);
                    if (!data.equals(cell.getData())) {
                        cell.setData(data);
                        fireTableCellUpdated(index[0], index[1]);
                    }
                }
            }
        });
        fireChangeEvent();
    }

    /**
     * Returns indices of all cells whose content is affected by the specified item and field.
     * The implementation provided by this class is universal but very inefficient - subclasses
     * handling specific types of tables are expected to override it.
     * 
     * @param item ChannelHandle.
     * @param field Field. If {@code null}, all cells affected by the specified item are included.
     * @return List of cells.
     */
    protected List<int[]> getCells(ChannelHandle item, MonitorField field) {
        List<int[]> out = new ArrayList<>();
        for (int row = 0; row < nRows; row++) {
            for (int col = 0; col < nColumns; col++) {
                Cell cell = cells.get(row).get(col);
                if (cell.getItems().contains(item) && (field == null || field.equals(cell.getField()))) {
                    out.add(new int[] {row, col});
                }
            }
        }
        return out;
    }
    
// -- Formatting cells : --------------------------------------------------------

    protected Data format(ChannelHandle item, MonitorField field) {

        if (MonitorField.NAME.equals(field)) {
            return new Data(item.getPath(), null, null, SwingConstants.LEFT);
        }

        AgentChannel channel = item.getChannel();
        if (channel == null) {
            return Data.NA;
        }
        Object value = channel.get(field.name());
        if (value == null) {
            return Data.NA;
        }
        
        String text = null;
        Color fgColor = null;
        Color bgColor = null;
        Integer horizontalAlignment = null;

        if (MonitorField.VALUE.equals(field)) {
            if (value instanceof Double) {
                if (((Double) value).isNaN()) {
                    text = "NaN";
                } else {
                    double d = (Double) value;
                    String f = (String) channel.get(MonitorField.FORMAT_KEY);
                    text = String.format((f == null ? "%f " : f), d);
                }
            } else {
                text = value.toString();
            }
            Object o = channel.get(MonitorField.STATE_KEY);
            if (o instanceof AgentChannelState) {
                AgentChannelState state = (AgentChannelState) o;
                if (state.isOnline()) {
                    bgColor = state.isGood() ? COLOR_GOOD : COLOR_ERR;
                } else {
                    bgColor = COLOR_OFF;
                }
            }
            horizontalAlignment = SwingConstants.RIGHT;

        } else if (MonitorField.HIGH.equals(field) || MonitorField.LOW.equals(field)) {
            double d = Double.NaN;
            if (value instanceof Double) {
                d = ((Double)value);
            } else if (value instanceof Float) {
                d = ((Float)value).doubleValue();
            } else if (value instanceof String) {
                try {
                    d = Double.parseDouble((String)value);
                } catch (NumberFormatException x) {
                    text = (String)value;
                }
            }
            if (Double.isNaN(d)) {
                if (text == null) text = "NaN";
            } else {
                String f = (String) channel.get(MonitorField.FORMAT_KEY);
                text = String.format((f == null ? "%f " : f), d);
            }
            Object o = channel.get(MonitorField.STATE_KEY);
            if (o instanceof AgentChannelState) {
                AgentChannelState state = (AgentChannelState) o;
                if (MonitorField.HIGH.equals(field) ? state.isHighLimitChange() : state.isLowLimitChange()) {
                    fgColor = Color.BLUE;
                }
            }

        } else if (MonitorField.LOW_WARN.equals(field) || MonitorField.HIGH_WARN.equals(field)) {
            if (value instanceof Double) {
                if (((Double) value).isNaN()) {
                    text = "NaN";
                } else {
                    double d = (Double) value;
                    String f = (String) channel.get(MonitorField.FORMAT_KEY);
                    text = f == null ? String.valueOf(d) : String.format(f, d);
                }
            } else {
                text = value.toString();
            }

        } else if (MonitorField.ALERT_HIGH.equals(field) || MonitorField.ALERT_LOW.equals(field)) {
            text = "  \u2713";

//            } else if (MonitorField.DESCR.equals(field)) {
//                
//            } else if (MonitorField.UNITS.equals(field)) {
//                
        } else {
            text = value.toString();
        }

        return new Data(text, fgColor, bgColor, horizontalAlignment);
    }

    protected boolean format(Cell cell) {
        MonitorField field = cell.getField();
        List<AbstractMonitorView.ChannelHandle> handles = cell.getItems();
        Data newData;
        switch (handles.size()) {
            case 0:
                newData = new Data(field.getTitle(), SwingConstants.CENTER);
                break;
            case 1:
                newData = format(handles.get(0), field);
                break;
            default:
                ArrayList<Data> all = new ArrayList<>(handles.size());
                for (AbstractMonitorView.ChannelHandle h : handles) {
                    all.add(format(h, field));
                }
                newData = Data.merge(all);
        }
        if (newData.equals(cell.getData())) {
            return false;
        } else {
            cell.setData(newData);
            return true;
        }
    }
    
    protected boolean format(Cell cell, ChannelHandle item) {
        return format(cell);
    }

        
// -- Cell content classes : ---------------------------------------------------

    protected static class Cell {
        
        static protected Cell EMPTY = new Cell(Collections.emptyList(), MonitorField.NULL, Data.EMPTY);
        
        private final List<ChannelHandle> items;
        private final MonitorField field;
        
        private Data data;
                
        Cell() {
            this(Collections.emptyList(), MonitorField.NULL);
        }
        
        Cell(ChannelHandle item, MonitorField field) {
            this(item == null ? Collections.emptyList() : Collections.singletonList(item), field);
        }

        Cell(List<? extends ChannelHandle> items, MonitorField field) {
            this(items, field, null);
        }

        Cell(List<? extends ChannelHandle> items, MonitorField field, Data data) {
            this.items = items == null ? Collections.emptyList() : (List<ChannelHandle>) items;
            this.field = field == null ? MonitorField.NULL : field;
            this.data = data;
        }
        
        public List<ChannelHandle> getItems() {
            return items;
        }
        
        public MonitorField getField() {
            return field;
        }
        
        public void setData(Data data) {
            this.data = data;
        }
        
        public Data getData() {
            return data;
        }
        
    }
    
    protected static class Data {
        
        static final Data MULTI = new Data("-", COLOR_FG, COLOR_MULTI, SwingConstants.CENTER);
        static final Data NA = new Data("", COLOR_FG, COLOR_NA, SwingConstants.CENTER);
        static final Data EMPTY = new Data("", COLOR_FG, COLOR_BG, SwingConstants.CENTER);

        final String text;
        final Color fgColor, bgColor;
        final int horizontalAlignment;
        final String toolTip;
        
        Data(String text, Color fgColor, Color bgColor, Integer horizontalAlignment, String toolTip) {
            this.text = text == null ? "" : text;
            this.fgColor = fgColor == null ? COLOR_FG : fgColor;
            this.bgColor = bgColor == null ? COLOR_BG : bgColor;
            this.horizontalAlignment = horizontalAlignment == null ? SwingConstants.CENTER : horizontalAlignment;
            this.toolTip = toolTip;
        }
        
        Data(String text, Color fgColor, Color bgColor, Integer horizontalAlignment) {
            this(text, fgColor, bgColor, horizontalAlignment, null);
        }
        
        Data() {
            this(null, null, null, null);
        }
        
        Data(String text, int horizontalAlignment) {
            this(text, null, null, horizontalAlignment, null);
        }
        
        Data(String text, int horizontalAlignment, String toolTip) {
            this(text, null, null, horizontalAlignment, toolTip);
        }

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof Data)) return false;
            Data other = (Data) obj;
            return Objects.equals(text, other.text) && Objects.equals(fgColor, other.fgColor) && Objects.equals(bgColor, other.bgColor) && Objects.equals(horizontalAlignment, other.horizontalAlignment);
        }

        @Override
        public int hashCode() {
            final int PRIME = 31;
            int result = 1;
            result = result * PRIME + (text == null ? 0 : text.hashCode());
            result = result * PRIME + (fgColor == null ? 0 : fgColor.hashCode());
            result = result * PRIME + (bgColor == null ? 0 : bgColor.hashCode());
            result = result * PRIME + horizontalAlignment;
            return result;
        }

        @Override
        public String toString() {
            return text;
        }
        
        static public Data merge(List<Data> dataset) {
            switch (dataset.size()) {
                case 0:
                    return EMPTY;
                case 1:
                    return dataset.get(0);
                default:
                    Iterator<Data> it = dataset.iterator();
                    Data first = it.next();
                    String text = first.text;
                    Color fgColor = first.fgColor;
                    Color bgColor = first.bgColor;
                    int horizontalAlignment = first.horizontalAlignment;
                    while (it.hasNext()) {
                        Data other = it.next();
                        if (!text.equals(other.text)) text = TEXT_MULTI;
                        fgColor = merge(fgColor, other.fgColor, COLOR_FG);
                        bgColor = merge(bgColor, other.bgColor, COLOR_BG);
                        if (horizontalAlignment !=other.horizontalAlignment) horizontalAlignment = HA_MULTI;
                    }
                    return new Data(text, fgColor, bgColor, horizontalAlignment);
            }
        }
        
        private static Color merge(Color c1, Color c2, Color defaultColor) {
            if (c1.equals(c2) || c2.equals(defaultColor)) {
                return c1;
            } else if (c1.equals(defaultColor)) {
                    return c2;
            } else {
                return COLOR_MULTI;
            }
        }
    }
    
    
// -- Handling change listener : -----------------------------------------------
    
    public void setChangeListener(ChangeListener listener) {
        this.listener = listener;
    }
    
    protected void fireChangeEvent() {
        if (listener != null) listener.stateChanged(CHANGE_EVENT);
    }
    
    
// -- Customized JTable : ------------------------------------------------------
    
    public JTable getTable() {
        
        JTable table = makeTable();
        
        JTableHeader header = table.getTableHeader();
        header.setReorderingAllowed(false);
        header.setResizingAllowed(false); // no resizing to avoid mismatch between renderer and editor tables
        TableCellRenderer headerRenderer = header.getDefaultRenderer();
        if (headerRenderer instanceof DefaultTableCellRenderer) {
            ((DefaultTableCellRenderer)headerRenderer).setHorizontalAlignment(SwingConstants.CENTER);
        }

        Renderer renderer = new Renderer();
        table.setDefaultRenderer(Object.class, renderer);
        table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
        table.setRowSelectionAllowed(false);
        table.setColumnSelectionAllowed(false);
        table.setRowHeight(table.getRowHeight() + 2);
        table.setShowGrid(true);
        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent evt) {
                tableMouseClicked(evt);
            }
        });

        for (int col = 0; col < nColumns; col++) {
            TableColumn column = table.getColumnModel().getColumn(col);
            Component comp = headerRenderer.getTableCellRendererComponent(null, column.getHeaderValue(), false, false, 0, 0);
            int maxWidth = comp.getPreferredSize().width;
            for (int row=0; row<nRows; row++) {
                comp = renderer.getTableCellRendererComponent(table, getValueAt(row, col), false, false, row, col);
                maxWidth = Math.max(maxWidth, comp.getPreferredSize().width);
            }
            column.setPreferredWidth(maxWidth);
        }
        return table;
    }
    
    protected JTable makeTable() {
        return new JTable(this);
    }
    
    protected void tableMouseClicked(MouseEvent evt) {
        int nClick = evt.getClickCount();
        JTable table = (JTable) evt.getSource();
        Point point = evt.getPoint();
        int row = table.rowAtPoint(point);
        int column = table.columnAtPoint(point);
        if (nClick == 1) {
            // FIXME
        } else if (nClick == 2) {
            Cell cc = cells.get(row).get(column);
            if (cc.getField().equals(MonitorField.VALUE)) {
                List<ChannelHandle> items = cc.getItems();
                if (items.size() == 1) {
                    ChannelHandle item = items.get(0);
                    String[] path = {item.getChannel().getAgentName(), item.getChannel().getLocalPath()};
                    TrendingService trending = (TrendingService) Console.getConsole().getConsoleLookup().lookup(TrendingService.class);
                    if (trending == null) {
                        return;
                    }
                    trending.show(path);
                }
            }
        }
    }
    
    static class Renderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            Data d = (Data) value;
            super.getTableCellRendererComponent(table, d.text, false, false, row, column);
            setBackground(d.bgColor);
            setForeground(d.fgColor);
            setHorizontalAlignment(d.horizontalAlignment);
            setToolTipText(d.toolTip);
            return this;
        }
    }
    
    public boolean showHeader() {
        return true;
    }
    
    
// -- Static utilities : -------------------------------------------------------
    
    static public List<MonitorField> translateAttributesToFields(String attribute) {
        if (MonitorField.STATE_KEY.equals(attribute)) {
            return AFFECTED_BY_STATE;
        } else {
            return Collections.singletonList(MonitorField.valueOf(attribute));
        }
    }
    
    static public HashSet<MonitorField> translateAttributesToFields(Collection<String> attributes) {
        HashSet<MonitorField> fields = new HashSet<>();
        if (attributes == null) {
            fields.add(null);
        } else {
            attributes.forEach(att -> {
                List<MonitorField> f = translateAttributesToFields(att);
                fields.addAll(f); 
            });
        }
        return fields;
    }
    
    static public List<MonitorField> trimAbsentFields(List<MonitorField> fields, Collection<ChannelHandle> channels, Collection<MonitorField> exclude) {
        LinkedHashSet<MonitorField> requestedFields = new LinkedHashSet<>(fields);
        if (exclude != null) {
            requestedFields.removeAll(exclude);
        }
        HashSet<MonitorField> presentFields = new HashSet<>();
        for (ChannelHandle ch : channels) {
            AgentChannel channel = ch.getChannel();
            if (channel != null) {
                Iterator<MonitorField> it = requestedFields.iterator();
                while (it.hasNext()) {
                    MonitorField f = it.next();
                    if (channel.get(f.name()) != null) {
                        presentFields.add(f);
                        it.remove();
                    }
                }
                if (requestedFields.isEmpty()) {
                    break;
                }
            }
        }
        if (fields.size() == presentFields.size()) {
            return fields;
        } else {
            ArrayList<MonitorField> out = new ArrayList<>(presentFields.size());
            for (MonitorField f : fields) {
                if (presentFields.contains(f)) {
                    out.add(f);
                }
            }
            return out;
        }
    }
    
    static public List<MonitorField> trimAbsentFields(List<MonitorField> fields, Collection<ChannelHandle> channels) {
        HashSet<MonitorField> presentFields = new HashSet<>();
        for (ChannelHandle ch : channels) {
            AgentChannel channel = ch.getChannel();
            if (channel != null) {
                for (MonitorField f : fields) {
                    if (channel.get(f.name()) != null) {
                        presentFields.add(f);
                    }
                }
            }
            if (presentFields.size() == fields.size()) {
                return fields;
            }
        }
        if (presentFields.size() > 0) {
            ArrayList<MonitorField> out = new ArrayList<>(fields);
            out.retainAll(presentFields);
            ((ArrayList<MonitorField>) out).trimToSize();
            return out;
        } else {
            return fields;
        }
    }
    
    static public List<MonitorField> moveNameToFront(List<MonitorField> fields) {
        if (fields.isEmpty()) {
            return Collections.singletonList(MonitorField.NAME);
        } else if (MonitorField.NAME.equals(fields.get(0))){
            return fields;
        } else {
            ArrayList<MonitorField> out = new ArrayList<>(fields.size()+1);
            out.add(MonitorField.NAME);
            for (MonitorField f : fields) {
                if (!MonitorField.NAME.equals(f)) {
                    out.add(f);
                }
            }
            out.trimToSize();
            return out;
        }
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * Returns JavaBean that contains data required to restore this table to its current state.
     * The default implementation provided by this class returns {@code null}, subclasses should override
     * if they need to support saving/restoring their state.
     * 
     * @return JavaBean that contains data required to restore this table to its current state.
     */
    public Serializable save() {
        return null;
    }
    
    /**
     * Restores this table to the state described by the provided object, to the extent possible.
     * The default implementation provided by this class does nothing.
     * 
     * @param storageBean JavaBean that contains data describing the state of this table.
     */
    public void restore(Serializable storageBean) {
    }

}
