package org.lsst.ccs.gconsole.plugins.demo.monitor;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.plugins.monitor.CompoundView;
import org.lsst.ccs.gconsole.plugins.monitor.ConfigView;
import org.lsst.ccs.gconsole.plugins.monitor.LsstMonitorPlugin;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.TreeView;

/**
 * New style monitoring GUI demo.
 * <p>
 * This plugin demonstrates two ways to define console panels that display monitoring data:
 * <ul>
 * <li><b>Embedded monitoring view.</b>
 * This plugin creates a custom GUI panel and opens it as a console page. Inside
 * this panel, there is a monitoring view (implements {@link MonitorView}). The 
 * view is submitted to the framework for management by calling 
 * {@link LsstMonitorPlugin.addView LsstMonitorPlugin.addView(...)} with a filter
 * that makes the view to display channels from all subsystems with values out of allowed range.
 * <li><b>Compound monitoring view that occupies the whole console page.</b>
 * This plugin creates a compound view that displays a data table tree and a control
 * panel. The view is displayed as a separate console page by calling
 * {@link LsstMonitorPlugin.addView LsstMonitorPlugin.openPage(...)}.
 * The filter associated with the page makes it display a fixed set of channels.
 * </ul>
 * 
 * 
 */
//@Plugin(name = "New Style Monitoring GUI Demo",
//        id="new-mon-demo",
//        description = "New style monitoring GUI demo plugin")
public class NewStylePlugin extends ConsolePlugin {

// -- Fields : -----------------------------------------------------------------
    
    private Action actDemo1, actDemo2;
    private JComponent page1;
    
    private MonitorView embeddedView;
    private CompoundView compoundView;

// -- Life cycle : -------------------------------------------------------------
    
    /** Called by the framework when the console starts, to initialize this plugin. */
    @Override
    public void initialize() {
        
        actDemo1 = new AbstractAction("Monitor GUI 1") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean)getValue(Action.SELECTED_KEY)) {
                    startDemo1();
                } else {
                    stopDemo1();
                }
            }
        };
        actDemo1.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(actDemo1, "Demo", "Monitoring");
        
        actDemo2 = new AbstractAction("Monitor GUI 2") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean)getValue(Action.SELECTED_KEY)) {
                    startDemo2();
                } else {
                    stopDemo2();
                }
            }
        };
        actDemo2.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(actDemo2, "Demo", "Monitoring");

    }
    
    private void startDemo1() {
        actDemo1.putValue(Action.SELECTED_KEY, true);
        if (page1 == null) {
            
            // Create monitor view:
            
            embeddedView = new TreeView(); // FUXME: how do I pass sender?
            embeddedView.setFilter(new FilterOutOfRange());
            
            // Create an arbitrary GUI and add the monitor view as one of its panels:
            
            page1 = new JPanel(new BorderLayout());
            page1.add(new JLabel("The pnanel below is an embedded view"), BorderLayout.NORTH);
            page1.add(embeddedView.getPanel(), BorderLayout.CENTER);
            
            // Open GUI in a graphical console page:
            
            PanelManager pm = getConsole().getPanelManager();
            pm.open(page1, "GUI with embedded monitoring view");
            pm.addListener(e -> {
                if (e.hasKey(Panel.OPEN) && (Boolean) e.getNewValue() == false) {
                    stopDemo1();
                }
            }, page1);
            
            // Submit the monitor view for updating:
            
            embeddedView.install(getConsole().getStatusAggregator());
        }
    }
    
    private void stopDemo1() {
        actDemo1.putValue(Action.SELECTED_KEY, false);
        if (page1 != null) {
            JComponent page = page1;
            page1 = null;
            embeddedView.uninstall(getConsole().getStatusAggregator());
            getConsole().getPanelManager().close(page);
        }
    }
    
    private void startDemo2() {
        actDemo2.putValue(Action.SELECTED_KEY, true);
        if (compoundView == null) {
            
            // Create a compound monitoring view:
            
            JPanel page2 = new JPanel(new BorderLayout());
            MonitorView treeView = new TreeView(); // FIXME
            page2.add(treeView.getPanel(), BorderLayout.CENTER);
            MonitorView controlView = new ConfigView(); // FIXME
            page2.add(controlView.getPanel(), BorderLayout.SOUTH);
            compoundView = new CompoundView(page2);
            compoundView.addView(treeView, false);
            compoundView.addView(controlView, false);
            compoundView.setFilter(FilterFixed.getInstance(1, "NAME", true, AlertState.NOMINAL));
            
            // Open the view as a graphical console page
            
            LsstMonitorPlugin monPlugin = (LsstMonitorPlugin) getConsole().getConsoleLookup().lookup(LsstMonitorPlugin.class);
            monPlugin.openPage(compoundView);
        }
    }

    private void stopDemo2() {
        actDemo2.putValue(Action.SELECTED_KEY, false);
        if (compoundView != null) {
            MonitorView view = compoundView;
            compoundView = null;
            getConsole().getPanelManager().close(view.getPanel());
        }
    }
    
}
