package org.lsst.ccs.gconsole.base;

import java.awt.event.ActionEvent;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.SwingUtilities;
import org.freehep.application.studio.Studio;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;
import org.lsst.ccs.gconsole.agent.AgentStatusListener;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.jas3.Jas3Console;

/**
 * 
 * - Maintains a list of actions
 * - Listens to agents connects/disconnects.
 * - On connection of an eligible agent:
 *   - creates an action if it does not already exists.
 *   - adds action to menu.
 *   - if auto-open is on, executes action
 * - On disconnection:
 * 
 *
 * @author onoprien
 */
class OptionalPageHandler implements AgentStatusListener {

// -- Fields : -----------------------------------------------------------------
    
    private final OptionalPage descriptor;
    private final Console console;
    
    private final TreeMap<String,Page> name2page = new TreeMap<>();

// -- Life cycle : -------------------------------------------------------------
    
    OptionalPageHandler(OptionalPage descriptor, Console console) {
        this.descriptor = descriptor;
        this.console = console;
    }
    
    void start() {
        console.getStatusAggregator().addListener(this, null, Collections.emptyList());
    }
    
    void stop() {
        console.getStatusAggregator().removeListener(this);
        SwingUtilities.invokeLater(() -> {
            new ArrayList<>(name2page.values()).forEach(page -> page.uninstall());
        });
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    OptionalPage getDescriptor() {
        return descriptor;
    }
    
    
// -- Listening to agent connects/disconnects : --------------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> onConnect(event.getSource()));
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> onDisconnect(event.getSource()));
    }
    
    private void onConnect(AgentInfo agent) {
        String name = descriptor.getPage(agent);
        if (name == null) return;
        if (name.trim().isEmpty()) name = agent.getName();
        Page page = name2page.get(name);
        if (page == null) {
            page = new Page(name);
            page.install();
        }
        page.agents.add(agent);
        if (page.isActive()) {
            page.open(descriptor.open(agent, page.component));
        }
    }

    private void onDisconnect(AgentInfo agent) {
        for (Page page : name2page.values()) {
            Iterator<AgentInfo> it = page.agents.iterator();
            while (it.hasNext()) {
                AgentInfo a = it.next();
                if (a.getName().equals(agent.getName())) {
                    it.remove();
                    boolean close = descriptor.agentDisconnected(a);
                    if (page.agents.isEmpty()) {
                        page.uninstall();
                    } else if (close) {
                        console.getPanelManager().close(page.component);
                    }
                    return;
                }
            }
        }
    }
    
    
// -- Handle for a specific page : ---------------------------------------------
    
    private class Page extends AbstractAction {
        
        final String name;
        JComponent component;
        final ArrayList<AgentInfo> agents = new ArrayList<>(1);
        
        Page(String name) {
            super(name);
            this.name = name;
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            if (isActive()) {
                component = null;
                for (AgentInfo agent : agents) {
                    open(descriptor.open(agent, component));
                }
            } else {
                if (component != null) console.getPanelManager().close(component);
            }
        }
        
        boolean isActive() {
            return (Boolean) getValue(Action.SELECTED_KEY);
        }
        
        /**
         * Called when this page is first created.
         * Initializes itself. Adds itself to the menu, to the list of pages.
         */
        void install() {
            putValue(Action.SELECTED_KEY, descriptor.isAutoOpen());
            name2page.put(name, this);
            console.addMenu(this, null, ("Window:1:1/CCS/"+ descriptor.getPath()).split("/"));
        }
        
        /**
         * Called to make this page visible.
         */
        void open(JComponent component) {
            if (this.component != null || component == null) return;
            this.component = component;
            TreeMap<Object,Object> par = new TreeMap<>();
            par.put(Panel.TITLE, name);
            Consumer<JComponent> onClose = c -> close();
            par.put(Panel.ON_CLOSE, onClose);
            console.getPanelManager().open(component, par);
        }
        
        /**
         * Called when this page is closed.
         */
        void close() {
            putValue(Action.SELECTED_KEY, false);
            descriptor.close(name);
        }
        
        /**
         * Called when this page is removed.
         */
        void uninstall() {
            console.removeMenu(("Window/CCS/"+ descriptor.getPath() +"/"+ name).split("/"));
            if (component != null) {
                console.getPanelManager().close(component);
            }
            name2page.remove(name);
        }

    }
    
}
