package org.lsst.ccs.gconsole.base;

import javax.swing.JComponent;
import org.lsst.ccs.bus.data.AgentInfo;

/**
 * Descriptor of a page that can be automatically opened by the graphical console.
 * See {@link Console#addOptionalPage(OptionalPage)}.
 *
 * @author onoprien
 */
public interface OptionalPage {
    
    /**
     * Returns a unique slash-separated path that determines placement of the menu item that opens this page.
     * @return Menu path.
     */
    String getPath();
    
    /**
     * Returns the default auto-open value. 
     * If {@code true}, the page(s) will be open automatically on compatible subsystem discovery.
     * <p>
     * The default implementation returns {@code true}.
     * 
     * @return Default {@code auto-open} value.
     */
    default boolean isAutoOpen() {
        return true;
    }
    
    /**
     * Called whenever a new subsystem is detected on the buses. 
     * The function should return the page name if the subsystem is compatible with this optional page. 
     * If not, it should return {@code null}. If an empty string is returned, the subsystem name is used.
     * <p>
     * If {@code auto-open} is {@code true}, the {@link open(...)} function will be called immediately
     * following this call. Otherwise, a menu item with the page name is added to
     * the graphical console "Window" menu with a path determined by {@link getPath()}.
     * 
     * @param agent Agent that has been discovered on the buses.
     * @return Page name.
     */
    String getPage(AgentInfo agent);
   
    /**
     * Called when a page is being opened.
     * The function should return the graphical component to be displayed. 
     * If {@code null} is returned, the page is not displayed.
     * If a page with the name returned by {@link #getPage(AgentInfo)} already exists
     * (this can happen if a single page communicates with multiple subsystems),
     * it is passed to this method as {@code existingComponent}. In this case, the
     * component returned by this method is ignored.
     * 
     * @param agent Agent.
     * @param existingComponent Existing component with the specified name.
     * @return Graphical component to be open.
     */
    JComponent open(AgentInfo agent, JComponent existingComponent);
    
    /**
     * Called whenever an agent for which a page has been opened disconnects from the buses.
     * Should return {@code true} to request closing the page. Regardless of the return value,
     * the page is closed if there are no more active agents associated with it.
     * <p>
     * The default implementation returns {@code false}.
     * 
     * @param agent Agent.
     * @return {@code true} to close the page.
     */
    default boolean agentDisconnected(AgentInfo agent) {
        return false;
    }
    
    /**
     * Called when the page is closed by the user.
     * 
     * @param page Name of the closed page.
     */
    default void close(String page) {
    }
    
}
