package org.lsst.ccs.gconsole.plugins.trending;

import java.util.*;

/**
 * Filter that modifies display paths and titles of trending channels.
 *
 * @author onoprien
 */
public class RenamingFilter implements TrendingFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final Map<TrendingChannel, String> nameMap = new HashMap<>();

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs the filter.
     * @param nameMap Map of channels available from the source produced by this filter to original paths.
     */
    public RenamingFilter(Map<TrendingChannel, String> nameMap) {
        this.nameMap.putAll(nameMap);
    }
    
    /**
     * Constructs the filter.
     * Each element of the collection passed to this method should be a 1, 2, or 3 element array.
     * The first element is the original (unfiltered) path associated with a trending channel.
     * The second element is the new path (if omitted, the channel retains its original path).
     * The third element is the title (if omitted, the path will be used as the title).
     * 
     * @param entries Mapping of original paths to filtered path and titles.
     */
    public RenamingFilter(Collection<String[]> entries) {
        entries.forEach(e -> {
            int n = e.length;
            switch (n) {
                case 1:
                    nameMap.put(new TrendingChannel(e[0]), e[0]); break;
                case 2:
                    nameMap.put(new TrendingChannel(e[1]), e[0]); break;
                case 3:
                    nameMap.put(new TrendingChannel(e[1], e[2]), e[0]); break;
                default:
                    throw new IllegalArgumentException("Mapping should be in {original_path, path [,title]} format.");
            }
        });
    }

// -- Filtering : --------------------------------------------------------------

    /**
     * Applies the filter.
     * @param source The original source.
     * @return The filtered source.
     */
    @Override
    public TrendingSource filter(TrendingSource source) {
        return new Source(source);
    }

// -- Trending source created by this filter : ---------------------------------

    private class Source extends TrendingSource implements TrendingSource.Listener {
        
        private final TrendingSource originalSource;
        private final Map<TrendingChannel,TrendingChannel> filtered2original = new HashMap<>();
        
        private Source(TrendingSource source) {
            originalSource = source;
            populate();
        }

        @Override
        synchronized public List<TrendingChannel> getChannels() {
            return new ArrayList<>(filtered2original.keySet());
        }

        @Override
        public TrendData get(TrendingChannel channel, long begin, long end, EnumSet<Trend.Meta> metadata, TrendData history) {
            TrendingChannel original;
            synchronized (this) {
                original = filtered2original.get(channel);
            }
            return original == null ? null : originalSource.get(original, begin, end, metadata, history);
        }

        @Override
        synchronized public void processEvent(Event event) {
            if (event.getSource() != originalSource) return;
            HashSet<TrendingChannel> before, removed, added;
            synchronized (this) {
                before = new HashSet<>(filtered2original.keySet());
                filtered2original.clear();
                populate();
                removed = new HashSet<>(before);
                removed.removeAll(filtered2original.keySet());
                added = new HashSet<>(filtered2original.keySet());
                added.removeAll(before);
            }
            fireEvent(new ArrayList<>(removed), new ArrayList<>(added));
        }
        
        synchronized private void populate() {
            HashMap<String,TrendingChannel> inputMap = new HashMap<>();
            originalSource.getChannels().forEach(channel -> inputMap.put(channel.getPath(), channel));
            nameMap.forEach((filtered, originalPath) -> {
                TrendingChannel original = inputMap.get(originalPath);
                if (original != null) {
                    filtered2original.put(filtered, original);
                }
            });
        }
        
    }

}
