package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.io.OutputStream;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.AbstractCellEditor;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeCellEditor;
import javax.swing.tree.TreePath;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.subsystem.monitor.ui.tree.MonitorDisplay;

/**
 * {@link MonitorView} that displays monitoring data as a tree of tables. All
 * access to instances of this class, including construction, should happen on
 * the EDT.
 * <p>
 * <h3>Implementation notes</h3>
 * This class maintains "data tree" (instances of {@code DataNode} and {@code DataNodeLeaf})
 * that represents hierarchy of data channels displayed by this view. The leaves correspond
 * to specific channels, non-leaf nodes have flags associated with them that affect the
 * display of everything below them.
 * <p>
 * Separately, there is a SWING JTree displayed by the GUI.Its table consists of 
 * {@code DefaultMutableTreeNode} instances that wrap either {@code DataNode} or {@code TableHolder}
 * as their "user objects".
 * <p>
 * Each {@code DataNode} that is not below a flattened {@code DataNode} is associated a JTree node.
 * Each JTree node that corresponds to a flattened {@code DataNode} has a child that
 * wraps a {@code TableHolder}.
 *
 * @author onoprien
 */
@ConsoleLookup(id="org.lsst.ccs.gconsole.plugins.monitor.MonitorView",
               name="Tree View",
               path="Built-In/Tree",
               description="Monitoring data view that displays its data in a tree of tables.")
public class TreeView extends AbstractMonitorView {

// -- Fields : -----------------------------------------------------------------

    private final Tree jTree;
    private final JScrollPane scrollPane;

    private List<MonitorField> columns = Arrays.asList(MonitorField.NAME, MonitorField.VALUE, MonitorField.UNITS, MonitorField.LOW, MonitorField.ALERT_LOW, MonitorField.HIGH, MonitorField.ALERT_HIGH, MonitorField.DESCR);
    private List<MonitorField> compactColumns = Arrays.asList(MonitorField.NAME, MonitorField.VALUE);
    
    private final DataNode dataRoot = new DataNode("root");
    private final LinkedHashMap<String,DataNodeLeaf> path2data = new LinkedHashMap<>(); // channels in addition order

// -- Life cycle : -------------------------------------------------------------
    
    public TreeView() {
        jTree = new Tree();
        scrollPane = new JScrollPane(jTree);
    }

// -- Getters and setters : ----------------------------------------------------

// <editor-fold defaultstate="collapsed">
    
    /**
     * Returns the graphical component maintained by this view.
     * @return Graphical component to be displayed by the GUI.
     */
    @Override
    public JComponent getPanel() {
        return scrollPane;
    }

    /**
     * Called to set a filter on this view.
     * Currently, the following is done in addition to saving e reference to the filter:<ul>
     * <li>Set monitor fields displayed by this view.
     * </ul>
     * @param filter Filter to use.
     */
    @Override
    public void setFilter(AgentChannelsFilter filter) {
        super.setFilter(filter);
        List<String> fields = filter.getFields(true);
        if (fields != null) {
            compactColumns = new ArrayList<>(fields.size());
            for (String field : fields) {
                compactColumns.add(MonitorField.getInstance(field));
            }
        }
        fields = filter.getFields(false);
        if (fields != null) {
            columns = new ArrayList<>(fields.size());
            for (String field : fields) {
                columns.add(MonitorField.getInstance(field));
            }
        }
    }
    
    @Override
    public boolean isEmpty() {
        return path2data.isEmpty();
    }
    
// </editor-fold>        

// -- Updates from status aggregator : -----------------------------------------

// <editor-fold defaultstate="collapsed">
    
    /** Updates {@code path2data} and calls {@code setChannels()}. */
    @Override
    protected void addChannels(AgentInfo agent, Map<String, AgentChannel> channels) {
        boolean modified = false;
        for (Map.Entry<String, AgentChannel> e : channels.entrySet()) {
            String path = e.getKey();
            int i = path.lastIndexOf("/");
            DataNodeLeaf leaf = new DataNodeLeaf(path.substring(i+1), e.getValue());
            modified = (path2data.put(path, leaf) == null) || modified;
        }
        if (modified) setChannels();
    }

    /** Updates {@code path2data} and calls {@code setChannels()}. */
    @Override
    protected void removeChannels(AgentInfo agent, List<String> paths) {
        boolean modified = false;
        for (String path : paths) {
            modified = (path2data.remove(path) != null) || modified;
        }
        if (modified) setChannels();
    }

    /** Forwards updates to leaves. */
    @Override
    protected void updateChannels(AgentInfo agent, Map<String, Map.Entry<AgentChannel, List<String>>> channels) {
        channels.forEach((path, e) -> {
            DataNodeLeaf leaf = path2data.get(path);
            if (leaf != null) {
                if (!Objects.equals(leaf.channel, e.getKey())) {
                    leaf.channel = e.getKey();
                    leaf.update(null);
                } else {
                    leaf.update(e.getValue());
                }
            }
        });
    }
    
    /** Rebuilds all trees based on {@code path2data}. */
    private void setChannels() {
        String treeStateString = saveTreeState();
        buildDataTree();
        Map<String,String> treeStateMap = restoreDataTreeState(treeStateString);
        DefaultMutableTreeNode root = buildJTreeModel(dataRoot, true);
        jTree.setModel(new DefaultTreeModel(root));
        restoreJTreeState(treeStateMap, dataRoot, "");
    }
    
// </editor-fold>        

// -- Data tree : --------------------------------------------------------------

// <editor-fold defaultstate="collapsed">    
    class DataNode {

        final String name;
        DataNode parent;
        boolean compact;
        boolean flatten;
        boolean flip;
        int depth;
        ArrayList<DataNodeLeaf> leaves;
        ArrayList<DataNode> children;

        DefaultMutableTreeNode jNode;

        DataNode(String name) {
            this.name = name;
        }

        void setCompact(boolean compact) {
            if (compact == this.compact) return;
            this.compact = compact;
            buildJTreeModel(this, true);
            jTree.expandPath(new TreePath(jNode.getPath()));
        }

        void setFlatten(boolean flatten) {
            if (flatten == this.flatten) return;
            this.flatten = flatten;
            buildJTreeModel(this, true);
            jTree.expandPath(new TreePath(jNode.getPath()));
        }

        void setFlipped(boolean flip) {
            if (flip == this.flip) return;
            this.flip = flip;
            jTree.saveExpansionState(0);
            buildJTreeModel(this, true);
            jTree.restoreExpanstionState(0);
        }

        DataNode addChild(String name) {
            if (children == null) {
                children = new ArrayList<>(1);
            }
            for (DataNode child : children) {
                if (child.name.equals(name)) {
                    return child;
                }
            }
            DataNode newChild = new DataNode(name);
            newChild.parent = this;
            children.add(newChild);
            return newChild;
        }

        void addLeaf(DataNodeLeaf leaf) {
            if (leaves == null) leaves = new ArrayList<>(1);
            leaves.add(leaf);
            leaf.parent = this;
        }
        
        void removeAllChildren() {
            leaves = null;
            children = null;
        }

        List<DataNodeLeaf> appendLeaves(List<DataNodeLeaf> out) {
            if (out == null) {
                out = new ArrayList<>();
            }
            if (leaves != null) {
                out.addAll(leaves);
            }
            if (children != null) {
                for (DataNode child : children) {
                    child.appendLeaves(out);
                }
            }
            return out;
        }

        @Override
        public String toString() {
            return name;
        }
        
        void showTable() {
            for (int i=0; i<jNode.getChildCount(); i++) {
                DefaultMutableTreeNode child = (DefaultMutableTreeNode) jNode.getChildAt(i);
                if (child.isLeaf()) {
                    TreePath path = new TreePath(child.getPath());
                    jTree.expandPath(path);
                    return;
                }
            }
        }
        
    }

    class DataNodeLeaf implements MonitorTable.Item {

        final String name;
        AgentChannel channel;
        DataNode parent;
        int id;
        
        DefaultMutableTreeNode jNode; // table node

        DataNodeLeaf(String name, AgentChannel channel) {
            this.name = name;
            this.channel = channel;
        }

        @Override
        public AgentChannel getChannel() {
            return channel;
        }

        @Override
        public String getName() {
            return name;
        }

        void update(List<String> attributes) {
            ((TableHolder)(jNode.getUserObject())).getTable().update(this, attributes);
            getJTreeModel().nodeChanged(jNode);
        }
    }
    
    /**  Sets dataRoot and path2data */
    private void buildDataTree() {
        
        ArrayList<Map.Entry<String,DataNodeLeaf>> channels = new ArrayList<>(path2data.entrySet());
        sortChannels(channels);
        
        dataRoot.removeAllChildren();
        
        for (Map.Entry<String,DataNodeLeaf> e : channels) {
            String[] path = e.getKey().split("/+");
            DataNode parent = dataRoot;
            for (int level=0; level<path.length-1; level++) {
                parent = parent.addChild(path[level]);
            }
            parent.addLeaf(e.getValue());
        }
        
        for (DataNodeLeaf leaf : path2data.values()) {
            int depth = 0;
            DataNode node = leaf.parent;
            while (node != null) {
                if (node.depth < ++depth) {
                    node.depth = depth;
                }
                node = node.parent;
            }
        }
    }
    
    private void sortChannels(ArrayList<Map.Entry<String,DataNodeLeaf>> channels) {
        // hook - FIXME
    }
    
// </editor-fold>    

// -- Customizing JTree and handling its model : -------------------------------
    
// <editor-fold defaultstate="collapsed">
    
    /** Builds JTree model branch under given root */
    private DefaultMutableTreeNode buildJTreeModel(DataNode dataNode, boolean start) {
        
        DefaultMutableTreeNode parent = null;
        DefaultMutableTreeNode node;
        if (start) {
            node = dataNode.jNode;
            if (node == null) {
                node = new DefaultMutableTreeNode(dataNode);
                dataNode.jNode = node;
            } else {
                parent = (DefaultMutableTreeNode) node.getParent(); 
                node.removeAllChildren();
            }
        } else {
            node = new DefaultMutableTreeNode(dataNode);
            dataNode.jNode = node;
        }
        
        if (dataNode.leaves != null) {
            node.add(createLeavesNode(dataNode));
        }
        
        if (dataNode.children != null) {
            if (dataNode.flatten) {
                node.add(createDeepNode(dataNode));
            } else {
                for (DataNode child : dataNode.children) {
                    node.add(buildJTreeModel(child, false));
                }
            }
        }
        
        if (parent != null) {
            getJTreeModel().nodeStructureChanged(parent);
        }
        return node;
    }
    
    /** Creates JTree node that contains TableLeaves. */
    private DefaultMutableTreeNode createLeavesNode(DataNode node) {
        TableLeaves model = new TableLeaves(node);
        DefaultMutableTreeNode jNode = new DefaultMutableTreeNode(new TableHolder(model));
        for (int i=0; i < node.leaves.size(); i++) {
            DataNodeLeaf leaf = node.leaves.get(i);
            leaf.id = i;
            leaf.jNode = jNode;
        }
        return jNode;
    }
    
    /** Creates JTree node that contains TableDeep. */
    private DefaultMutableTreeNode createDeepNode(DataNode node) {
        TableModelDeep model = new TableModelDeep(node);
        DefaultMutableTreeNode jNode = new DefaultMutableTreeNode(new TableHolder(model));
        node.appendLeaves(new ArrayList<>()).forEach(leaf -> leaf.jNode = jNode);
        return jNode;
    }
    
    class Tree extends JTree implements MonitorDisplay {

        private String expansionState;

        Tree() {
            super(new DefaultMutableTreeNode(""));
            setRootVisible(false);
            setShowsRootHandles(true);
            setRowHeight(0);
            setCellRenderer(new TreeRenderer(false));
            setCellEditor(new TreeEditor());
            addMouseMotionListener(new MouseMotionListener() {
                @Override
                public void mouseMoved(MouseEvent e) {
                    if (jTree.getRowForLocation(e.getX(), e.getY()) != -1) {
                        jTree.startEditingAtPath(jTree.getPathForLocation(e.getX(), e.getY()));
                    }
                }

                @Override
                public void mouseDragged(MouseEvent e) {
                }
            });
            setEditable(true);
        }

        @Override
        public void saveData(OutputStream out, String mimeType) {
            List<AgentChannel> channels = path2data.values().stream().map(leaf -> leaf.getChannel()).filter(channel -> channel != null).collect(Collectors.toList());
            LsstMonitorPlugin.saveData(out, mimeType, channels, columns);
        }

        private boolean isDescendant(TreePath path1, TreePath path2) {
            int count1 = path1.getPathCount();
            int count2 = path2.getPathCount();
            if (count1 <= count2) {
                return false;
            }
            while (count1 != count2) {
                path1 = path1.getParentPath();
                count1--;
            }
            return path1.equals(path2);
        }

        void saveExpansionState(int row) {
            TreePath rowPath = getPathForRow(row);
            StringBuilder sb = new StringBuilder();
            int rowCount = getRowCount();
            for (int i = row; i < rowCount; i++) {
                TreePath path = getPathForRow(i);
                if (i == row || isDescendant(path, rowPath)) {
                    if (isExpanded(path)) {
                        sb.append(",").append(i - row);
                    }
                } else {
                    break;
                }
            }
            expansionState = sb.toString();
        }

        void restoreExpanstionState(int row) {
            StringTokenizer stok = new StringTokenizer(expansionState, ",");
            while (stok.hasMoreTokens()) {
                int token = row + Integer.parseInt(stok.nextToken());
                jTree.expandRow(token);
            }
        }

    }

    class TreeRenderer extends DefaultTreeCellRenderer {

        private final NodePanel nodePanel = new NodePanel();
        private final boolean editable;

        TreeRenderer(boolean editable) {
            this.editable = editable;
        }

        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            DefaultMutableTreeNode treeNode = (DefaultMutableTreeNode) value;
            Object payload = treeNode.getUserObject();
//            System.out.println("Renderer: "+ payload);
            if (payload instanceof JComponent) {
                return (JComponent) payload;
            } else if (payload instanceof TableHolder) {
                return ((TableHolder) payload).getComponent(editable);
            } else if (payload instanceof DataNode) {
                nodePanel.setNode((DataNode) payload);
//                System.out.println(nodePanel.getSize());
                return nodePanel;
            } else {
                return super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            }
        }

    }

    class TreeEditor extends AbstractCellEditor implements TreeCellEditor {

        private final TreeRenderer renderer;
        private Object editorValue;
        private Component component;

        public TreeEditor() {
            renderer = new TreeRenderer(true);
        }

        @Override
        public Component getTreeCellEditorComponent(JTree tree, Object value, boolean isSelected, boolean expanded, boolean leaf, int row) {
            editorValue = value;
            return renderer.getTreeCellRendererComponent(tree, value, isSelected, expanded, leaf, row, true);
        }

        /** At this point the component is added to the tree (undocumented!) but tree's internal cleanup might not yet be done. */
        @Override
        public boolean shouldSelectCell(EventObject anEvent) {
            editorValue = null;
            if (anEvent instanceof MouseEvent) {
                redirect((MouseEvent) anEvent);
            }
            return false;
        }

        private void redirect(final MouseEvent anEvent) {
            SwingUtilities.invokeLater(() -> {
                try {
                    MouseEvent ev = SwingUtilities.convertMouseEvent(anEvent.getComponent(), anEvent, component);
                    component.dispatchEvent(ev);
                } catch (Exception x) {
                } // FIXME
            });
        }

        @Override
        public Object getCellEditorValue() {
            return editorValue;
        }
    }

    final class NodePanel extends Box {

        private final JLabel label = new JLabel();
        private JCheckBox compact, flatten, flip;
        private DataNode node;

        NodePanel() {
            super(BoxLayout.X_AXIS);
            add(label);
            compact = new JCheckBox();
            compact.setToolTipText("Compact");
            compact.setOpaque(false);
            compact.addActionListener(e -> {
                if (node != null) {
                    node.setCompact(compact.isSelected());
                }
            });
            add(compact);
            flatten = new JCheckBox();
            flatten.setToolTipText("Flatten");
            flatten.setOpaque(false);
            flatten.addActionListener(e -> {
                if (node != null) {
                    boolean selected = flatten.isSelected();
                    flip.setEnabled(selected);
                    node.setFlatten(selected);
                }
            });
            add(flatten);
            flip = new JCheckBox();
            flip.setToolTipText("Flip");
            flip.setOpaque(false);
            flip.addActionListener(e -> {
                if (node != null) {
                    node.setFlipped(flip.isSelected());
                }
            });
            add(flip);
            add(Box.createRigidArea(new Dimension(500, 0))); // FIXME: need to figure out why the box is not being resized properly in "editor" mode
            add(Box.createHorizontalGlue());
        }

        void setNode(DataNode node) {
            this.node = null;
            if (node != null) {
                label.setText(" " + node.name + " ");
                compact.setSelected(node.compact);
                boolean hasLeaves = false;
                for (int i=0; i<node.jNode.getChildCount(); i++) {
                    if (node.jNode.getChildAt(i).isLeaf()) {
                        hasLeaves = true;
                        break;
                    }
                }
                compact.setEnabled(hasLeaves);
                flatten.setSelected(node.flatten);
                flip.setSelected(node.flip);
                flatten.setEnabled(node.depth == 2 || node.depth == 3);
                flip.setEnabled(node.flatten);
                this.node = node;
//                setSize(0, 0);

//                validate();
//                if (node.name.equals("monitor-test")) {
//                    System.out.println("monitor-test: "+ getSize());
//                }
            }
        }

    }

    class TableHolder {

        private final MonitorTable model;
        private final JPanel renderPanel, editPanel;

        TableHolder(MonitorTable model) {
            this.model = model;
            JTable table = model.makeTable();
            renderPanel = new JPanel(new BorderLayout());
            renderPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0), BorderFactory.createLineBorder(Color.BLACK)));
            if (model instanceof TableLeaves) {
                renderPanel.add(table.getTableHeader(), BorderLayout.NORTH);
            }
            renderPanel.add(table, BorderLayout.CENTER);
            table = model.makeTable();
            editPanel = new JPanel(new BorderLayout());
            editPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0), BorderFactory.createLineBorder(Color.BLACK)));
            if (model instanceof TableLeaves) {
                editPanel.add(table.getTableHeader(), BorderLayout.NORTH);
            }
            editPanel.add(table, BorderLayout.CENTER);
        }

        JPanel getComponent(boolean editable) {
            return editable ? editPanel : renderPanel;
        }
        
        MonitorTable getTable() {
            return model;
        }

    }
    
// </editor-fold>    

// -- Monitoring table classes : -----------------------------------------------
    
// <editor-fold defaultstate="collapsed">        

    class TableLeaves extends MonitorTable {
        
        private DataNode node;
        private List<MonitorField> cols;

        TableLeaves(DataNode node) {
            this.node = node;
            
            nRows = node.leaves.size();
            cols = node.compact ? compactColumns : columns;
            nColumns = cols.size();
            
            HashSet<MonitorField> presentFields = new HashSet<>();
            presentFields.add(MonitorField.NAME);
            for (DataNodeLeaf leaf : node.leaves) {
                AgentChannel channel = leaf.getChannel();
                if (channel != null) {
                    for (MonitorField f : cols) {
                        if (channel.get(f.name()) != null) {
                            presentFields.add(f);
                        }
                    }
                }
                if (presentFields.size() == nColumns) break;
            }
            if (presentFields.size() != nColumns) {
                cols = new ArrayList<>(cols);
                cols.retainAll(presentFields);
                ((ArrayList<MonitorField>)cols).trimToSize();
                nColumns = cols.size();
            }
            
            cells = new Cell[nRows][nColumns];
            for (int row = 0; row < nRows; row++) {
                DataNodeLeaf leaf = node.leaves.get(row);
                for (int col = 0; col < nColumns; col++) {
                    MonitorField field = cols.get(col);
                    Cell cell = new Cell(leaf, field);
                    format(cell);
                    cells[row][col] = cell;
                }
            }
        }

        @Override
        public String getColumnName(int column) {
            return cols.get(column).getTitle();
        }

        @Override
        public List<int[]> getCells(Item item, MonitorField field) {
            int column = cols.indexOf(field);
            if (column == -1) return Collections.emptyList();
            int row = node.leaves.indexOf(item);
            if (row == -1) return Collections.emptyList();
            return Collections.singletonList(new int[] {row, column});
        }

    }

    class TableModelDeep extends MonitorTable {
        
        private DataNode node;
        private final HashMap<String, int[]> path2cell;  // 3-node path to [row,column]
        private final int nValueCols;
        private final List<MonitorField> extraColumns;

        TableModelDeep(DataNode node) {
            
            this.node = node;
            
            // Compile list of leaves to include

            List<DataNodeLeaf> leaves = new ArrayList<>();
            List<DataNodeLeaf> leaves3 = new ArrayList<>();
            for (DataNodeLeaf leaf : node.appendLeaves(new ArrayList<>())) {
                String[] path = getPath(node, leaf).split("/+");
                if (path.length == 2) {
                    leaves.add(leaf);
                } else if (path.length == 3) {
                    leaves3.add(leaf);
                }
            }
            leaves.addAll(leaves3);
            
            // Compile list of non-value columns to display

            if (node.compact) {
                extraColumns = Collections.emptyList();
            } else {
                int n = columns.size();
                HashSet<MonitorField> presentFields = new HashSet<>();
                for (DataNodeLeaf leaf : leaves) {
                    AgentChannel channel = leaf.getChannel();
                    if (channel != null) {
                        for (MonitorField f : columns) {
                            if (channel.get(f.name()) != null) {
                                presentFields.add(f);
                            }
                        }
                    }
                    if (presentFields.size() == n) break;
                }
                presentFields.remove(MonitorField.VALUE);
                presentFields.remove(MonitorField.NAME);
                extraColumns = new ArrayList<>(presentFields.size());
                for (MonitorField f : columns) {
                    if (presentFields.contains(f)) {
                        extraColumns.add(f);
                    }
                }
            }
            
            // Make list of rows and pap paths to cells

            ArrayList<String> rowNames = new ArrayList<>();
            ArrayList<String> columnNames = new ArrayList<>();
            path2cell = new LinkedHashMap<>();
            ArrayList<ArrayList<DataNodeLeaf>> rows = new ArrayList<>();
            rows.add(null);
            String currentSection = "";
            int currentRow = 1;
            int currentColumn = 1;
            for (DataNodeLeaf leaf : leaves) {
                String path = getPath(node, leaf);
                String[] sPath = path.split("/");
                String section, row, column;
                if (sPath.length == 2) {
                    section = "";
                    row = node.flip ? sPath[1] : sPath[0];
                    column = node.flip ? sPath[0] : sPath[1];
                } else {
                    section = sPath[0];
                    row = node.flip ? sPath[2] : sPath[1];
                    row = row + "/" + section;
                    column = node.flip ? sPath[1] : sPath[2];
                }
                if (!section.equals(currentSection)) {
                    rowNames.add(section);
                    rows.add(null);
                    currentRow++;
                    currentSection = section;
                }
                int iRow = rowNames.indexOf(row) + 1;
                if (iRow == 0) {
                    iRow = currentRow++;
                    rowNames.add(row);
                    rows.add(new ArrayList<>());
                }
                int iColumn = columnNames.indexOf(column) + 1;
                if (iColumn == 0) {
                    iColumn = currentColumn++;
                    columnNames.add(column);
                }
                path2cell.put(path, new int[]{iRow, iColumn});
                rows.get(iRow).add(leaf);
            }
            nValueCols = columnNames.size();

            nRows = rows.size();
            nColumns = nValueCols + 1 + extraColumns.size();
            cells = new Cell[nRows][];
            
            // Fill all non-empty cells except value cells
            
            for (int ir = 0; ir < nRows; ir++) {
                cells[ir] = new Cell[nColumns];
                if (ir == 0) {  // title row
                    cells[0][0] = new Cell();
                    for (int ic = 0; ic < nValueCols; ic++) {
                        Cell cell = new Cell();
                        cell.setData(new MonitorTable.Data(columnNames.get(ic), SwingConstants.CENTER));
                        cells[0][ic + 1] = cell;
                    }
                    for (int ic = 0; ic < extraColumns.size(); ic++) {
                        Cell cell = new Cell();
                        cell.setData(new MonitorTable.Data(extraColumns.get(ic).getTitle(), SwingConstants.CENTER));
                        cells[0][ic + nValueCols + 1] = cell;
                    }
                } else if (rows.get(ir) == null) { // section title row
                    Cell cell = new Cell();
                    cell.setData(new MonitorTable.Data("<html><b> " + rowNames.get(ir - 1) + "</b>", SwingConstants.LEFT));
                    cells[ir][0] = cell;
                } else { // normal row (except value cells)
                    Cell cell = new Cell();
                    cell.setData(new MonitorTable.Data(rowNames.get(ir - 1).split("/")[0], SwingConstants.LEFT));
                    cells[ir][0] = cell;
                    for (int column = 0; column < extraColumns.size(); column++) {
                        cells[ir][column + nValueCols + 1] = new Cell(rows.get(ir), extraColumns.get(column));
                    }
                }
            }
            
            // Fill value cells

            for (DataNodeLeaf leaf : leaves) {
                int[] cell = path2cell.get(getPath(node, leaf));
                cells[cell[0]][cell[1]] = new Cell(leaf, MonitorField.VALUE);
            }
            
            // Fill empty cells and format those not already formatted.
            
            for (Cell[] cc : cells) {
                for (int i=0; i<cc.length; i++) {
                    Cell c = cc[i];
                    if (c == null) {
                        cc[i] = Cell.EMPTY;
                    } else {
                        if (c.getData() == null) {
                            format(c);
                        }
                    }
                }
            }

        }

        @Override
        public List<int[]> getCells(Item item, MonitorField field) {
            
            DataNodeLeaf leaf = (DataNodeLeaf) item;
            
            int[] cell = path2cell.get(getPath(node, leaf));
            if (cell == null) return Collections.emptyList();
            
            if (extraColumns.isEmpty()) {
                return (field == null || MonitorField.VALUE.equals(field)) ? Collections.singletonList(cell) : Collections.emptyList();
            } else {
                if (MonitorField.VALUE.equals(field)) {
                    return Collections.singletonList(cell);
                } else if (field == null) {
                    ArrayList<int[]> out = new ArrayList<>(extraColumns.size() + 1);
                    out.add(cell);
                    for (int i = 0; i < extraColumns.size(); i++) {
                        out.add(new int[]{cell[0], i + nValueCols + 1});
                    }
                    return out;
                } else {
                    int col = extraColumns.indexOf(field);
                    if (col == -1) {
                        return Collections.emptyList();
                    } else {
                        cell[1] = col + nValueCols + 1;
                        return Collections.singletonList(cell);
                    }
                }
            }
            
        }

    }
    
// </editor-fold>    

// -- Local methods : ----------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    /**
     * Returns JTree table.
     */
    private DefaultTreeModel getJTreeModel() {
        return (DefaultTreeModel) jTree.getModel();
    }

    /**
     * Returns path from {@code leaf} to {@code node}}
     */
    static private String getPath(DataNode node, DataNodeLeaf leaf) {
        StringBuilder sb = new StringBuilder(leaf.name);
        DataNode parent = leaf.parent;
        while (parent != null) {
            if (parent == node) {
                return sb.toString();
            } else {
                sb.insert(0, "/").insert(0, parent.name);
                parent = parent.parent;
            }
        }
        return null;
    }
    
    private String saveTreeState() {
        StringBuilder sb = new StringBuilder();
        saveTreeState(sb, dataRoot);
        return sb.length() == 0 ? "" : sb.substring(1);
    }
    
    private Map<String,String> restoreDataTreeState(String state) {
        if (state.isEmpty()) return Collections.emptyMap();
        HashMap<String,String> data = new HashMap<>();
        String[] nodes = state.split("\\+");
        for (String node : nodes) {
            String[] ss = node.split("=");
            data.put(ss[0], ss[1]);
        }
        restoreDataTreeState(data, dataRoot, "");
        return data;
    }
    
    private void restoreJTreeState(Map<String,String> data, DataNode node, String path) {
        path = path.isEmpty() ? node.name : path +"/"+ node.name;
        String stringState = data.get(path);
        if (stringState != null) {
            int state = Integer.parseInt(stringState);
            if ((state & 8) > 0) {
                TreePath tp = new TreePath(node.jNode.getPath());
                jTree.expandPath(tp);
                if (node.children != null) {
                    for (DataNode child : node.children) {
                        restoreJTreeState(data, child, path);
                    }
                }
            }
        }
    }
    
    private void saveTreeState(StringBuilder sb, DataNode node) {
        int state = 0;
        if (node.compact) state |= 1;
        if (node.flatten) state |= 2;
        if (node.flip) state |= 4;
        if (node.jNode != null) {
            if (jTree.isExpanded(new TreePath(node.jNode.getPath()))) {
                state |= 8;
            }
        }
        if (state != 0) {
            StringBuilder s = new StringBuilder(node.name);
            DataNode parent = node.parent;
            while (parent != null) {
                s.insert(0, "/").insert(0, parent.name);
                parent = parent.parent;
            }
            s.append("=").append(state);
            sb.append("+").append(s);
        }
        if (node.children != null) {
            for (DataNode child : node.children) {
                saveTreeState(sb, child);
            }
        }
    }
    
    private void restoreDataTreeState(Map<String,String> data, DataNode node, String path) {
        path = path.isEmpty() ? node.name : path +"/"+ node.name;
        String stringState = data.get(path);
        if (stringState != null) {
            int state = Integer.parseInt(stringState);
            if ((state & 1) > 0) node.compact = true;
            if ((state & 2) > 0) node.flatten = true;
            if ((state & 4) > 0) node.flip = true;
        }
        if (node.children != null) {
            for (DataNode child : node.children) {
                restoreDataTreeState(data, child, path);
            }
        }
        
    }
    
// </editor-fold>
    
}
